package com.transsion.ad.view.native_ad

import android.content.Context
import android.view.LayoutInflater
import android.view.View
import androidx.appcompat.widget.AppCompatTextView
import com.blankj.utilcode.util.Utils
import com.hisavana.common.bean.TAdNativeInfo
import com.hisavana.common.constant.NativeContextMode
import com.hisavana.mediation.ad.ViewBinder
import com.transsion.ad.util.AdUtils
import com.transsion.ad.view.AdTagView

/**
 * @author shmizhangxinbing
 * @date : 2025/6/5 10:52
 * @description:
 */
abstract class AbsBiddingAdLayoutProvider(val context: Context) {

    fun getAdLayoutProvider(): BiddingAdLayoutProviderBean {
        return BiddingAdLayoutProviderBean(
            layoutId = getLayoutId(),                           // layout id
            layout = getAdLayout(context),                  // LayoutInflater View
            mediaId = getMediaId(),                             // TMediaView
            titleId = getTitleId(),                             // TextView
            iconId = getIconId(),                               // TIconView
            callToActionId = getCallToActionId(),               // TextView
            descriptionId = getDescriptionId(),                 // TextView

            adChoicesView = getAdChoicesView(),                 // TAdChoicesView
            adStoreMarkView = getAdStoreMarkView(),             // TStoreMarkView
            adCloseView = getAdCloseView(),                     // TAdCloseView

            ratingId = getRatingId(),                           // 评分 TODO
            priceId = getPriceId(),                             // 价格 TODO
            sponsoredId = getSponsoredId(),                     // 赞助 TODO
            likesId = getLikesId(),                             // 喜欢 TODO
            downloadsId = getDownloadsId(),                     // 下载数量 TODO
            adDisclaimerView = getAdDisclaimerView(),           // 免责声明 TODO
            adBadgeView = getAdBadgeView(),                     // 徽章 TODO
            domainId = getDomainId(),                           // 领域 TODO
            feedbackId = getFeedbackId(),                       // 反馈 TODO

            apkSizeId = getApkSizeId(),                         // APK 大小 TextView
            burlBgId = getBurlBgId(),                           // 高斯模糊背景 ImageView
            adTagId = getAdTagId(),                             // AdTagView
        )
    }


    // =============================================================================================


    private fun getAdLayout(viewContext: Context?): View {
        // 如果没有Icon那就不展示了
        return LayoutInflater.from(viewContext ?: Utils.getApp()).inflate(getLayoutId(), null)
    }

    abstract fun getLayoutId(): Int
    abstract fun getTitleId(): Int?
    abstract fun getDescriptionId(): Int?
    abstract fun getMediaId(): Int?
    abstract fun getIconId(): Int?
    abstract fun getCallToActionId(): Int?
    abstract fun getAdChoicesView(): Int?
    abstract fun getAdStoreMarkView(): Int?
    abstract fun getAdTagId(): Int?

    open fun getRatingId(): Int? = null
    open fun getPriceId(): Int? = null
    open fun getSponsoredId(): Int? = null
    open fun getLikesId(): Int? = null
    open fun getDownloadsId(): Int? = null
    open fun getAdDisclaimerView(): Int? = null
    open fun getAdBadgeView(): Int? = null
    open fun getAdCloseView(): Int? = null
    open fun getDomainId(): Int? = null
    open fun getFeedbackId(): Int? = null
    open fun getApkSizeId(): Int? = null
    open fun getBurlBgId(): Int? = null

    open fun isShowVolumeIcon(): Boolean = true // 是否展示音量Icon

    open fun getHiViewBinder(
        adLayoutProvider: BiddingAdLayoutProviderBean, mNativeInfo: TAdNativeInfo?
    ): ViewBinder {
        val builder = ViewBinder.Builder(adLayoutProvider.layout)
        builder.contextMode(NativeContextMode.LIST)

        // 标题
        adLayoutProvider.titleId?.let {
            builder.titleId(it)
        }

        // 简介
        adLayoutProvider.descriptionId?.let {
            builder.descriptionId(it)
        }

        // 媒体容器
        adLayoutProvider.mediaId?.let {
            builder.mediaId(it)
        }

        // Icon 主图
        adLayoutProvider.iconId?.let {
            builder.iconId(it)
        }

        // 按钮
        adLayoutProvider.callToActionId?.let {
            builder.callToActionId(it)
            // 修改大小写
            adLayoutProvider.layout.findViewById<AppCompatTextView>(it)?.apply {
                post {
                    kotlin.runCatching {
                        val txt = text.toString().lowercase()
                        if (AdUtils.isEnglishString(txt)) {
                            text = txt.capitalize()
                        }
                    }
                }
            }
        }

        // 广告来源
        adLayoutProvider.adChoicesView?.let {
            builder.adChoicesView(it)
        }

        // PS应用市场标识
        adLayoutProvider.adStoreMarkView?.let {
            builder.storeMarkView(it)
        }

        // Hi程序化统一不展示AD标识
        adLayoutProvider.adTagId?.let {
            adLayoutProvider.layout.findViewById<AdTagView>(it).visibility = View.GONE
        }

        //
        val viewBinder = builder.build()
        rebuildButtonIfNeed(viewBinder)

        // 构建容器
        return viewBinder
    }

    /**
     * 重建广告底部的按钮文案大小写，这里是因为 HiSavana 广告的按钮默认是全大写的，所以需要在这里进行首字母大写
     */
    private fun rebuildButtonIfNeed(viewBinder: ViewBinder) {
        AdUtils.setTitleCapitalizeForView(viewBinder.layout, viewBinder.callToActionId)
    }
}