package com.transsion.ad.bidding.base

import android.app.Activity
import android.content.Context
import android.os.Handler
import android.os.Looper
import android.text.TextUtils
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.hisavana.common.bean.AdditionalInfo
import com.hisavana.common.bean.TAdErrorCode
import com.hisavana.common.bean.TAdNativeInfo
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.gemini.AbsBiddingBuyOutGemini
import com.transsion.ad.monopoly.intercept.NonAdShowedTimesManager
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdShowLevel
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.monopoly.model.MbAdType
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.monopoly.plan.AdPlanUtil
import com.transsion.ad.report.AdReportProvider
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.scene.SceneOnOff
import com.transsion.ad.config.TestConfig
import com.transsion.ad.log.ILog
import com.transsion.ad.ps.PsDbManager
import com.transsion.ad.ps.PsOfferProvider
import com.transsion.ad.ps.attribution.AttributionProduceManager
import com.transsion.ad.report.BiddingStateEnum
import com.transsion.ad.strategy.AdClickManager
import com.transsion.ad.strategy.AdOverridePendingTransitionManager
import com.transsion.ad.util.RandomUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/5/27 19:21
 * @description: 竞价逻辑封装
 *
 * loadAd() --> 前置条件判断 --> 触发埋点 --> 读取所有广告计划 --> 找出可用计划 --> 竞价 --> 回调
 */
abstract class AbsAdBidding : BiddingTAdditionalListener(), ILog {

    /**
     * 传递进来的数据
     */
    private var mSceneId: String? = null // 场景调用的时候传进来的场景ID
    private var mCtxMap: Map<String, Any>? = null// 上下文信息
    private var mListener: BiddingTAdditionalListener? = null // 广告监听回调 加载的场景提供
    private var mContext: Context? = null // 上下文

    /**
     * 埋点使用 链路关系
     */
    private var triggerId: String = ""
    private fun getTriggerId(): String = triggerId

    /**
     * 竞价 中间产物数据
     */
    private val mBiddingHandler: Handler = Handler(Looper.getMainLooper()) // 竞价计时
    private var biddingPlanList: MutableList<BiddingIntermediateMaterialBean>? = null // 符合条件的计划
    private var maxEcpmObject: BiddingIntermediateMaterialBean? = null // 竞价胜出对象

    /*** 是否正在加载中*/
    private val isLoading: AtomicBoolean = AtomicBoolean(false)

    /*** 线程切换*/
    private val mUIHandler: Handler = Handler(Looper.getMainLooper())

    /*** 广告Activity页面打开样式*/
    private var mAdOverridePendingTransitionEnum: AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum? =
        null

    /**
     * 是否已经上报过一次曝光了
     *
     * 包断广告是客户端去重复，HI程序化是服务端去重复
     */
    private var isReportAdDisplay: AtomicBoolean = AtomicBoolean(false)


    // ===================================== 子类需要重写的API ========================================


    override fun getSceneId(): String = mSceneId ?: "default_id"

    /**
     * 获取广告类型，埋点使用
     */
    abstract fun getAdType(): Int

    /**
     * Activity 广告必须实现
     */
    abstract fun getGemini(): AbsBiddingBuyOutGemini?

    /**
     * 添加HiSavana Provider
     */
    abstract fun addHiSavanaProvider(
        biddingPlan: MutableList<BiddingIntermediateMaterialBean>, adPlans: AdPlans
    )


    // ======================================== get set ============================================


    fun getContext(): Context? = mContext
    fun getListener(): BiddingTAdditionalListener? = mListener
    fun getCtxMap(): Map<String, Any> = mCtxMap ?: emptyMap()

    /**
     * Activity跳转的时候动画样式
     */
    open fun getAdOverridePendingTransition(): AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum? {
        return mAdOverridePendingTransitionEnum
    }

    /**
     * 场景ID
     */
    fun setSceneId(sceneId: String?): AbsAdBidding {
        mSceneId = sceneId
        return this@AbsAdBidding
    }

    /**
     * 场景下文
     */
    fun setCtxMap(ctxMap: Map<String, Any>?): AbsAdBidding {
        mCtxMap = ctxMap
        return this@AbsAdBidding
    }

    /**
     * 广告加载回调
     */
    fun setListener(listener: BiddingTAdditionalListener?): AbsAdBidding {
        mListener = listener
        return this@AbsAdBidding
    }

    /**
     * 上下文
     */
    fun setContext(context: Context): AbsAdBidding {
        mContext = context
        return this@AbsAdBidding
    }

    /**
     * Activity 广告转场样式
     */
    fun setAdOverridePendingTransition(adOverridePendingTransitionEnum: AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum?): AbsAdBidding {
        mAdOverridePendingTransitionEnum = adOverridePendingTransitionEnum
        return this@AbsAdBidding
    }


    // =============================================================================================


    /*** 获取当前是否已经曝光*/
    fun isReportAdDisplay(): Boolean = isReportAdDisplay.get()

    /*** 获取竞价成功的对象*/
    fun getMaxEcpmObject(): BiddingIntermediateMaterialBean? = maxEcpmObject

    /*** 参与竞价的广告计划列表*/
    fun getBiddingPlanList(): MutableList<BiddingIntermediateMaterialBean>? = biddingPlanList

    /**
     * 展示广告
     *
     * 这个方法仅用于 激励视频、插屏广告展示的调用
     */
    fun startAdActivity(activity: Activity?) {
        if (InterceptUtil.isInterceptAdShowed()) {
            onBiddingWrapperAdShowError(MbAdSource.MB_AD_SOURCE_WRAPPER_AD)
            return
        }

        (activity as? AppCompatActivity)?.lifecycleScope?.launch {
            val hiProvider = getMaxEcpmObject()?.hiSavanaInterceptProvider
            if (hiProvider == null) {
                if (null != getMaxEcpmObject()?.plans) {
                    activity.let {
                        getGemini()?.setListener(this@AbsAdBidding)?.registerReceiver()
                            ?.setSceneId(getSceneId())
                            ?.startActivity(it, getSceneId(), getMaxEcpmObject()?.plans) ?: run {
                            onBiddingWrapperAdShowError(MbAdSource.MB_AD_SOURCE_WRAPPER_AD)
                        }
                    }
                } else {
                    // 理论上来说这里是不会调用的，不排除API调用错误在展示之前调用了 destroy()
                    onBiddingWrapperAdShowError(MbAdSource.MB_AD_SOURCE_WRAPPER_AD)
                }
            } else {
                hiProvider.setListener(this@AbsAdBidding)
                hiProvider.showAd(activity, getSceneId())
            }
        } ?: run {
            onBiddingWrapperAdShowError(MbAdSource.MB_AD_SOURCE_WRAPPER_AD)
        }
    }

    open fun checkContextNonNull(): Boolean = false // 是否需要检查context是否为空

    /*** 资源回收*/
    open fun destroy() {

        setListener(null)
        setCtxMap(null)
        getMaxEcpmObject()?.hiSavanaInterceptProvider?.destroy(this@AbsAdBidding) // 激励视频、插屏广告移除监听
        mBiddingHandler.removeCallbacksAndMessages(null)
        maxEcpmObject = null

        getGemini()?.destroy() // 子类对象释放

        if (getAdType() != MbAdType.MB_AD_TYPE_NATIVE) {
            onLog(Log.DEBUG, "destroy() --> 资源回收")
        }
        setSceneId(null)
    }

    /*** 是否使用Hi ecpm*/
    protected fun isUseHiEcpm(adPlans: AdPlans?): Boolean {
        return adPlans?.bidEcpmCent == null || (adPlans.bidEcpmCent ?: 0.0) == 0.0
    }


    // =================================== 广告回调 ==================================================


    /**
     * Hi程序化广告回调 埋点
     */
    override fun onShow(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onShow(p0, p1)

        // TNativeAd 会统一回调
        // 需要特殊处理Native广告
        if (getAdType() == MbAdType.MB_AD_TYPE_NATIVE && TextUtils.equals(
                getMaxEcpmObject()?.nativeInfo?.adId, p0?.adId
            ).not()
        ) {
            return
        }

        onBiddingWrapperAdDisplay(MbAdSource.MB_AD_SOURCE_HISAVANA)

        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = p1.source,
            adId = p1.codeSeatId,
            adType = getAdType(),
            isAdShowFinal = false,
            psId = null,
            bidEcpmCent = getMaxEcpmObject()?.ecpm,
            ecpmCent = null
        )
    }

    override fun onShowError(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onShowError(p0, p1)
        onLog(
            level = Log.ERROR,
            msg = "onShowError() --> errorMessage = ${p0?.errorMessage} --> placementId = ${p1.placementId}"
        )
        onBiddingWrapperAdShowError(MbAdSource.MB_AD_SOURCE_HISAVANA)
    }

    override fun onClick(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onClick(p0, p1)
        if (getAdType() == MbAdType.MB_AD_TYPE_NATIVE && TextUtils.equals(
                getMaxEcpmObject()?.nativeInfo?.adId, p0?.adId
            ).not()
        ) {
            return
        }
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = p1.source,
            adId = p1.codeSeatId,
            adType = getAdType(),
            isAdShowFinal = false,
            psId = null,
            bidEcpmCent = getMaxEcpmObject()?.ecpm,
            ecpmCent = null
        )
        onBiddingWrapperAdClick(MbAdSource.MB_AD_SOURCE_HISAVANA)
    }

    override fun onRewarded() {
        super.onRewarded()
        onBiddingWrapperAdRewarded(MbAdSource.MB_AD_SOURCE_HISAVANA)
    }

    override fun onClosed(p0: Int) {
        super.onClosed(p0)
        onBiddingWrapperAdClose(MbAdSource.MB_AD_SOURCE_HISAVANA)
    }

    /**
     * BuyOut 广告回调 埋点
     */
    override fun onBiddingBuyOutDisplay(plans: AdPlans?) {
        super.onBiddingBuyOutDisplay(plans)
        // 标记当前广告已经曝光了
        isReportAdDisplay.set(true)

        // PS直投归因处理
        AttributionProduceManager.onBiddingBuyOutDisplay(plans)
        // 广告埋点上报
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = plans?.id,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = AdPlanUtil.getAdMaterial(plans)?.id,
            adType = getAdType(),
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(plans),
            psId = AdPlanUtil.getPsId(plans),
            psPackageName = AdPlanUtil.getPsPackageName(plans),
            bidEcpmCent = plans?.bidEcpmCent,
            ecpmCent = plans?.ecpmCent
        )
        // PS Offer 处理
        PsDbManager.onBiddingBuyOutDisplay(plans)
        // 回调给媒体
        onBiddingWrapperAdDisplay(MbAdSource.MB_AD_SOURCE_BUY_OUT)
    }

    override fun onBiddingBuyOutShowError(p0: TAdErrorCode?, plans: AdPlans?) {
        super.onBiddingBuyOutShowError(p0, plans)
        onBiddingWrapperAdShowError(MbAdSource.MB_AD_SOURCE_BUY_OUT)
    }

    override fun onBiddingBuyOutClick(plans: AdPlans?) {
        super.onBiddingBuyOutClick(plans)
        // PS直投归因处理 特殊处理
        // AttributionProduceManager.onBiddingBuyOutClick(plans)
        // 广告埋点上报
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = plans?.id,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = AdPlanUtil.getAdMaterial(plans)?.id,
            adType = getAdType(),
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(plans),
            psId = AdPlanUtil.getPsId(plans),
            psPackageName = AdPlanUtil.getPsPackageName(plans),
            bidEcpmCent = plans?.bidEcpmCent,
            ecpmCent = plans?.ecpmCent
        )
        // 点击事件统一处理
        AdClickManager.onBiddingAdClick(
            adPlan = plans,
            overridePendingTransition = getAdOverridePendingTransition(),
            logTag = getLogTag(),
            sceneId = getSceneId()
        )
        // PS Offer 处理
        PsDbManager.onBiddingBuyOutClick(plans)
        // 回调给媒体
        onBiddingWrapperAdClick(MbAdSource.MB_AD_SOURCE_BUY_OUT)
    }

    override fun onBiddingBuyOutRewarded(plans: AdPlans?) {
        super.onBiddingBuyOutRewarded(plans)
        onBiddingWrapperAdRewarded(MbAdSource.MB_AD_SOURCE_BUY_OUT)
    }

    override fun onBiddingBuyOutDisplayTimestamp(plans: AdPlans?, displayTimestamp: Long) {
        super.onBiddingBuyOutDisplayTimestamp(plans, displayTimestamp)
        onLog(msg = "onBiddingBuyOutDisplayTimestamp() --> name = ${getMaxEcpmObject()?.plans?.name} --> displayTimestamp = $displayTimestamp")

        AdReportProvider.adShowTime(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = plans?.id,
            displayTime = displayTimestamp,
            adId = AdPlanUtil.getAdMaterial(plans)?.id,
            adType = getAdType(),
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(plans)
        )
        mUIHandler.post {
            getListener()?.onBiddingBuyOutDisplayTimestamp(plans, displayTimestamp)
        }
    }

    override fun onBiddingBuyOutClose(plans: AdPlans?) {
        super.onBiddingBuyOutClose(plans)
        onBiddingWrapperAdClose(MbAdSource.MB_AD_SOURCE_BUY_OUT)
    }

    /**
     * 统一处理广告 加载展示 回调
     */
    override fun onBiddingError(p0: TAdErrorCode?) {
        super.onBiddingError(p0)
        onLog(
            level = Log.ERROR, msg = "onBiddingError() --> errorMessage = ${p0?.errorMessage}"
        )
        // 重置状态
        isLoading.set(false)
        mUIHandler.post {
            getListener()?.onBiddingError(p0) // 传递出去
        }
    }

    override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingLoad(maxEcpmObject)

        // 获取所有的ecpm
        val ecpmList = mutableListOf<String?>()
        getBiddingPlanList()?.forEach {
            ecpmList.add(it.ecpm?.toString())
        }
        onLog(
            level = Log.DEBUG,
            msg = "onBiddingLoad() --> 竞价完成 --> ecpmList = $ecpmList --> ecpm = ${getMaxEcpmObject()?.ecpm} --> plans?.id = ${getMaxEcpmObject()?.plans?.id} --> plans?.name = ${getMaxEcpmObject()?.plans?.name}"
        )

        // 竞价成功上报
        AdReportProvider.biddingReport(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adType = getAdType(),
            msg = "竞价成功 --> ecpmList = $ecpmList , ecpm = ${getMaxEcpmObject()?.ecpm} , plans?.id = ${getMaxEcpmObject()?.plans?.id} , plans?.name = ${getMaxEcpmObject()?.plans?.name}",
            result = BiddingStateEnum.BIDDING_REPORT_BIDDING_SUCCESS,
            ecpmList = ecpmList.toString(),
            ecpm = "${getMaxEcpmObject()?.ecpm}",
            planId = "${getMaxEcpmObject()?.plans?.id}",
            planName = "${getMaxEcpmObject()?.plans?.name}"
        )

        isLoading.set(false)
        mUIHandler.post {
            getListener()?.onBiddingLoad(maxEcpmObject) // 传递出去
        }
    }

    /**
     * Wrapper 层通知
     */
    override fun onBiddingWrapperAdDisplay(adSource: Int) {
        //super.onBiddingWrapperAdDisplay(adSource)
        onLog(msg = "onBiddingWrapperAdDisplay() --> name = ${getMaxEcpmObject()?.plans?.name} --> adSource = $adSource")

        if (InterceptUtil.isInterceptAd(getAdType())) {
            InterceptUtil.onAdShow()
        }
        // 回调给媒体
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdDisplay(adSource)
        }

        // 现在所有的广告都走计划，所以广告展示需要将对应的计划展示数自增
        // 统一处理 广告计划展示数自增
        NonAdShowedTimesManager.saveShowedTimes(getMaxEcpmObject()?.plans)
    }

    override fun onBiddingWrapperAdShowError(adSource: Int) {
        //super.onBiddingWrapperAdShowError(adSource)
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdShowError(adSource)
        }
    }

    override fun onBiddingWrapperAdClick(adSource: Int) {
        //super.onBiddingWrapperAdClick(adSource)
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdClick(adSource)
        }
    }

    override fun onBiddingWrapperAdRewarded(adSource: Int) {
        //super.onBiddingWrapperAdRewarded(adSource)
        onLog(msg = "onBiddingWrapperAdRewarded() --> name = ${getMaxEcpmObject()?.plans?.name} --> adSource = $adSource")
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdRewarded(adSource)
        }
    }

    override fun onBiddingWrapperAdClose(adSource: Int) {
        //super.onBiddingWrapperAdClose(adSource)
        onLog(msg = "onBiddingWrapperAdClose() --> name = ${getMaxEcpmObject()?.plans?.name} --> adSource = $adSource")
        if (InterceptUtil.isInterceptAd(getAdType())) {
            InterceptUtil.onAdClose()
        }
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdClose(adSource)
        }
    }


    // ========================================== 加载广告 ===========================================


    /**
     * 通用API 加载广告
     */
    suspend fun loadAd() {

        // 测试广告加载失败
        if (TestConfig.isGlobalAdOff()) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "客户端 全局 关闭广告"
                )
            )
        }

        // 如果当前是开屏广告、插屏广告、激励视频广告才需要判断拦截
        if (InterceptUtil.isInterceptAdShowed() && InterceptUtil.isInterceptAd(getAdType())) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "intercept ad is showing"
                )
            )
        }

        if (isLoading.get()) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING,
                    "isLoading -- Try again when it's a little thicker"
                )
            )
        }
        // 加载广告
        isLoading.set(true)

        // 广告触发 统一入口处理
        // 从这个版本开始 场景触发将无法区分广告源
        triggerId = RandomUtils.getTriggerId()
        AdReportProvider.trigger(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adType = getAdType(),
            adSource = MbAdSource.MB_AD_SOURCE_BIDDING, // 特殊标记
            planId = ""
        )

        // 场景关闭了直接返回
        val errorMsg = SceneOnOff.isSceneOffV2(getSceneId())
        if (TextUtils.isEmpty(errorMsg).not()) {
            AdReportProvider.reject(
                triggerId = triggerId,
                sceneId = getSceneId(),
                adType = getAdType(),
                adSource = MbAdSource.MB_AD_SOURCE_BIDDING,
                rejectMsg = errorMsg
            )
            return onBiddingError(TAdErrorCode(MbAdSource.MB_AD_SOURCE_BIDDING, errorMsg))
        }

        // 是否需要检查 context
        if (checkContextNonNull()) {
            if (getContext() == null) {
                return onBiddingError(
                    TAdErrorCode(
                        MbAdSource.MB_AD_SOURCE_BIDDING, "context is null"
                    )
                )
            }
        }

        // 耗时任务在子线程中完成
        withContext(Dispatchers.IO) {
            // 数据库获取所有可用计划
            val planList = AdPlansStorageManager.getAdPlan(
                sceneId = getSceneId(), ctxMap = getCtxMap(), logTag = getLogTag()
            )
            // 没有可用计划直接返回失败
            if (planList.isEmpty()) {
                // 失败暂不上报
//                AdReportProvider.biddingReport(
//                    triggerId = RandomUtils.getTriggerId(),
//                    sceneId = getSceneId(),
//                    adType = getAdType(),
//                    msg = "there are currently no plans available",
//                    result = BiddingStateEnum.BIDDING_REPORT_BIDDING_FAIL
//                )
                return@withContext onBiddingError(
                    TAdErrorCode(
                        MbAdSource.MB_AD_SOURCE_BIDDING, "there are currently no plans available"
                    )
                )
            }

            // 找出所有符合条件的广告计划
            val pair = getBiddingPlan(planList)
            biddingPlanList = pair.first
            val hasHiAdPlan = pair.second

            // 输出一下获取的广告计划数量
            onLog(
                level = Log.DEBUG,
                msg = "loadAd() --> 开始竞价 --> sceneId = ${getSceneId()} --> ctxMap = ${getCtxMap()} --> planList?.size = ${planList.size} --> hasHiAdPlan = $hasHiAdPlan --> biddingPlanList.size = ${biddingPlanList?.size}"
            )

            // 如果仅有包断没有Hi程序化，那就不需要竞价时间
            if (hasHiAdPlan) {
                mBiddingHandler.postDelayed({
                    onBiddingTime()
                }, getBiddingTime())
            } else {
                onBiddingTime()
            }
        }
    }

    /**
     * 获取竞价时间
     *
     * 目前仅有Banner需要竞价时间
     */
    private fun getBiddingTime(): Long {
        return if (getAdType() == MbAdType.MB_AD_TYPE_BANNER) {
            SceneCommonConfig.getBiddingTime(getSceneId()) * 1000L
        } else {
            0
        }
    }

    /**
     * 竞价时间到 需要调用此方法
     */
    private fun onBiddingTime() {
        // 选出ECPM最高的那个广告计划返回
        maxEcpmObject = getMaxEcpmBiddingPlan(getBiddingPlanList())

        if (maxEcpmObject == null) {
            // 失败 暂不上报
//            AdReportProvider.biddingReport(
//                triggerId = RandomUtils.getTriggerId(),
//                sceneId = getSceneId(),
//                adType = getAdType(),
//                msg = "竞价失败 --> maxEcpmObject == null --> biddingPlanList?.size = ${getBiddingPlanList()?.size}",
//                result = BiddingStateEnum.BIDDING_REPORT_BIDDING_FAIL
//            )

            onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING,
                    "竞价失败 --> maxEcpmObject == null --> biddingPlanList?.size = ${getBiddingPlanList()?.size}"
                )
            )
        } else {
            // 通知场景加载成功
            onBiddingLoad(maxEcpmObject)
        }
    }

    /**
     * 获取最大值
     * 根据比较规则，ecpm 优先级更高，其次是 plans?.sort 的排序值
     *
     * compareBy：优先按照 ecpm 字段进行比较。
     * 如果 ecpm 为 null，默认会将其视为较小的值。
     *
     * thenBy：在 ecpm 相等的情况下，按照 plans?.sort 字段进行比较。
     * 如果 plans 为 null 或 sort 为 null，则默认取 Int.MAX_VALUE。
     *
     * maxWithOrNull：根据自定义的比较器，返回 biddingPlanList 中的最大对象。如果集合为空，返回 null。
     */
    private fun getMaxEcpmBiddingPlan(biddingList: MutableList<BiddingIntermediateMaterialBean>?): BiddingIntermediateMaterialBean? {
        if (biddingList.isNullOrEmpty()) {
            onLog(Log.WARN, "getMaxEcpmBiddingPlan() --> biddingList.isNullOrEmpty() == true")
            return null
        }

        // 找到最大的对象
        //
        // 如果 ecpm 和 sort 都相等，比较器认为这些对象是“等价”的。
        // 在 biddingList 中，如果存在多个这样的对象，maxWithOrNull() 会返回它们中最后出现的那个对象（也就是在原始列表中靠后的那一个）。
        //
        // index	ecpm	sort
        //  0	    10	    5
        //  1	    10	    5
        //  2	    10	    5
        // 那么 maxWithOrNull() 返回的是 index 为 2 的那个对象。
        return biddingList.maxWithOrNull(compareBy<BiddingIntermediateMaterialBean> { it.ecpm }.thenBy {
            it.plans?.sort ?: Int.MAX_VALUE
        })
    }


    // ======================================= 内部逻辑 ==============================================


    /**
     * 获取竞价计划集合
     */
    private suspend fun getBiddingPlan(planList: MutableList<AdPlans>): Pair<MutableList<BiddingIntermediateMaterialBean>, Boolean> {
        val biddingPlan = mutableListOf<BiddingIntermediateMaterialBean>()
        // 获取广告计划对应的素材资源
        var hasHiAdPlan = false
        planList.forEach {
            when (it.adSource) {
                AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_HI.value -> {
                    // 需要判断开关是否打开
                    if (SceneOnOff.isSceneHiOff(getSceneId()).not()) {
                        // 前置判断条件
                        val hiSavanaPlacementId =
                            SceneCommonConfig.getHiSavanaPlacementId(getSceneId())
                        if (TextUtils.isEmpty(hiSavanaPlacementId).not()) {
                            // 添加HiSavana 广告源
                            addHiSavanaProvider(biddingPlan, it)
                            hasHiAdPlan = true
                        } else {
                            onLog(level = Log.WARN, msg = "hiSavanaPlacementId is empty")
                        }
                    } else {
                        onLog(level = Log.WARN, msg = "程序化广告场景关闭")
                    }
                }

                AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS.value -> {
                    if (SceneOnOff.isSceneNonOff(getSceneId()).not()) { // 场景没有关闭
                        // PS Offer 获取成功 添加广告计划
                        if (PsOfferProvider.getPsAdPlans(it)) {
                            biddingPlan.add(
                                BiddingIntermediateMaterialBean(
                                    ecpm = it.bidEcpmCent,
                                    plans = it,
                                    nativeInfo = null,
                                    isExpend = false
                                )
                            )
                        } else {
                            onLog(level = Log.WARN, msg = "PS Offer 获取失败")
                        }
                    } else {
                        onLog(level = Log.WARN, msg = "包断广告场景关闭")
                    }
                }

                else -> {
                    if (SceneOnOff.isSceneNonOff(getSceneId()).not()) {
                        // 包断广告默认所有的素材都是可用的
                        biddingPlan.add(
                            BiddingIntermediateMaterialBean(
                                ecpm = it.bidEcpmCent,
                                plans = it,
                                nativeInfo = null,
                                isExpend = false
                            )
                        )
                    } else {
                        onLog(level = Log.WARN, msg = "包断广告场景关闭")
                    }
                }
            }
        }
        return Pair(biddingPlan, hasHiAdPlan)
    }
}