package com.transsion.ad.bidding.base

import android.app.Activity
import android.content.Context
import android.os.SystemClock
import android.text.TextUtils
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.TextureView
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import com.blankj.utilcode.util.SizeUtils
import com.bumptech.glide.Glide
import com.transsion.ad.R
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.log.ILog
import com.transsion.ad.monopoly.model.AdMaterialList
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.strategy.MeasureManager
import com.transsion.ad.util.ViewUtil
import com.transsion.player.MediaSource
import com.transsion.player.config.VodConfig
import com.transsion.player.enum.ScaleMode
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.ORPlayer
import com.transsion.player.orplayer.PlayError
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author: zhangxinbing
 * @date : 2025/5/27 11:37
 * @description: 自定义包断广告 当前仅处理曝光展示时间监测
 */
abstract class AbsBiddingBuyOutView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null,
) : FrameLayout(context, attrs), MeasureManager.ViewVisibilityListener, ILog {

    private var mSceneId: String? = null
    private var mListener: BiddingTAdditionalListener? = null

    /**
     * 广告计划
     */
    private var mAdPlans: AdPlans? = null
    private var mAdMaterialList: AdMaterialList? = null

    /**
     * PS 兜底 Offer 数据
     */
    private var mRecommendInfo: RecommendInfo? = null

    /**
     * 播放器
     */
    private var mOrPlayer: ORPlayer? = null
    protected fun getOrPlayer(): ORPlayer? = mOrPlayer
    private var mIsMute: Boolean = true // 是否打开视频的声音

    /**
     * 复用View
     */
    private val imageView = ImageView(this.context)
    private val textureView = TextureView(this.context)


    // =============================================================================================


    init {
        // 统一处理点击事件 --> 曝光之后可点击
        this.setOnClickListener {
            onAdClick()
        }
    }


    // =============================== 曝光监测 =====================================================


    /**
     * 这里后面可以优化
     *
     * 默认 true，调用showMedia() 设置为false
     */
    private var isReportAdDisplay: AtomicBoolean = AtomicBoolean(false) // 是否已经上报过一次曝光了
    private var showTimestamp = 0L // 广告开始展示的时间 时间戳 毫秒

    override fun onVisibilityChanged(isVisible: Boolean) {
        if (isVisible) {
            // 开始展示的时间
            if (showTimestamp == 0L) {
                showTimestamp = SystemClock.elapsedRealtime()
            }

            // 曝光上报
            if (isReportAdDisplay.get().not()) {
                isReportAdDisplay.set(true)
                onAdDisplay()
            }

            // 页面可见的时候继续播放
            if (getOrPlayer()?.isPlaying() == false) {
                getOrPlayer()?.play()
            }

        } else {
            if (showTimestamp > 0) {
                onAdDisplayTimestamp(SystemClock.elapsedRealtime() - showTimestamp)
            }
            showTimestamp = 0L

            getOrPlayer()?.pause() // 页面不可见的时候暂停播放
        }
    }

    override fun getVisibilityView(): View? = getMediaContainer()


    // =============================================================================================


    abstract fun getMediaContainer(): FrameLayout?

    abstract fun isShowVolumeIcon(): Boolean


    // =============================================================================================


    /**
     * 场景信息
     */
    fun setSceneId(sceneId: String?) {
        mSceneId = sceneId
    }

    override fun getSceneId(): String? = mSceneId

    /**
     * 广告加载回调
     */
    fun setListener(listener: BiddingTAdditionalListener?) {
        mListener = listener
    }

    fun getListener(): BiddingTAdditionalListener? = mListener

    /**
     * 广告计划数据
     */
    fun setAdPlans(plans: AdPlans?) {
        mAdPlans = plans
        // 目前规定一个计划只有一个素材
        if (mAdPlans?.adMaterialList?.isNotEmpty() == true) {
            mAdMaterialList = mAdPlans?.adMaterialList?.get(0)
        }
    }

    fun getAdPlans(): AdPlans? = mAdPlans

    fun getAdMaterialList(): AdMaterialList? = mAdMaterialList

    /**
     * PS 兜底 Icon 广告数据
     */
    fun setRecommendInfo(recommendInfo: RecommendInfo?) {
        mRecommendInfo = recommendInfo
    }

    fun getRecommendInfo(): RecommendInfo? = mRecommendInfo

    /**
     * 资源回收
     */
    fun destroy() {
        isReportAdDisplay.set(false)
        reset()
        MeasureManager.removeSession(this@AbsBiddingBuyOutView)
    }

    /**
     * reset
     */
    fun reset() {
        // 资源回收之前需要判断一下是否需要上报一下展示时长埋点
        if (showTimestamp > 0) {
            onAdDisplayTimestamp(SystemClock.elapsedRealtime() - showTimestamp)
        }
        setListener(null)
        removeAllViews()
        mOrPlayer?.release()
    }

    /**
     * 修改曝光状态
     */
    fun isReportAdDisplay(isReportAdDisplay: Boolean) {
        this.isReportAdDisplay.set(isReportAdDisplay)
    }


    // =============================================================================================


    /**
     * 广告曝光回调
     *
     * 生命周期内仅曝光一次
     */
    private fun onAdDisplay() {
        getAdPlans()?.let {
            getListener()?.onBiddingBuyOutDisplay(it) // 回调出去统一处理
        }
        getRecommendInfo()?.let {
            getListener()?.onMbIconShow(it)
        }
    }

    /**
     * 广告点击
     */
    fun onAdClick() {
        getAdPlans()?.let {
            getListener()?.onBiddingBuyOutClick(it)
        }
        getRecommendInfo()?.let {
            getListener()?.onMbIconClick(it)
        }
    }

    /**
     * 广告累计时间回调
     */
    private fun onAdDisplayTimestamp(displayTimestamp: Long) {
        getAdPlans()?.let {
            getListener()?.onBiddingBuyOutDisplayTimestamp(it, displayTimestamp)
        }
        getRecommendInfo()?.let {
            getListener()?.onMbIconDisplayTimestamp(it, displayTimestamp)
        }
    }


    // ================================== 展示Native广告 ===================================================


    /**
     * 曝光监测
     *
     * TODO 曝光监测添加时机点需要优化，待多媒体展示出来之后添加监听
     */
    fun addSession() {
        MeasureManager.addSession(this@AbsBiddingBuyOutView)
    }

    /**
     * 展示媒体
     */
    fun showMedia() {
        if (getAdMaterialList() == null) {
            return
        }

        if (getAdMaterialList()?.type == AdMaterialList.NON_AD_TYPE_TEXT) {
            showImage(getAdMaterialList())
        } else {
            playVideo(getAdMaterialList())
        }

        // 添加曝光监测
        addSession()
    }

    /**
     * 加载图片
     */
    private fun showImage(adMaterialList: AdMaterialList?) {
        val mediaContainer = getMediaContainer() ?: return

        // 安全地获取 Activity
        val context = mediaContainer.context
        val activity = context as? Activity
        // 检查 Activity 是否有效
        if (activity != null && (activity.isFinishing || activity.isDestroyed)) {
            return // Activity 已经销毁，不加载图片
        }

        mediaContainer.apply {
            removeAllViews()
            val imageView = ImageView(context)
            imageView.scaleType = ImageView.ScaleType.CENTER_CROP

            val imagePath = adMaterialList?.image?.path
            val imageUrl = adMaterialList?.image?.url
            val glideLoad = if (TextUtils.isEmpty(imagePath)) imageUrl else imagePath

            try {
                Glide.with(context).load(glideLoad).centerCrop().into(imageView)
            } catch (e: IllegalArgumentException) {
                e.printStackTrace() // 记录异常，避免崩溃
            }
            addView(imageView)
        }
    }

    /**
     * 加载视频
     */
    private fun playVideo(adMaterialList: AdMaterialList?) {
        getMediaContainer()?.apply {
            // 初始化
            getOrPlayer()?.release() // 列表滑动View会复用
            removeAllViews() // 删除所有View
            mIsMute = true // 默认静音

            // 创建展示视频的View
            ViewUtil.removeSelfFromParent(textureView)
            addView(textureView)

            // 控制音频的按钮
            if (isShowVolumeIcon()) {
                val layoutParams = LayoutParams(SizeUtils.dp2px(24f), SizeUtils.dp2px(24f))
                layoutParams.gravity = Gravity.END or Gravity.TOP
                layoutParams.topMargin = SizeUtils.dp2px(8f)
                layoutParams.marginEnd = SizeUtils.dp2px(8f)
                imageView.tag = "adVolumeImage"
                imageView.setOnClickListener {
                    onLog(
                        level = Log.DEBUG,
                        msg = "playVideo() --> 点击了音频按钮",
                        writeToFile = false
                    )
                    // 设置静音
                    mIsMute = mIsMute.not()
                    mOrPlayer?.setMute(mIsMute)
                    setVolumeImage(imageView)
                }
                setVolumeImage(imageView) // 默认样式
                ViewUtil.removeSelfFromParent(imageView)
                addView(imageView, layoutParams)
            }

            // 创建播放器
            mOrPlayer = ORPlayer.Builder(context)
                .vodConfig(VodConfig(clearFrameWhenStop = false, openAudioFocus = false)).builder()
                .apply {
                    setMute(true)
                    setTextureView(textureView)
                    setScaleMode(ScaleMode.SCALE_ASPECT_FILL)
                    setLooping(true)
                    setPlayerListener(object : IPlayerListener {

                        override fun onPlayError(
                            errorInfo: PlayError, mediaSource: MediaSource?,
                        ) {
                            super.onPlayError(errorInfo, mediaSource)
                            onLog(
                                level = Log.ERROR,
                                msg = "playVideo() --> onPlayError() --> errorInfo = $errorInfo -- sceneId = ${getSceneId()}"
                            )
                        }

                        override fun onPrepare(mediaSource: MediaSource?) {
                            super.onPrepare(mediaSource)
                            play()
                        }

                        override fun onVideoPause(mediaSource: MediaSource?) {
                            super.onVideoPause(mediaSource)
//                            onLog(
//                                level = Log.DEBUG,
//                                msg = "playVideo() --> onVideoPause() --> sceneId = ${getSceneId()}"
//                            )
                        }

                        override fun onVideoStart(mediaSource: MediaSource?) {
                            super.onVideoStart(mediaSource)
//                            onLog(
//                                level = Log.DEBUG,
//                                msg = "playVideo() --> onVideoStart() --> sceneId = ${getSceneId()}"
//                            )
                        }

                    })
                }

            // 设置资源
            if (TextUtils.isEmpty(adMaterialList?.video?.path)) {
                mOrPlayer?.setDataSource(
                    MediaSource(
                        adMaterialList?.video?.url ?: "", adMaterialList?.video?.url ?: ""
                    )
                )
            } else {
                mOrPlayer?.setDataSource(
                    MediaSource(
                        adMaterialList?.video?.path ?: "", adMaterialList?.video?.path ?: ""
                    )
                )
            }

            // 开始播放
            mOrPlayer?.prepare()
        } ?: run {
            onLog(level = Log.ERROR, msg = "playVideo() --> getMediaContainer() == null")
        }
    }

    /**
     * 设置音频按钮的Icon
     */
    private fun setVolumeImage(imageView: ImageView) {
        if (mIsMute) {
            imageView.setImageResource(R.mipmap.ad_volumeoff)
        } else {
            imageView.setImageResource(R.mipmap.ad_volumeon)
        }
    }

}