package com.transsion.ad.bidding.gemini

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.blankj.utilcode.util.Utils
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.log.ILog
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.scene.SceneCommonConfig

/**
 * @author shmizhangxinbing
 * @date : 2025/6/3 14:08
 * @description: 开屏广告、插屏广告、激励视频广告
 * 通过广播传递事件
 */
abstract class AbsBiddingBuyOutGemini : ILog {

    companion object {
        const val EXT_SCENE_ID = "scene_id"         // 场景ID
        const val EXT_AD_PLAN = "ad_plan"         // 场景ID
        const val EXT_APP_LAYOUT_ID = "app_layout_id"     // App 信息布局
        const val EXT_DISPLAY_TIMESTAMP = "display_timestamp"     // App 信息布局

        const val EXT_ACTION_DISPLAY = "action_display"
        const val EXT_ACTION_CLICK = "action_click"
        const val EXT_ACTION_REWARDED = "action_rewarded"
        const val EXT_ACTION_DISPLAY_TIMESTAMP = "action_display_timestamp"
        const val EXT_ACTION_CLOSE = "action_close"

        const val ACTIVITY_STYLE_AUTO_FULLSCREEN = "auto_fullscreen"
    }


    // =============================================================================================


    /**
     * 传递过来的数据
     */
    private var mListener: BiddingTAdditionalListener? = null // 广告监听回调 加载的场景提供
    private var mSceneId = ""

    private var isReceiverRegistered = false // 一个生命周期仅注册一次

    /**
     * 创建一个BroadcastReceiver
     */
    private val myReceiver: BroadcastReceiver = object : BroadcastReceiver() {
        @Override
        override fun onReceive(context: Context?, intent: Intent) {
            // 子类自己处理对应的类型
            onSubReceive(context, intent)
        }
    }


    // =============================================================================================


    private fun getListener(): BiddingTAdditionalListener? = mListener
    override fun getSceneId(): String? = mSceneId

    /*** 广告加载回调*/
    fun setListener(listener: BiddingTAdditionalListener?): AbsBiddingBuyOutGemini {
        mListener = listener
        return this
    }

    fun setSceneId(sceneId: String): AbsBiddingBuyOutGemini {
        mSceneId = sceneId
        return this
    }


    // =============================================================================================


    /*** 注册事件广播*/
    fun registerReceiver(): AbsBiddingBuyOutGemini {
        if (isReceiverRegistered.not()) {
            isReceiverRegistered = true
            val filter = IntentFilter()
            filter.addAction(getDisplayAction())            // 展示
            filter.addAction(getClickAction())              // 点击
            filter.addAction(getRewardedAction())           // 获取激励
            filter.addAction(getDisplayTimestampAction())   // 展示时间
            filter.addAction(getCloseAction())              // 关闭
            LocalBroadcastManager.getInstance(Utils.getApp()).registerReceiver(myReceiver, filter)
        }
        return this
    }

    /*** 资源回收*/
    fun destroy() {
        LocalBroadcastManager.getInstance(Utils.getApp()).unregisterReceiver(myReceiver)
        setListener(null)
        isReceiverRegistered = false
    }

    /*** 获取当前页面的样式*/
    fun getAdActivityStyle(): String {
        return SceneCommonConfig.getAdActivityStyle(getSceneId())
    }

    /**
     * 开屏广告使用
     */
    open fun getAppLayoutId(): Int? = null

    fun startActivity(context: Context?, sceneId: String, plans: AdPlans?) {
        runCatching {
            context?.let {
                //val intent = Intent(it, BiddingBuyOutSplashActivity::class.java)
                val intent = Intent(it, getActivityClass())
                intent.putExtra(EXT_SCENE_ID, sceneId) // 场景信息
                intent.putExtra(EXT_AD_PLAN, plans) // 广告计划
                intent.putExtra(EXT_APP_LAYOUT_ID, getAppLayoutId()) // 开屏广告使用到的

                intent.putExtra(EXT_ACTION_DISPLAY, getDisplayAction())
                intent.putExtra(EXT_ACTION_CLICK, getClickAction())
                intent.putExtra(EXT_ACTION_DISPLAY_TIMESTAMP, getDisplayTimestampAction())
                intent.putExtra(EXT_ACTION_REWARDED, getRewardedAction())
                intent.putExtra(EXT_ACTION_CLOSE, getCloseAction())
                it.startActivity(intent)
            }
        }.getOrElse {
            getListener()?.onBiddingBuyOutShowError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BUY_OUT, "${it.message}"
                ), null
            )
        }
    }


    // =============================================================================================


    /**
     * 获取广告标识
     */
    abstract fun getAdTag(): String

    /**
     * 子类可以根据不同的配置跳转不同的页面
     */
    abstract fun getActivityClass(): Class<*>?


    // ====================================== 下面是广播处理 ==========================================


    /**
     * 意图配置
     */
    private fun getDisplayAction(): String = "com.transsion.ad.bidding.${getAdTag()}.display"
    private fun getClickAction(): String = "com.transsion.ad.bidding.${getAdTag()}.click"
    private fun getRewardedAction(): String = "com.transsion.ad.bidding.${getAdTag()}.rewarded"
    private fun getDisplayTimestampAction(): String =
        "com.transsion.ad.bidding.${getAdTag()}.display_timestamp"

    private fun getCloseAction(): String = "com.transsion.ad.bidding.${getAdTag()}.close"

    /**
     * 子类处理对应的广告
     */
    private fun onSubReceive(context: Context?, intent: Intent) {
        // val data = intent.getStringExtra("key")
        // 接受传递过来的数据
        val mAdPlan = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            intent.getParcelableExtra(EXT_AD_PLAN, AdPlans::class.java)
        } else {
            intent.getParcelableExtra(EXT_AD_PLAN)
        }
        val displayTimestamp = intent.getLongExtra(EXT_DISPLAY_TIMESTAMP, 0)

//        // 执行相关的操作
//        if (displayTimestamp > 0) {
//            onLog(
//                Log.DEBUG,
//                "onSubReceive() --> 包断广告 --> 接收到本地广播 --> action = ${intent.action} --> displayTimestamp = $displayTimestamp"
//            )
//        } else {
//            onLog(
//                Log.DEBUG,
//                "onSubReceive() --> 包断广告 --> 接收到本地广播 --> action = ${intent.action}"
//            )
//        }

        when (intent.action) {
            getDisplayAction() -> {
                getListener()?.onBiddingBuyOutDisplay(mAdPlan)
            }

            getClickAction() -> {
                getListener()?.onBiddingBuyOutClick(mAdPlan)
            }

            getRewardedAction() -> {
                getListener()?.onBiddingBuyOutRewarded(mAdPlan)
            }

            getDisplayTimestampAction() -> {
                getListener()?.onBiddingBuyOutDisplayTimestamp(mAdPlan, displayTimestamp)
            }

            getCloseAction() -> {
                getListener()?.onBiddingBuyOutClose(mAdPlan)
            }
        }
    }
}