package com.transsion.ad.bidding.icon

import android.os.Handler
import android.os.Looper
import android.text.TextUtils
import android.util.Log
import com.hisavana.common.bean.AdditionalInfo
import com.hisavana.common.bean.TAdErrorCode
import com.hisavana.common.bean.TAdNativeInfo
import com.hisavana.mediation.ad.TNativeAd
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.monopoly.model.MbAdType
import com.transsion.ad.ps.PsLinkUtils
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.report.AdReportProvider
import com.transsion.ad.scene.SceneOnOff
import com.transsion.ad.scene.SceneStorage
import com.transsion.ad.strategy.AdClickManager
import com.transsion.ad.util.RandomUtils
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/8/20 20:27
 * @description: Icon 广告加载,当前仅处理数据的加载和曝光处理
 *
 * 1. 并发请求 --> Hi程序化+PS兜底
 * 2. 竞价时间到 --> 组合返回数据
 * 3. 回调结果 --> 将结果回调
 */
class BiddingIconAdManager : BiddingTAdditionalListener() {

    /**
     * 两个接口暂存数据
     */
    private var temporaryStorageHi: MutableList<TAdNativeInfo> = mutableListOf()    // Hi程序化结果集合
    private var temporaryStoragePs: MutableList<RecommendInfo> = mutableListOf()    // PS兜底接口结果集合
    private var valueList: MutableList<BiddingWrapperIconBean>? = mutableListOf()   // 返回结果结合

    private fun getClassTag(): String = javaClass.simpleName

    private var mSceneId: String? = null                            // 广告场景ID
    private var mCallback: BiddingTAdditionalListener? = null       // 回调
    private var targetNum: Int = 0                                  // 默认返回一条数据
    private var isLoading: AtomicBoolean = AtomicBoolean(false)     // 标记是否正在加载中

    /*** 用于竞价计时*/
    private val mHandler by lazy {
        Handler(Looper.getMainLooper())
    }

    /*** 延时活动*/
    private val delayRunnable = Runnable {
        assemblyData("竞价时间到")
        isLoading.set(false)
    }

    /**
     * Icon 广告加载对象
     */
    private var hiSavanaIconAdManager: BiddingHiSavanaIconAdManager? = null
    private var pSDistribution: BiddingPSDistributionProvider? = null

    /**
     * 展示Icon的View
     */
    private val mBiddingWrapperIconView = mutableSetOf<BiddingWrapperIconView>()

    /**
     * 埋点使用 链路关系
     */
    private var triggerId: String = ""
    private fun getTriggerId(): String = triggerId


    // ================================ 广告数据结果回调 ==============================================


    /**
     * PS 兜底
     */
    override fun onPSDistributionReady(data: MutableList<RecommendInfo>?) {
        super.onPSDistributionReady(data)
        AdLogger.logIcon("${getClassTag()} --> onPSDistributionReady() --> mSceneId = ${getSceneId()} --> data.size = ${data?.size}")
        temporaryStoragePs.clear()
        data?.let {
            temporaryStoragePs.addAll(data)
        }
        // 其他数据是否回来了
        if (temporaryStorageHi.isNotEmpty()) {
            mHandler.post {
                assemblyData("数据都回来了me --> onPSDistributionReady")
            }
        }
    }

    override fun onMbIconShow(info: RecommendInfo?) {
        super.onMbIconShow(info)
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            psId = info?.id,
            psPackageName = info?.packageName
        )

        // 保存已经展示标识
        valueList?.forEach {
            if (it.recommendInfo?.id == info?.id) {
                it.isDisplay = true
            }
        }

        AdLogger.logIcon("${getClassTag()} --> onMbIconShow() --> sceneId = ${getSceneId()}")
    }

    override fun onMbIconClick(info: RecommendInfo?) {
        super.onMbIconClick(info)
        // 统一上报点击埋点
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            psId = info?.id,
            psPackageName = info?.packageName
        )

        // ICON 点击事件需要特殊处理
        // 广告点击回调
        AdClickManager.onClick(null)

        // 本地检测没有安装 -- 打开PS半屏
        // 需要检测已安装的ps版本是否支持半屏，不支持的走全屏
        PsLinkUtils.adClick(info, isAutoDownload = true, scene = getSceneId() ?: "")

        // 回调给场景
        mCallback?.onMbIconClick(info)

        AdLogger.logIcon("${getClassTag()} --> onMbIconClick() --> sceneId = ${getSceneId()}")
    }

    override fun onBiddingBuyOutError(p0: TAdErrorCode?, plans: AdPlans?) {
        super.onBiddingBuyOutError(p0, plans)
        AdLogger.logIcon("${getClassTag()} --> onBiddingBuyOutError() --> sceneId = ${getSceneId()} --> errorMessage = ${p0?.errorMessage}")
    }

    /**
     * 程序化
     */
    override fun onHiIconAdReady(tAdNativeInfos: MutableList<TAdNativeInfo>) {
        super.onHiIconAdReady(tAdNativeInfos)
        AdLogger.logIcon("${getClassTag()} --> onIconAdReady() --> mSceneId = ${getSceneId()} --> tAdNativeInfos.size =  = ${tAdNativeInfos.size}")

        // 保存数据
        tAdNativeInfos.let {
            temporaryStorageHi.clear()
            temporaryStorageHi.addAll(it)
        }
        // 其他数据是否回来了
        if (temporaryStoragePs.isNotEmpty()) {
            mHandler.post {
                assemblyData("数据都回来了me --> onHiIconAdReady")
            }
        }
    }

    override fun onShow(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onShow(p0, p1)
        // 统一上报展示埋点
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = p0?.adSource,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            psId = null
        )

        AdLogger.logIcon("${getClassTag()} --> onShow() --> sceneId = ${getSceneId()}")
    }

    override fun onClick(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onClick(p0, p1)
        // 统一上报点击埋点
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = p0?.adSource,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            psId = null
        )
        // 回调给场景
        mCallback?.onClick(p0, p1)

        AdLogger.logIcon("${getClassTag()} --> onClick() --> sceneId = ${getSceneId()}")
    }

    override fun onLoadFailure(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onLoadFailure(p0, p1)
        mCallback?.onLoadFailure(p0, p1)
        AdLogger.logIcon("${getClassTag()} --> onLoadFailure() --> sceneId = ${getSceneId()} --> errorMessage = ${p0?.errorMessage} --> placementId = ${p1.placementId}")
    }


    // =============================================================================================


    fun addBiddingWrapperIconView(iconView: BiddingWrapperIconView) {
        mBiddingWrapperIconView.add(iconView)
    }

    /**
     * 资源回收
     */
    fun destroy() {
        mHandler.removeCallbacksAndMessages(null)
        mCallback = null
        temporaryStorageHi.clear()
        temporaryStoragePs.clear()
        pSDistribution?.destroy()
        mBiddingWrapperIconView.forEach {
            it.destroy()
        }
        valueList?.clear()
        hiSavanaIconAdManager?.destroy()

        AdLogger.logIcon("${getClassTag()} --> destroy() --> 资源回收 --> sceneId = ${getSceneId()}")
    }

    fun getSceneId(): String? = mSceneId

    fun getListener(): BiddingTAdditionalListener = this

    fun getNativeAd(): TNativeAd? = hiSavanaIconAdManager?.getNativeAd()

    /**
     * 加载广告
     */
    suspend fun loadIconAd(
        sceneId: String,
        isFilteringInstalled: Boolean,
        targetNum: Int,
        listener: BiddingTAdditionalListener?
    ) {
        // 场景开关
        val errorMsg = SceneOnOff.isSceneOffV2(sceneId)
        if (TextUtils.isEmpty(errorMsg).not()) {
            onFailCallback(listener = listener, errorMsg = errorMsg)
            return
        }

        if (isLoading.get()) {
            return
        }
        isLoading.set(true)

        // 上报埋点
        triggerId = RandomUtils.getTriggerId()
        AdReportProvider.trigger(
            triggerId = getTriggerId(),
            sceneId = sceneId,
            adType = MbAdType.MB_AD_TYPE_ICON,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            planId = ""
        )

        mCallback = listener
        mSceneId = sceneId
        this.targetNum = targetNum

        // 加载广告
        innerLoadAd(sceneId, isFilteringInstalled)
    }


    // ===================================== 数据加载 ===============================================


    /**
     * 失败回调
     */
    private fun onFailCallback(listener: BiddingTAdditionalListener?, errorMsg: String) {
        AdLogger.logIcon(msg = errorMsg, level = Log.ERROR)
        listener?.onBiddingError(TAdErrorCode(MbAdSource.MB_AD_SOURCE_WRAPPER_AD, errorMsg))
    }

    /**
     * 获取优先级
     * ps hi
     */
    private fun isHiPriority(): Boolean {
        return TextUtils.equals(
            SceneStorage.getSceneConfig(getSceneId())?.get("priority")?.asString ?: "hi", "hi"
        )
    }

    /**
     * 加载广告
     */
    private suspend fun innerLoadAd(sceneId: String, isFilteringInstalled: Boolean) {
        val error = SceneOnOff.isSceneOffV2(sceneId)
        if (TextUtils.isEmpty(error).not()) {
            onFailCallback(listener = mCallback, errorMsg = error)
            return
        }

        val bindingTime = SceneStorage.getSceneConfig(getSceneId())?.get("bindingTime")?.asInt ?: 2
        AdLogger.logIcon(
            msg = "${getClassTag()} --> innerLoadAd() --> 开始加载广告 --> sceneId = $sceneId --> bindingTime = $bindingTime",
            writeToFile = false
        )

        // 场景开关判断
        if (SceneOnOff.isSceneHiOff(sceneId).not()) {
            // PS 接口和EW同时请求
            // 1. 加载HiSavana广告
            if (hiSavanaIconAdManager == null) {
                hiSavanaIconAdManager = BiddingHiSavanaIconAdManager()
                hiSavanaIconAdManager?.setSceneId(sceneId)
                hiSavanaIconAdManager?.setListener(this)
            }
            hiSavanaIconAdManager?.loadAd()
        } else {
            AdLogger.logIcon(
                msg = "${getClassTag()} --> innerLoadAd() --> 程序化广告关闭", level = Log.WARN
            )
        }

        // 场景开关判断
        if (SceneOnOff.isSceneNonOff(sceneId).not()) {
            // 2. 加载PS商单
            if (pSDistribution == null) {
                pSDistribution = BiddingPSDistributionProvider()
                pSDistribution?.setListener(this)
                pSDistribution?.setFilteringInstalled(isFilteringInstalled)
            }
            pSDistribution?.loadAd(scene = sceneId)
        } else {
            AdLogger.logIcon(
                msg = "${getClassTag()} --> innerLoadAd() --> 包断广告关闭", level = Log.WARN
            )
        }

        // 3. 竞价时间是4秒
        mHandler.postDelayed(delayRunnable, bindingTime * 1000L)
    }


    // ================================== 处理数据 ===================================================


    /**
     * 结果回来之后进行展示
     *
     * 因为是并行发起两个请求，最多等待4秒之后开始展示数据。
     */
    private fun assemblyData(msg: String = "") {
        // 重置延时消息
        mHandler.removeCallbacks(delayRunnable)
        runCatching {
            // 组装数据
            val excessiveList = combineAndRetrieveData(isHiFirst = isHiPriority())
            AdLogger.logIcon(msg = "${getClassTag()} --> assemblyData() --> msg = $msg --> targetNum = $targetNum --> 组装数据 --> size = ${excessiveList.size}")

            val value = if (targetNum > 0) {
                excessiveList.take(targetNum).toMutableList()
            } else {
                excessiveList
            }

            valueList?.clear()
            valueList?.addAll(value)

            mCallback?.onWrapperIconReady(value)
        }
    }

    private fun getHi(): BiddingWrapperIconBean {
        val downloadInterceptEntity = BiddingWrapperIconBean(
            type = BiddingWrapperIconBean.WRAPPER_ICON_ITEM_HI,
            recommendInfo = null,
            nativeInfo = temporaryStorageHi.removeAt(0)
        )
        return downloadInterceptEntity
    }

    private fun getPs(): BiddingWrapperIconBean {
        val downloadInterceptEntity = BiddingWrapperIconBean(
            type = BiddingWrapperIconBean.WRAPPER_ICON_ITEM_PS,
            recommendInfo = temporaryStoragePs.removeAt(0),
            nativeInfo = null
        )
        return downloadInterceptEntity
    }

    private fun combineAndRetrieveData(isHiFirst: Boolean = false): MutableList<BiddingWrapperIconBean> {
        val result = mutableListOf<BiddingWrapperIconBean>()
        var index = 0

        while (temporaryStorageHi.isNotEmpty() || temporaryStoragePs.isNotEmpty()) {
            val isHiTurn = if (isHiFirst) index % 2 == 0 else index % 2 == 1

            if (isHiTurn) {
                if (temporaryStorageHi.isNotEmpty()) {
                    result.add(getHi())
                } else if (temporaryStoragePs.isNotEmpty()) {
                    result.add(getPs())
                }
            } else {
                if (temporaryStoragePs.isNotEmpty()) {
                    result.add(getPs())
                } else if (temporaryStorageHi.isNotEmpty()) {
                    result.add(getHi())
                }
            }
            index++
        }
        return result
    }

}