package com.transsion.ad.bidding.nativead

import android.util.Log
import com.blankj.utilcode.util.Utils
import com.hisavana.common.bean.AdditionalInfo
import com.hisavana.common.bean.TAdErrorCode
import com.hisavana.common.bean.TAdNativeInfo
import com.hisavana.common.bean.TAdRequestBody
import com.hisavana.common.interfacz.TAdditionalListener
import com.hisavana.mediation.ad.TNativeAd
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.hi.HiSavanaAdManager
import com.transsion.ad.log.AdLogger
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.scene.SceneGlobalConfig
import java.util.concurrent.ConcurrentLinkedDeque
import java.util.concurrent.ConcurrentSkipListSet
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/5/29 15:40
 * @description:
 */
class BiddingHiSavanaNativeProvider : TAdditionalListener() {

    companion object {

        /**
         * 当前Map存储所有广告加载缓存池
         */
        private val map by lazy {
            mutableMapOf<String, BiddingHiSavanaNativeProvider>()
        }

        /**
         * 通过广告位ID -- 获取广告池 -- 在缓存池里面获取广告
         */
        fun getNativeManager(sceneId: String?): BiddingHiSavanaNativeProvider? {
            val placementId =
                SceneCommonConfig.getHiSavanaPlacementId(sceneId = sceneId) ?: return null

            if (placementId.isEmpty()) {
                return null
            }

            var manager = map[placementId]
            if (null == manager) {
                val adNativeManager = BiddingHiSavanaNativeProvider()
                adNativeManager.setPlacementId(placementId)
                map[placementId] = adNativeManager
                manager = adNativeManager
            }
            return manager
        }

    }


    private val taskList = ConcurrentLinkedDeque<BiddingTAdditionalListener>() // 任务队列 --> 添加、移除、遍历

    // 定义一个 ConcurrentSkipListSet，使用自定义 Comparator 按 ecpm 降序排序
    // 是一个线程安全的集合，属于 Java 的 java.util.concurrent 包，
    // 底层使用**跳表（SkipList）**作为数据结构来存储元素。
    // 它既保证了线程安全，又能确保元素始终按顺序排列。
    // 每次添加数据时都会自动按照指定的排序规则进行排序。
    private val nativeInfoList = ConcurrentSkipListSet<TAdNativeInfo>(Comparator { ad1, ad2 ->
        //ad2.ecpmPrice.compareTo(ad1.ecpmPrice) // 按 ecpm 降序排序
        val result = ad2.ecpmPrice.compareTo(ad1.ecpmPrice) // 按 ecpm 降序排序
        if (result == 0) {
            ad1.adId.compareTo(ad2.adId) // 次要排序条件（假设对象有唯一 id 字段）
        } else {
            result
        }
    })

    private var mPlacementId: String? = null // 场景调用的时候传进来的场景ID
    private var tNativeAd: TNativeAd? = null // 广告加载对象 负责广告的加载

    private var isLoading: AtomicBoolean = AtomicBoolean(false) // 当前正在请求中

    private fun getClassTag(): String = javaClass.simpleName


    // ============================广告SDK回调========================================================


    override fun onLoadSuccess(p0: AdditionalInfo) {
        super.onLoadSuccess(p0)

        // 获取所有NativeInfo
        tNativeAd?.nativeAdInfo?.let {
            nativeInfoList.addAll(it)
        }

        // 检查容量
        if (isUpperLimit().not()) {
            loadAd("缓存池没有达到阈值，继续请求")
        }
        isLoading.set(false)
    }

    override fun onLoadFailure(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onLoadFailure(p0, p1)
        isLoading.set(false)
        AdLogger.logSdk(
            tag = AdLogger.TAG_AD_NATIVE,
            msg = "${getClassTag()} --> onLoadFailure() --> placementId = ${getPlacementId()} --> errorMessage = ${p0?.errorMessage}",
            level = Log.ERROR
        )
    }

    override fun onShow(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onShow(p0, p1)
        taskList.forEach {
            it.onShow(p0, p1)
        }
    }

    override fun onClick(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onClick(p0, p1)
        taskList.forEach {
            it.onClick(p0, p1)
        }
    }


    // ============================================ 对外API =========================================


    /**
     * 场景ID
     */
    fun setPlacementId(placementId: String?) {
        mPlacementId = placementId
    }

    private fun getPlacementId(): String = mPlacementId ?: ""

    /**
     * 广告加载回调
     */
    fun addListener(listener: BiddingTAdditionalListener?) {
        taskList.add(listener)
    }

    fun removeListener(listener: BiddingTAdditionalListener?) {
        taskList.remove(listener)
    }

    /**
     * 判断当前缓存池是否有广告
     */
    fun hasCache(): Boolean {
        return nativeInfoList.isNotEmpty()
    }

    /**
     * 同步获取广告
     */
    fun getAdsSync(): TAdNativeInfo? {
        // 缓存池获取到了，那就直接返回
        val nativeInfo = getNativeInfo()
        if (isUpperLimit().not()) {
            loadAd("同步获取广告，继续装填缓存池")
        }
        return nativeInfo
    }

    /**
     * 将没有使用过的广告重新添加到缓存池中
     */
    fun addUnusedAdToPool(nativeInfo: TAdNativeInfo?) {
        nativeInfo?.let {
            nativeInfoList.add(it)
        }
    }

    fun getNativeAd(): TNativeAd? = tNativeAd


    // =============================================================================================


    /**
     * 从缓存池获取NativeInfo --> nativeInfoList
     */
    private fun getNativeInfo(): TAdNativeInfo? {
        // 缓存池是空的
        if (nativeInfoList.isEmpty()) {
            return null
        }

        // 过期了
        val info = nativeInfoList.pollFirst()
        if (info?.isExpired == true) {
            // 释放资源 --> 继续从缓存池获取
            info.release()
            return getNativeInfo()
        }

        // 返回结果
        return info
    }

    /**
     * 缓存池是否达到阈值,大于等于阈值了说明广告有富余没有消费完
     * 如果配置了0那就意味着不开启缓存的能力
     */
    private fun isUpperLimit(): Boolean {
        val cacheUpperLimit = SceneGlobalConfig.getNativeCacheUpperLimit()
        val b = nativeInfoList.size >= cacheUpperLimit
        if (b) {
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "${getClassTag()} --> isUpperLimit() --- 达到阈值了 --> nativeInfoList.size = ${nativeInfoList.size} -- cacheUpperLimit = $cacheUpperLimit",
                level = Log.WARN
            )
        }
        return b
    }

    /**
     * 加载广告 --> 现在仅支持串行请求
     */
    private fun loadAd(mag: String = "") {
        // 如果没有初始化，那就不进行任何操作
        if (HiSavanaAdManager.isInitialized().not()) {
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "${getClassTag()} --> loadAd($mag) --> 广告SDK没有初始化",
                level = Log.ERROR
            )
            return
        }

        // 如果正在请求中那就return 这个isLoading 仅在SDK回调的onLoad和onError修改
        if (isLoading.get()) {
            return
        }
        isLoading.set(true)

//        AdLogger.logSdk(
//            tag = AdLogger.TAG_AD_NATIVE,
//            msg = "${getClassTag()} --> loadAd() --> mag = $mag",
//            level = Log.DEBUG
//        )

        // 加载广告
        if (null == tNativeAd) {
            tNativeAd = TNativeAd(Utils.getApp(), getPlacementId())
            tNativeAd?.setRequestBody(
                TAdRequestBody.AdRequestBodyBuild().setAdditionalListener(this).build()
            )
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "============= ${getClassTag()} --> loadAd() --> mPlacementId = ${getPlacementId()} --> create TNativeAd() =================== ",
                level = Log.WARN
            )
        }
        tNativeAd?.loadAd()
    }

}