package com.transsion.ad.db

import android.content.Context
import androidx.room.Database
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.room.TypeConverters
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import com.transsion.ad.db.mcc.LocalMcc
import com.transsion.ad.db.mcc.LocalMccDao
import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.db.plan.MbAdPlansDao
import com.transsion.ad.db.pslink.AttributionPoint
import com.transsion.ad.db.pslink.AttributionPointDao
import com.transsion.ad.db.pslink.PsLinkAdPlan
import com.transsion.ad.db.pslink.PsLinkAdPlanDao
import com.transsion.ad.log.AdLogger

/**
 * @author: zhangxinbing
 * @date : 2024/5/13 17:17
 * @description: 字幕数据库入口
 */
@Database(
    entities = [MbAdDbPlans::class, LocalMcc::class, PsLinkAdPlan::class, AttributionPoint::class],
    version = 8,
    exportSchema = true
)
@TypeConverters(AttributionPoint.TypeConverter::class) // 注册 TypeConverter
abstract class MbAdDatabase : RoomDatabase() {

    abstract fun mbAdPlansDao(): MbAdPlansDao

    abstract fun localMccDao(): LocalMccDao

    abstract fun psLinkAdPlanDao(): PsLinkAdPlanDao

    abstract fun attributionPointDao(): AttributionPointDao


    // =============================================================================================


    companion object {

        @Volatile
        private var instance: MbAdDatabase? = null

        private const val MB_AD_DATABASE_NAME = "mb-ad-plans-db"

        /**
         * 数据库只能当前Module里面使用
         */
        fun getInstance(context: Context): MbAdDatabase {
            return instance ?: synchronized(this) {
                instance ?: buildDatabase(context).also { instance = it }
            }
        }

        // Create and pre-populate the database. See this article for more details:
        // https://medium.com/google-developers/7-pro-tips-for-room-fbadea4bfbd1#4785
        private fun buildDatabase(context: Context): MbAdDatabase {
            return Room.databaseBuilder(
                context.applicationContext, MbAdDatabase::class.java, MB_AD_DATABASE_NAME
            )
                //是否允许在主线程进行查询
                //.allowMainThreadQueries()
                .addMigrations(
                    MIGRATION_1_2,
                    MIGRATION_2_3,
                    MIGRATION_3_4,
                    MIGRATION_4_5,
                    MIGRATION_5_6,
                    MIGRATION_6_7,
                    MIGRATION_7_8
                ).addCallback(object : Callback() {
                    override fun onCreate(db: SupportSQLiteDatabase) {
                        super.onCreate(db)
                        AdLogger.logAdInit(msg = "${MbAdDatabase::class.simpleName} --> getInstance() --> buildDatabase() --> onCreate() --> db.path = " + db.path)
                    }

                    override fun onOpen(db: SupportSQLiteDatabase) {
                        super.onOpen(db)
                        AdLogger.logAdInit(msg = "${MbAdDatabase::class.simpleName} --> getInstance() --> buildDatabase() --> onOpen()")
                    }
                }).build()
        }

        /**
         *         MIGRATION
         * 如果有多个MIGRATION则往后添加：
        .addMigrations(MIGRATION_1_2, MIGRATION_3_4, MIGRATION_4_5, MIGRATION_5_6, MIGRATION_6_7)
         */
        private val MIGRATION_1_2 = object : Migration(1, 2) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // 广告表新增分发时间段
                database.execSQL("ALTER TABLE MB_AD_DB_PLANS ADD COLUMN dispatchTimeStart TEXT")
                database.execSQL("ALTER TABLE MB_AD_DB_PLANS ADD COLUMN dispatchTimeEnd TEXT")
            }
        }

        private val MIGRATION_2_3 = object : Migration(2, 3) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // mcc
                database.execSQL(
                    "CREATE TABLE IF NOT EXISTS LOCAL_MCC (`id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, " + "`Country` TEXT NOT NULL, `Mcc` TEXT NOT NULL, `Iso` TEXT NOT NULL, `CountryCode` TEXT NOT NULL)"
                )
            }
        }

        private val MIGRATION_3_4 = object : Migration(3, 4) {
            override fun migrate(database: SupportSQLiteDatabase) {
                database.execSQL("ALTER TABLE MB_AD_DB_PLANS ADD COLUMN adSource TEXT")
                database.execSQL("ALTER TABLE MB_AD_DB_PLANS ADD COLUMN extAdSlot TEXT")
                database.execSQL(
                    "CREATE TABLE IF NOT EXISTS ps_link_ad (`id` INTEGER NOT NULL,`nonId` TEXT NOT NULL, `adSource` TEXT NOT NULL, `extAdSlot` TEXT NOT NULL, `rank` INTEGER NOT NULL, `psPlanId` TEXT, `psLinkAdInfoStr` TEXT, `psInfoJson` TEXT, PRIMARY KEY(`id`))"
                )

                // 首页Banner展示广告计划 扩展图片
                database.execSQL("ALTER TABLE MB_AD_DB_PLANS ADD COLUMN extImage TEXT")
            }
        }

        private val MIGRATION_4_5 = object : Migration(4, 5) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // 广告表新增ecpm 字段
                database.execSQL("ALTER TABLE mb_ad_db_plans ADD COLUMN bidEcpmCent REAL")
                database.execSQL("ALTER TABLE mb_ad_db_plans ADD COLUMN ecpmCent INTEGER")
            }
        }

        private val MIGRATION_5_6 = object : Migration(5, 6) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // 新增时间戳字段，用户数据库获取数据判断条件
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN updateTimestamp INTEGER")
                // 新增 PS Offer 归因数据表
                database.execSQL("CREATE TABLE IF NOT EXISTS `attribution_points` (`id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, `psId` TEXT NOT NULL, `failCount` INTEGER NOT NULL, `reportUrl` TEXT NOT NULL, `type` TEXT NOT NULL, `updateTimestamp` INTEGER NOT NULL)")
            }
        }

        private val MIGRATION_6_7 = object : Migration(6, 7) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // Room 用 INTEGER 类型存储 Boolean，0 表示 false，1 表示 true。
                // NOT NULL DEFAULT 0 保证老数据迁移时 isValid 字段有默认值（false）。
                database.execSQL("ALTER TABLE mb_ad_db_plans ADD COLUMN isValid INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_7_8 = object : Migration(7, 8) {
            override fun migrate(database: SupportSQLiteDatabase) {
                // PS Offer 拉活 频控需求
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN showMax INTEGER")
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN clickMax INTEGER")
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN showHours TEXT")
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN showedTimes INTEGER NOT NULL DEFAULT 0")
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN clickedTimes INTEGER NOT NULL DEFAULT 0")
                database.execSQL("ALTER TABLE ps_link_ad ADD COLUMN showDate TEXT")
            }
        }

    }

}