package com.transsion.ad.monopoly.intercept

import android.text.TextUtils
import com.blankj.utilcode.util.GsonUtils
import com.transsion.ad.db.mcc.LocalMcc
import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.monopoly.model.AdPlansHitResponse
import com.transsion.ad.strategy.NationalInformationManager

/**
 * @author: zhangxinbing
 * @date : 2024/3/18 20:37
 * @description: 当前国家是否在白名单中判断
 */
internal class CountryInWhitelistHandler : AbsAdPlanInterceptHandler() {

    override suspend fun doHandle(
        sceneId: String, plans: MbAdDbPlans?, ctxMap: Map<String, Any>
    ): AdPlansHitResponse {

        val country = plans?.country
        val countryFromJson = GsonUtils.fromJson(country, Array<String>::class.java)
        // 如果没有配置国家，那就返回匹配成功
        val isCountryInWhitelist = if (countryFromJson.isNullOrEmpty()) {
            true
        } else {
            isCountryInWhitelist(countryFromJson, country)
        }

        // 当前条件通过继续下一个条件判断 -- 否则直接返回失败
        return if (isCountryInWhitelist) {
            getChain()?.doHandle(sceneId, plans, ctxMap) ?: getDefaultRes()
        } else {
            AdPlansHitResponse(isHit = false, rejectMsg = "当前国家不在白名单中")
        }
    }


    // =============================================================================================


    /**
     * 是否在白名单
     */
    private suspend fun isCountryInWhitelist(
        countryFromJson: Array<String>, planCountry: String?
    ): Boolean {
        // 通过国家获取 MCC
        val mccList = mutableListOf<LocalMcc>()
        countryFromJson.forEach {
            val localMccByColumnValue =
                NationalInformationManager.getNationalInformation(it.lowercase())
            if (localMccByColumnValue != null) {
                mccList.add(localMccByColumnValue)
            }
        }

        // 获取手机当前的Mcc
        val currentMcc = NationalInformationManager.getCurrentMcc()

        // 遍历匹配 MCC
        mccList.forEach {
            if (TextUtils.equals(it.Mcc, currentMcc)) {
                return true
            }
        }

        //logErrorMsg("isCountryInWhitelist() --> 当前国家不在白名单中 -- planCountry = $planCountry -- currentMcc = $currentMcc")
        return false
    }
}