package com.transsion.ad.monopoly.plan

import android.text.TextUtils
import com.transsion.ad.report.AdReportProvider
import com.transsion.ad.monopoly.intercept.NonAdShowedTimesManager
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.model.AdMaterialList
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdImage
import com.transsion.ad.monopoly.model.MbAdShowLevel
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.monopoly.model.MbAdType
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * @author: zhangxinbing
 * @date : 2025/3/4 10:32
 * @description: 获取广告计划
 */
object MonopolyAdPlanProvider {


    // ================================== 广告计划获取 ===============================================


    /**
     * 获取优先展示广告计划
     */
    suspend fun getShowFirstAdPlan(sceneId: String, ctxMap: Map<String, Any>): AdPlans? {
        return withContext(Dispatchers.IO) {
            AdPlansStorageManager.getAdPlans(sceneId = sceneId, ctxMap = ctxMap)
        }
    }

    /**
     * 获取兜底广告计划
     */
    suspend fun getShowFinalAdPlan(sceneId: String, ctxMap: Map<String, Any>): AdPlans? {
        return withContext(Dispatchers.IO) {
            AdPlansStorageManager.getAdPlans(
                sceneId = sceneId, adShowLevel = MbAdShowLevel.MB_AD_SHOW_FINAL, ctxMap = ctxMap
            )
        }
    }

    /**
     * 获取可用的计划
     */
    suspend fun getAdPlan(sceneId: String, ctxMap: Map<String, Any>): AdPlans? {
        // 2025-06-16 改造支持竞价
        val adPlanList = AdPlansStorageManager.getAdPlan(
            sceneId = sceneId, ctxMap = ctxMap, logTag = "MonopolyAdPlanProvider",
            //AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS,
            AdPlanSourceManager.AdPlanEnum.AD_PLAN_DEFAULT_AD_SRC
        )
        return AdPlanUtil.getHighestEcpmAdPlan(adPlanList)
        //return getShowFirstAdPlan(sceneId, ctxMap) ?: getShowFinalAdPlan(sceneId, ctxMap)
    }


    // ============================= 资源获取判断 ====================================================


    /**
     * 是否是视频
     */
    fun isVideo(plan: AdPlans): Boolean {
        return getMaterial(plan)?.type == AdMaterialList.NON_AD_TYPE_VIDEO
    }

    /**
     * 获取资源对象
     */
    fun getMaterial(plan: AdPlans): AdMaterialList? {
        val adMaterial = plan.adMaterialList
        return if (adMaterial?.isNotEmpty() == true) {
            adMaterial[0]
        } else {
            null
        }
    }

    /**
     * 获取图片的连接
     */
    fun getImageUrl(plan: AdPlans): String? {
        // 如果本地地址是空的，那就不使用线上地址在线加载
        val adMaterial = getMaterial(plan)
        return if (TextUtils.isEmpty(getMaterial(plan)?.image?.path)) {
            adMaterial?.image?.url
        } else {
            adMaterial?.image?.path
        }
    }

    /**
     * 获取图片对象
     */
    fun getImage(plan: AdPlans): MbAdImage? {
        return getMaterial(plan)?.image
    }

    /**
     * 获取视频的连接
     */
    fun getVideoUrl(plan: AdPlans): String? {
        // 如果本地地址是空的，那就不使用线上地址在线加载
        val adMaterial = getMaterial(plan)
        return if (TextUtils.isEmpty(getMaterial(plan)?.video?.path)) {
            adMaterial?.video?.url
        } else {
            adMaterial?.video?.path
        }
    }

    /**
     * 获取扩展图片
     */
    fun getExtImage(plan: AdPlans): MbAdImage? {
        return plan.extImage
    }


    // =================================== 埋点 =====================================================


    private fun getAdId(adPlan: AdPlans): String {
        val adMaterialList = adPlan.adMaterialList
        return if (adMaterialList?.isNotEmpty() == true) {
            adMaterialList[0].id ?: "null"
        } else {
            "null"
        }
    }

    /**
     * 触发加载
     */
    fun onAdPlanTrigger(sceneId: String, id: String?, triggerId: String) {
        // 广告触发 统一入口处理
        AdReportProvider.trigger(
            triggerId = triggerId,
            sceneId = sceneId,
            adType = MbAdType.MB_AD_TYPE_BANNER,
            adSource = MbAdSource.MB_AD_SOURCE_BIDDING,
            planId = id
        )
    }

    /**
     * 点击事件回调
     */
    fun onAdPlanClick(sceneId: String, @MbAdType adType: Int, adPlan: AdPlans?, triggerId: String) {
        if (adPlan == null) {
            return
        }

        AdReportProvider.adClick(
            triggerId = triggerId,
            sceneId = sceneId,
            adPlanId = adPlan.id,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = getAdId(adPlan),
            adType = adType,
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(adPlan),
            extMap = emptyMap(),
            psId = AdPlanUtil.getPsId(adPlan),
            bidEcpmCent = adPlan.bidEcpmCent,
            ecpmCent = adPlan.ecpmCent
        )
    }

    /**
     * 曝光回调
     */
    fun onAdPlanShow(sceneId: String, @MbAdType adType: Int, adPlan: AdPlans?, triggerId: String) {
        if (adPlan == null) {
            return
        }

        // 保存展示次数
        NonAdShowedTimesManager.saveShowedTimes(adPlan)

        // 上报
        AdReportProvider.display(
            triggerId = triggerId,
            sceneId = sceneId,
            adPlanId = adPlan.id,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = getAdId(adPlan),
            adType = adType,
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(adPlan),
            extMap = emptyMap(),
            psId = AdPlanUtil.getPsId(adPlan),
            bidEcpmCent = adPlan.bidEcpmCent,
            ecpmCent = adPlan.ecpmCent
        )
    }

    /**
     * 曝光时常回调
     */
    fun onAdPlanShowTime(
        sceneId: String, @MbAdType adType: Int, showTime: Long, adPlan: AdPlans?, triggerId: String
    ) {
        if (adPlan == null) {
            return
        }

        AdReportProvider.adShowTime(
            triggerId = triggerId,
            sceneId = sceneId,
            adPlanId = adPlan.id,
            displayTime = showTime,
            adId = getAdId(adPlan),
            adType = adType,
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(adPlan),
        )
    }
}