package com.transsion.ad.ps.attribution

import android.text.TextUtils
import android.util.Log
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.ad.db.pslink.AttributionPoint
import com.transsion.ad.http.OkHttpProvider
import com.transsion.ad.log.AdLogger
import com.transsion.ad.ps.PSReportUtil
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import okhttp3.Request
import okhttp3.Response
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/7/15 10:51
 * @description: 数据消费 仅处理消费
 */
object AttributionConsumeManager : BaseAttributionProvider() {

    private var isLoading = AtomicBoolean(false) // 当前处理串行消费
    private var point: AttributionPoint? = null  // 当前正在处理的归因埋点对象

    private const val FAILURE_LIMIT = 5 // 失败次数上限
    private const val NET_WORK_FAIL_SPACING_INTERVAL = 10 * 1000L // 网络请求失败 请求间隔


    // =============================================================================================


    /**
     * 触发消费数据
     */
    fun consume() {
        CoroutineScope(Dispatchers.IO).launch {
            runCatching {
                safeCall()
            }.getOrElse {
                AdLogger.logPS(
                    "${getClassTag()} --> consume() --> it = $it", level = Log.ERROR
                )

                // 网络请求异常 --> 这里延时处理 --> 避免不必要的资源浪费
                delay(NET_WORK_FAIL_SPACING_INTERVAL)
                next()
            }
        }
    }


    // =============================================================================================


    private suspend fun safeCall() {
        // 如果没有网络，直接返回等待下次触发
        if (NetworkUtil.hasCapabilityAsync().not()) {
//            AdBiddingLogger.logPS(
//                "${getClassTag()} --> consume() --> 当前无网络不处理", level = Log.ERROR
//            )
            return
        }

        // 有任务处理中
        if (isLoading.get()) {
            return
        }
        isLoading.set(true)

        // 当前没有数据了，等待下次触发
        point = getAttributionPoint()
        if (point == null) {
            isLoading.set(false) // 重置数据
            return
        }

        // 异常数据处理
        if (TextUtils.isEmpty(point?.reportUrl)) {
            //AdBiddingLogger.logPS("${getClassTag()} --> consume() --> point?.reportUrl isEmpty --> 删除数据 --> 处理下一条数据")
            delete(point) // 删除无效数据
            next() // 执行下一条数据
            return
        }

        // 网络请求
        doNetWork()
    }

    private fun next() {
        // 触发下一次任务
        isLoading.set(false)
        consume()
    }

    private suspend fun doNetWork() {
        AdLogger.logPS("${getClassTag()} --> consume() --> 开始归因 --> type = ${point?.type} --> id = ${point?.id} --> failCount = ${point?.failCount} --> psId = ${point?.psId} --> url = ${point?.reportUrl}")

        // 发起请求
        val request = Request.Builder().url(point?.reportUrl ?: "").build()
        val response: Response = OkHttpProvider.getOkHttpClient().newCall(request).execute()
        // 结果
        if (response.isSuccessful) {
            AdLogger.logPS("${getClassTag()} --> consume() --> 归因成功 --> type = ${point?.type} --> id = ${point?.id} --> psId = ${point?.psId} --> url = ${point?.reportUrl} --> response = ${response.message}")
            // 将数据删除，触发下一次消费
            delete(point)
            PSReportUtil.onPsAttribution(type = point?.type, psId = point?.psId)
        } else {
            AdLogger.logPS("${getClassTag()} --> consume() --> 归因失败 --> type = ${point?.type} --> id = ${point?.id} --> psId = ${point?.psId}--> url = ${point?.reportUrl} --> response = ${response.message}")
            // 修改数据
            point?.let { po ->
                po.failCount += 1
                // 超过最大失败次数
                if (po.failCount >= FAILURE_LIMIT) {
                    delete(po)
                } else {
                    update(po)
                }
            }
        }

        // 执行下一条数据
        next()
    }

}