package com.transsion.ad.view.native_ad

import android.content.Context
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.animation.Animation
import android.view.animation.TranslateAnimation
import android.widget.FrameLayout
import android.widget.ImageView
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.RecyclerView.OnScrollListener
import com.blankj.utilcode.util.SizeUtils
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.R
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.AbsBiddingBuyOutView
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.bidding.nativead.BiddingNativeManager
import com.transsion.ad.log.AdLogger
import com.transsion.ad.util.ViewUtil
import kotlinx.coroutines.launch

/**
 * @author shmizhangxinbing
 * @date : 2025/7/21 14:06
 * @description: 非标广告
 */
class BiddingFloatingIrregularView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
) : AbsBiddingBuyOutView(context, attrs) {

    private var mMaxEcpmObject: BiddingIntermediateMaterialBean? = null // 广告数据

    private var isCollapsed = false // 是否是折叠状态
    private var currentTranslationX = 0f // 记录当前的 X 偏移量

    private val mediaContainer: FrameLayout = FrameLayout(context) // 广告容器
    private val closeImg = ImageView(context) // 关闭按钮

    private var adManager: BiddingNativeManager? = null // 广告加载对象
    private val adListener = object : BiddingTAdditionalListener() {

        override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
            super.onBiddingLoad(maxEcpmObject)
            // 广告数据
            mMaxEcpmObject = maxEcpmObject

            // 设置数据
            setAdPlans(maxEcpmObject?.plans)
            // 展示
            showMedia()
            closeImg.visibility = View.VISIBLE
            visibility = View.VISIBLE // 整体可见
        }

        override fun onBiddingError(p0: TAdErrorCode?) {
            super.onBiddingError(p0)
//            onLog(
//                level = Log.ERROR,
//                msg = "${getClassTag()} --> onBiddingError() --> p0 = ${p0?.errorMessage}"
//            )
            visibility = View.GONE // 整体不可见
        }
    }


    // =============================================================================================


    init {

    }

    override fun getMediaContainer(): FrameLayout = mediaContainer

    override fun isShowVolumeIcon(): Boolean = false

    override fun getLogTag(): String = AdLogger.TAG_AD_IRREGULAR


    // =============================================================================================


    /**
     * 资源回收
     */
    fun onAdViewDestroy() {
        adManager?.destroy()
        removeAllViews()
        destroy()
    }

    /**
     * 修改状态
     */
    fun changeState(isCollapsed: Boolean) {
        if (mMaxEcpmObject == null) {
            return
        }

        if (isCollapsed == this.isCollapsed) {
            return
        }

        if (isCollapsed) {
            // 如果是展开状态，仅保留 1/3 的宽度可见
            animateCollapse()
        } else {
            // 如果是折叠状态，展开内容
            animateExpand()
        }
    }

    /**
     * 加载广告
     */
    fun loadAd() {

        // 已经展示了就不再重复加载
        if (mMaxEcpmObject != null) {
            return
        }

        if (adManager == null) {
            adManager = BiddingNativeManager()
            adManager?.setSceneId(getSceneId())
            adManager?.setListener(adListener)

            // 将View和Manager关联起来
            setListener(adManager)
        }

        addDefaultView()

        // 展示View
        animateExpand()

        (context as? AppCompatActivity)?.lifecycleScope?.launch {
            adManager?.loadAd()
        }
    }

    /**
     * 设置RecyclerView
     */
    fun setRecyclerView(recyclerView: RecyclerView?) {
        recyclerView?.addOnScrollListener(object : OnScrollListener() {
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                super.onScrollStateChanged(recyclerView, newState)
                when (newState) {
                    RecyclerView.SCROLL_STATE_DRAGGING -> {
                        // 当用户正在拖动 RecyclerView 时触发
                        changeState(isCollapsed = true)
                    }

                    RecyclerView.SCROLL_STATE_SETTLING -> {
                        // 当 RecyclerView 正在惯性滚动时触发
                        //println("RecyclerView is settling (flinging)")
                    }

                    RecyclerView.SCROLL_STATE_IDLE -> {
                        // 当 RecyclerView 静止时触发
                        changeState(isCollapsed = false)
                    }
                }
            }
        })
    }


    // =============================================================================================


    private fun addDefaultView(){
        // 广告主图
        ViewUtil.removeSelfFromParent(mediaContainer)
        addView(mediaContainer)

        // 关闭按钮
        closeImg.setImageResource(R.mipmap.ad_irregular_close)
        val layoutParams = LayoutParams(SizeUtils.dp2px(16f), SizeUtils.dp2px(16f))
        layoutParams.gravity = Gravity.END or Gravity.TOP
        closeImg.visibility = View.GONE
        closeImg.setOnClickListener {
            animateClose()
        }
        ViewUtil.removeSelfFromParent(closeImg)
        addView(closeImg, layoutParams)

        setOnClickListener {
            if (isCollapsed.not()) { // 展开才能点击
                onAdClick()
            }
        }
    }

    /**
     * 折叠内容，仅展示 1/3 宽度
     */
    private fun animateCollapse() {
        val toX = this.width * 4 / 5f // 向右移动 2/3 宽度
        val animation = TranslateAnimation(currentTranslationX, toX, 0f, 0f).apply {
            duration = 300 // 动画持续时间
            fillAfter = true // 不保留动画结束后的状态，避免影响位置更新
            setAnimationListener(object : Animation.AnimationListener {
                override fun onAnimationStart(animation: Animation?) {}
                override fun onAnimationRepeat(animation: Animation?) {}
                override fun onAnimationEnd(animation: Animation?) {
                    // 动画结束后更新实际位置
                    currentTranslationX = toX
                }
            })
        }
        this.startAnimation(animation)

        isCollapsed = true // 更新为折叠状态
        alpha = 0.4f
    }

    /**
     * 展开内容，显示完整内容
     */
    private fun animateExpand() {
        val animation = TranslateAnimation(currentTranslationX, 0f, 0f, 0f).apply {
            duration = 300 // 动画持续时间
            fillAfter = true // 不保留动画结束后的状态，避免影响位置更新
            setAnimationListener(object : Animation.AnimationListener {
                override fun onAnimationStart(animation: Animation?) {}
                override fun onAnimationRepeat(animation: Animation?) {}
                override fun onAnimationEnd(animation: Animation?) {
                    // 动画结束后更新实际位置
                    currentTranslationX = 0f
                }
            })
        }
        this.startAnimation(animation)

        isCollapsed = false // 更新为展开状态
        alpha = 1.0f
    }


    /**
     * 折叠内容
     */
    private fun animateClose() {
        val marginEnd = (layoutParams as? MarginLayoutParams)?.marginEnd ?: 0
        val toX = this.width.toFloat() + marginEnd// 向右移动关闭
        val animation = TranslateAnimation(currentTranslationX, toX, 0f, 0f).apply {
            duration = 300 // 动画持续时间
            fillAfter = true // 不保留动画结束后的状态，避免影响位置更新
            setAnimationListener(object : Animation.AnimationListener {
                override fun onAnimationStart(animation: Animation?) {}
                override fun onAnimationRepeat(animation: Animation?) {}
                override fun onAnimationEnd(animation: Animation?) {
                    // 动画结束后更新实际位置
                    currentTranslationX = toX
                }
            })
        }
        this.startAnimation(animation)

        // 释放资源
        destroy()
        adManager?.destroy()
        adManager = null
        mediaContainer.removeAllViews()
        mMaxEcpmObject = null

        isCollapsed = true // 更新为折叠状态
    }

}