package com.transsion.ad.bidding.gemini

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.blankj.utilcode.util.Utils
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.log.ILog
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.scene.SceneCommonConfig

/**
 * @author shmizhangxinbing
 * @date : 2025/6/3 14:08
 * @description: 开屏广告、插屏广告、激励视频广告
 * 通过广播传递事件
 */
abstract class AbsBiddingBuyOutGemini : ILog {

    companion object {
        const val EXT_SCENE_ID = "scene_id"         // 场景ID
        const val EXT_AD_MAX_ECPM_OBJECT = "ad_max_ecpm_object"         // 场景ID
        const val EXT_APP_LAYOUT_ID = "app_layout_id"     // App 信息布局
        const val EXT_DISPLAY_TIMESTAMP = "display_timestamp"     // App 信息布局

        const val EXT_ACTION_RESUME = "action_resume"
        const val EXT_ACTION_DISPLAY = "action_display"
        const val EXT_ACTION_CLICK = "action_click"
        const val EXT_ACTION_REWARDED = "action_rewarded"
        const val EXT_ACTION_DISPLAY_TIMESTAMP = "action_display_timestamp"
        const val EXT_ACTION_CLOSE = "action_close"
        const val EXT_ACTION_SHOW_ERROR = "show_error"

        const val ACTIVITY_STYLE_AUTO_FULLSCREEN = "auto_fullscreen"
        const val ACTIVITY_STYLE_ACTION_DIALOG = "action_dialog"
    }


    // =============================================================================================


    /**
     * 传递过来的数据
     */
    private var mListener: BiddingTAdditionalListener? = null // 广告监听回调 加载的场景提供
    private var mSceneId = ""
    private var mLayoutId: Int? = null // 开屏使用的自定义布局

    private var isReceiverRegistered = false // 一个生命周期仅注册一次

    /**
     * 创建一个BroadcastReceiver
     */
    private val myReceiver: BroadcastReceiver = object : BroadcastReceiver() {
        @Override
        override fun onReceive(context: Context?, intent: Intent) {
            // 子类自己处理对应的类型
            onSubReceive(context, intent)
        }
    }


    // =============================================================================================


    private fun getListener(): BiddingTAdditionalListener? = mListener
    override fun getSceneId(): String? = mSceneId

    /*** 广告加载回调*/
    fun setListener(listener: BiddingTAdditionalListener?): AbsBiddingBuyOutGemini {
        mListener = listener
        return this
    }

    fun setSceneId(sceneId: String): AbsBiddingBuyOutGemini {
        mSceneId = sceneId
        return this
    }

    fun setAppLayoutId(layoutId: Int?): AbsBiddingBuyOutGemini {
        mLayoutId = layoutId
        return this
    }


    // =============================================================================================


    /*** 注册事件广播*/
    fun registerReceiver(): AbsBiddingBuyOutGemini {
        if (isReceiverRegistered.not()) {
            isReceiverRegistered = true
            val filter = IntentFilter()
            filter.addAction(getResumeAction())             // 展示
            filter.addAction(getDisplayAction())            // 展示
            filter.addAction(getClickAction())              // 点击
            filter.addAction(getRewardedAction())           // 获取激励
            filter.addAction(getDisplayTimestampAction())   // 展示时间
            filter.addAction(getCloseAction())              // 关闭
            filter.addAction(getShowErrorAction())          // 关闭
            LocalBroadcastManager.getInstance(Utils.getApp()).registerReceiver(myReceiver, filter)
        }
        return this
    }

    /*** 资源回收*/
    fun destroy() {
        LocalBroadcastManager.getInstance(Utils.getApp()).unregisterReceiver(myReceiver)
        setListener(null)
        isReceiverRegistered = false
    }

    /*** 获取当前页面的样式*/
    fun getAdActivityStyle(): String {
        return SceneCommonConfig.getAdActivityStyle(getSceneId())
    }


    fun startActivity(
        context: Context?,
        sceneId: String,
        maxEcpmObject: BiddingIntermediateMaterialBean?,
    ) {
        runCatching {
            context?.let {
                //val intent = Intent(it, BiddingBuyOutSplashActivity::class.java)
                val intent = Intent(it, getActivityClass())
                intent.putExtra(EXT_SCENE_ID, sceneId) // 场景信息
                intent.putExtra(EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject) // 广告计划
                intent.putExtra(EXT_APP_LAYOUT_ID, mLayoutId) // 开屏广告使用到的

                intent.putExtra(EXT_ACTION_RESUME, getResumeAction())
                intent.putExtra(EXT_ACTION_DISPLAY, getDisplayAction())
                intent.putExtra(EXT_ACTION_CLICK, getClickAction())
                intent.putExtra(EXT_ACTION_DISPLAY_TIMESTAMP, getDisplayTimestampAction())
                intent.putExtra(EXT_ACTION_REWARDED, getRewardedAction())
                intent.putExtra(EXT_ACTION_CLOSE, getCloseAction())
                intent.putExtra(EXT_ACTION_SHOW_ERROR, getShowErrorAction())
                it.startActivity(intent)
            }
        }.getOrElse {
            getListener()?.onBiddingBuyOutShowError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BUY_OUT, "${it.message}"
                ), null
            )
        }
    }


    // =============================================================================================


    /**
     * 获取广告标识
     */
    abstract fun getAdTag(): String

    /**
     * 子类可以根据不同的配置跳转不同的页面
     */
    abstract fun getActivityClass(): Class<*>?


    // ====================================== 下面是广播处理 ==========================================


    /**
     * 意图配置
     */
    private fun getResumeAction(): String = "com.transsion.ad.bidding.${getAdTag()}.resume"
    private fun getDisplayAction(): String = "com.transsion.ad.bidding.${getAdTag()}.display"
    private fun getClickAction(): String = "com.transsion.ad.bidding.${getAdTag()}.click"
    private fun getRewardedAction(): String = "com.transsion.ad.bidding.${getAdTag()}.rewarded"
    private fun getDisplayTimestampAction(): String =
        "com.transsion.ad.bidding.${getAdTag()}.display_timestamp"

    private fun getCloseAction(): String = "com.transsion.ad.bidding.${getAdTag()}.close"
    private fun getShowErrorAction(): String = "com.transsion.ad.bidding.${getAdTag()}.show_error"

    /**
     * 子类处理对应的广告
     */
    private fun onSubReceive(context: Context?, intent: Intent) {
        // val data = intent.getStringExtra("key")
        // 接受传递过来的数据
        val maxEcpmObject = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            intent.getParcelableExtra(
                EXT_AD_MAX_ECPM_OBJECT, BiddingIntermediateMaterialBean::class.java
            )
        } else {
            intent.getParcelableExtra(EXT_AD_MAX_ECPM_OBJECT)
        }
        val displayTimestamp = intent.getLongExtra(EXT_DISPLAY_TIMESTAMP, 0)

//        // 执行相关的操作
//        if (displayTimestamp > 0) {
//            onLog(
//                Log.DEBUG,
//                "onSubReceive() --> 包断广告 --> 接收到本地广播 --> action = ${intent.action} --> displayTimestamp = $displayTimestamp"
//            )
//        } else {
//            onLog(
//                Log.DEBUG,
//                "onSubReceive() --> 包断广告 --> 接收到本地广播 --> action = ${intent.action}"
//            )
//        }

        when (intent.action) {
            getResumeAction() -> {
                getListener()?.onBiddingBuyOutResume(maxEcpmObject)
            }

            getDisplayAction() -> {
                getListener()?.onBiddingBuyOutDisplay(maxEcpmObject)
            }

            getClickAction() -> {
                getListener()?.onBiddingBuyOutClick(maxEcpmObject)
            }

            getRewardedAction() -> {
                getListener()?.onBiddingBuyOutRewarded(maxEcpmObject)
            }

            getDisplayTimestampAction() -> {
                getListener()?.onBiddingBuyOutDisplayTimestamp(maxEcpmObject, displayTimestamp)
            }

            getCloseAction() -> {
                getListener()?.onBiddingBuyOutClose(maxEcpmObject)
            }

            getShowErrorAction() -> {
                getListener()?.onBiddingBuyOutShowError(null, maxEcpmObject)
            }
        }
    }
}