package com.transsion.ad.ps

import android.text.TextUtils
import android.util.Log
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.Utils
import com.transsion.ad.db.pslink.PsLinkAdPlan
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.AdMaterialList
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdImage
import com.transsion.ad.ps.model.PSAdTypeEnum
import com.transsion.ad.ps.model.PsLinkAdInfo
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.util.AppUtil
import com.transsion.ad.util.TimeUtil

/**
 * @author shmizhangxinbing
 * @date : 2025/7/11 14:09
 * @description: PS  Offer 提供
 */
object PsOfferProvider {

    private fun getClassTag(): String = javaClass.simpleName


    // ================================ 获取可用的 PS Offer ==========================================


    /**
     * 获取可用的 PS Offer
     */
    suspend fun getPsAdPlans(adPlans: AdPlans?): Boolean {
        // 获取可用的PS Offer
        val availableAdPlan = getAvailablePsOffer(adPlans)
            ?: //            AdLogger.logPS(
//                "${getClassTag()} --> getPsAdPlans() --> 没有获取到可用Offer", level = Log.WARN
//            )
            return false

        // 赋值 -- 无论PS是否有数据都得赋值
        val materialList = AdMaterialList()
        adPlans?.adMaterialList = listOf(materialList) // 素材信息
        // 获取到数据了
        assignment(adPlans, materialList, availableAdPlan)
        return true
    }


    // =============================================================================================


    /**
     * 获取一个可用的 PS Plan
     */
    private suspend fun getAvailablePsOffer(adPlans: AdPlans?): PsLinkAdPlan? {
        // PS 广告位ID,设置默认值
        val extAdSlot = adPlans?.extAdSlot

        // 获取数据库中所有记录的数量
        var totalRecords = PsDbManager.getCountByExtAdSlot(extAdSlot.toString())
        // 如果没有记录，直接返回 null
        if (totalRecords == 0) {
            PsAdPlanRetryManager.retry(adPlans) // 没有数据，重发重新请求
            return null
        }

        // 这一步处理是为了防止代码异常导致的死循环
        val maxAttempts = 100 // 设定一个合理的最大尝试次数
        var processedCount = 0 // 处理计数器

        // 循环依次获取数据
        while (processedCount < totalRecords) {
            // 获取 updateTimestamp 最小的对象
            // 如果没有对象（不应该出现这种情况，因为我们已经检查过 totalRecords），退出
            val dbPsLinkAdPlan = PsDbManager.getPsLinkAdBySlotWithMinTimestamp(extAdSlot.toString())
                ?: return null // 没有获取到直接返回

            // 数据转换
            val psRecommendInfo =
                GsonUtils.fromJson(dbPsLinkAdPlan.psInfoJson, RecommendInfo::class.java)
                    ?: return null

            // 测试用日志 暂不需要输出
            // AdLogger.logPS("${getClassTag()} --> getAvailablePsOffer() --> 获取 updateTimestamp 最小的对象 --> id = ${dbPsLinkAdPlan.id}")

            // 无论是否符合展示条件，更新时间戳
            updateTimestamp(dbPsLinkAdPlan)

            // 时间判断
            if (isTimesAvailable(dbPsLinkAdPlan)) {
                // 如果当前Offer是拉活单子，那就需要判断是否已经安装，否则展示下一个Offer
                when (psRecommendInfo.adType) {
                    PSAdTypeEnum.PS_AD_TYPE_USER_ACQUISITION_0.value -> { // 拉新
                        return dbPsLinkAdPlan // 返回可用Offer
                    }

                    PSAdTypeEnum.PS_AD_TYPE_USER_RETENTION_1.value -> { // 拉活
                        // 检查当前应用是否已安装
                        val isInstalled = AppUtil.isAppInstalled(
                            context = Utils.getApp(), packageName = psRecommendInfo.packageName
                        )
                        if (isInstalled) {
                            return dbPsLinkAdPlan
                        }
                    }

                    else -> { // 未知类型，不处理
                        return null
                    }
                }
            }

            // 增加已处理计数
            processedCount++
            // 这一步是为了处理在查询的过程中有新增数据
            totalRecords = PsDbManager.getCountByExtAdSlot(extAdSlot.toString())

            // 异常兜底
            if (processedCount > maxAttempts) {
                AdLogger.logPS(
                    "${getClassTag()} --> getAvailableAdPlan() --> 赶紧找开发，出现死循环了、赶紧找开发，出现死循环了、赶紧找开发，出现死循环了",
                    level = Log.ERROR
                )
                return null
            }
        }

        // 所有记录都判断过了，无可用对象，返回 null
        return null
    }

    /**
     * 将PS信息赋值到广告计划
     */
    private fun assignment(
        adPlans: AdPlans?, materialList: AdMaterialList, psAdPlan: PsLinkAdPlan?
    ) {
        if (psAdPlan != null) {
            val psAdInfo = GsonUtils.fromJson(psAdPlan.psLinkAdInfoStr, PsLinkAdInfo::class.java)
            val psRecommendInfo = GsonUtils.fromJson(psAdPlan.psInfoJson, RecommendInfo::class.java)

            materialList.psLinkAdPlan = psAdPlan // PS Offer 数据

            // 素材信息
            materialList.downloadMaterialSuccess = true // PS计划 默认有资源的
            materialList.type = AdMaterialList.NON_AD_TYPE_TEXT
            materialList.title = psAdInfo?.title
            materialList.desc = psAdInfo?.desc
            materialList.psRecommendInfo = psRecommendInfo
            //todo 宽高信息
            materialList.image = MbAdImage(url = psAdInfo?.url, path = psAdInfo.path)

            // 按钮文案
            materialList.buttonText = psAdInfo?.buttonText

            // 广告主头像
            adPlans?.apply {
                advertiserName = psAdInfo?.advertiserName
                advertiserAvatar = psAdInfo?.advertiserAvatar
                advertiserAvatarPath = psAdInfo?.advertiserAvatarPath
            }

            // 封面图
            // 如果是PS广告计划，产品确认使用广告主头像
            adPlans?.extImage = MbAdImage(
                url = adPlans?.advertiserAvatar, path = adPlans?.advertiserAvatarPath
            )
        }
    }


    // =============================================================================================


    /**
     * 更新时间戳
     */
    private suspend fun updateTimestamp(dbPsLinkAdPlan: PsLinkAdPlan?) {
        // 日期判断 如果不是同一天需要重置数据
        if (TextUtils.equals(dbPsLinkAdPlan?.showDate, TimeUtil.getCurrentDate()).not()) {
            dbPsLinkAdPlan?.showedTimes = 0
            dbPsLinkAdPlan?.clickedTimes = 0
        }

        // 更新当前对象的时间戳
        dbPsLinkAdPlan?.updateTimestamp = System.currentTimeMillis()
        PsDbManager.updatePsLinkAdPlan(dbPsLinkAdPlan)
    }

    /**
     * 判断是否满足展示次数和时间段条件
     */
    private fun isTimesAvailable(dbPsLinkAdPlan: PsLinkAdPlan): Boolean {
        // 1. 展示数量判断 展示数量上限, 0不限制（默认）
        if (dbPsLinkAdPlan.showMax != 0 && (dbPsLinkAdPlan.showedTimes >= (dbPsLinkAdPlan.showMax
                ?: 0))
        ) {
            AdLogger.logPS(
                msg = "${getClassTag()} --> isTimesAvailable() --> 不符合展示条件 --> id = ${dbPsLinkAdPlan.id} --> showedTimes = ${dbPsLinkAdPlan.showedTimes} --> showMax = ${dbPsLinkAdPlan.showMax}",
                level = Log.WARN
            )
            return false
        }

        // 2. 点击数量判断 点击数量上限, 0不限制（默认）
        if (dbPsLinkAdPlan.clickMax != 0 && (dbPsLinkAdPlan.clickedTimes >= (dbPsLinkAdPlan.clickMax
                ?: 0))
        ) {
            AdLogger.logPS(
                msg = "${getClassTag()} --> isTimesAvailable() --> 不符合展示条件 --> id = ${dbPsLinkAdPlan.id} --> clickedTimes = ${dbPsLinkAdPlan.clickedTimes} --> clickMax = ${dbPsLinkAdPlan.clickMax}",
                level = Log.WARN
            )
            return false
        }

        // 3. 时间段判断
        if (isInShowHours(dbPsLinkAdPlan.showHours).not()) {
            // 获取当前小时（0~23）
            val currentHour = java.util.Calendar.getInstance().get(java.util.Calendar.HOUR_OF_DAY)
            AdLogger.logPS(
                msg = "${getClassTag()} --> isTimesAvailable() --> 不符合展示条件 --> id = ${dbPsLinkAdPlan.id} --> showHours = ${dbPsLinkAdPlan.showHours} --> currentHour = $currentHour",
                level = Log.WARN
            )
            return false
        }

        // 4. 默认符合展示条件
        return true
    }

    /**
     * 判断当前小时是否在 showHours 内（showHours为字符串类型）
     * @param showHours 展示时段字符串，比如 "0,1,2,3,10"
     * @return 是否在展示时段内
     */
    private fun isInShowHours(showHours: String?): Boolean {
        return kotlin.runCatching {
            if (showHours.isNullOrBlank()) {
                // 历史数据为 null 或空时 默认展示
                return true
            }
            // 拆分字符串为小时集合
            val hourSet = showHours.split(",").mapNotNull { it.trim().toIntOrNull() }.toSet()
            // 获取当前小时（0~23）
            val currentHour = java.util.Calendar.getInstance().get(java.util.Calendar.HOUR_OF_DAY)
            return hourSet.contains(currentHour)
        }.getOrElse {
            false
        }
    }

}