package com.transsion.ad

import android.util.Log
import androidx.lifecycle.ProcessLifecycleOwner
import com.blankj.utilcode.util.AppUtils
import com.blankj.utilcode.util.Utils
import com.tn.lib.net.env.Host
import com.transsion.ad.config.BusinessBridge
import com.transsion.ad.hi.HiSavanaAdManager
import com.transsion.ad.monopoly.MonopolyAdManager
import com.transsion.ad.strategy.AppManager
import com.transsion.ad.config.HotSceneManager
import com.transsion.ad.log.AdLogger
import com.transsion.ad.strategy.AdObserveNetworkState
import com.transsion.ad.strategy.KtxAppLifeObserver
import com.transsion.ad.strategy.NationalInformationManager
import com.transsion.ad.util.FileUtil
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.withContext

/**
 * @author: zhangxinbing
 * @date : 2025/4/17 20:42
 * @description: 广告初始化入口
 */
object MBAd {

    private fun getClassTag(): String = javaClass.simpleName

    data class AdInitParams(
        val hisavanaAppId: String,                          // HiSavana 应用ID
        val adTestDevice: Boolean,                          // 程序化-是否是测试设备
        val isDebug: Boolean,                               // 程序换-是否是debug
        val appEnum: AppManager.AppEnum,                    // 媒体类型
        val hotSceneId: String,                             // 热启动布局
        val adConfigUrl: String,                            // 广告配置接口
        val mediaDisposeAppForeground: Boolean = false,     // 媒体自定义处理热启动广告
        val businessBridge: BusinessBridge?,                // 业务交互
    )

    class Builder {
        private var hisavanaAppId: String = ""
        private var adTestDevice: Boolean = false
        private var isDebug: Boolean = false
        private var appEnum: AppManager.AppEnum = AppManager.AppEnum.APP_MOVIE_BOX
        private var hotSceneId: String = ""
        private var adConfigUrl: String = ""
        private var mediaDisposeAppForeground: Boolean = false
        private var businessBridge: BusinessBridge? = null

        fun setHiSavanaAppId(hisavanaAppId: String) = apply { this.hisavanaAppId = hisavanaAppId }
        fun setAdTestDevice(adTestDevice: Boolean) = apply { this.adTestDevice = adTestDevice }
        fun setIsDebug(isDebug: Boolean) = apply { this.isDebug = isDebug }
        fun setAppEnum(appEnum: AppManager.AppEnum) = apply { this.appEnum = appEnum }
        fun setHotSceneId(hotSceneId: String) = apply { this.hotSceneId = hotSceneId }
        fun setAdConfigUrl(adConfigUrl: String) = apply { this.adConfigUrl = adConfigUrl }

        /**
         * 设置了true，一定要处理热启动开屏广告的触发逻辑 否则广告将无法展示
         */
        fun setMediaDisposeAppForeground(mediaDisposeAppForeground: Boolean) =
            apply { this.mediaDisposeAppForeground = mediaDisposeAppForeground }

        /**
         * 业务交互
         */
        fun setBusinessBridge(businessBridge: BusinessBridge?) = apply {
            this.businessBridge = businessBridge
        }

        fun build(): AdInitParams {
            // 参数验证
            if (hisavanaAppId.isEmpty()) {
                throw IllegalArgumentException("HiSavana App ID must not be empty")
            }
            if (adConfigUrl.isEmpty()) {
                throw IllegalArgumentException("AdConfigUrl must not be empty")
            }
            return AdInitParams(
                hisavanaAppId = hisavanaAppId,
                adTestDevice = adTestDevice,
                isDebug = isDebug,
                appEnum = appEnum,
                hotSceneId = hotSceneId,
                adConfigUrl = adConfigUrl,
                mediaDisposeAppForeground = mediaDisposeAppForeground,
                businessBridge = businessBridge
            )
        }
    }


    // =============================================================================================


    private var adInitParams: AdInitParams? = null

    fun getAdInitParams(): AdInitParams? = adInitParams

    /**
     * 初始化
     */
    suspend fun initAd(params: AdInitParams) {

        if (adInitParams != null) {
            AdLogger.logAdInit(
                level = Log.ERROR, msg = "${getClassTag()} --> initAd() --> 广告SDK，已经初始化"
            )
            return
        }
        adInitParams = params

        withContext(Dispatchers.IO) {
            kotlin.runCatching {
                AdLogger.logAdInit("${getClassTag()} --> initAd() --> 当前包名 = ${AppUtils.getAppPackageName()} --> monopoly广告SDK 开始初始化 --> hisavanaAppId = ${params.hisavanaAppId} --> adTestDevice = ${params.adTestDevice} --> isDebug = ${params.isDebug} --> appEnum = ${params.appEnum.value} --> host = ${Host.getHost()}")

                // 初始化配置相关
                AppManager.setApp(params.appEnum) // 标记当前是哪个媒体接入的
                HotSceneManager.setHotSceneId(params.hotSceneId) // 热启动插屏广告位ID
                NationalInformationManager.initLocalMcc(Utils.getApp()) // 国家MCC初始化
                FileUtil.createDirectoryIfNotExists(MbAdContents.NON_AD_VIDEO_DOWNLOAD_FILE_PATH) // 创建缓存文件夹 包断
                FileUtil.createDirectoryIfNotExists(MbAdContents.NON_AD_PS_DOWNLOAD_FILE_PATH) // 创建缓存文件夹 PS

                // App 状态回调 这个需要在主线程中执行
                // 需要判断媒体是否自定义处理热启动开屏广告逻辑
                if (params.mediaDisposeAppForeground) {
                    AdLogger.logAdInit(
                        "${getClassTag()} --> initAd() --> 媒体自行处理宿主App前后台切换逻辑",
                        level = Log.WARN
                    )
                } else {
                    withContext(Dispatchers.Main) {
                        ProcessLifecycleOwner.get().lifecycle.addObserver(KtxAppLifeObserver)
                        AdLogger.logAdInit("${getClassTag()} --> initAd() --> KtxAppLifeObserver 注册完成")
                    }
                }

                // 网络变化
                AdObserveNetworkState().registerNetworkStatusChangedListener(params.adConfigUrl)

                val sdkHi = async {
                    // 初始化HiSavana广告SDK
                    HiSavanaAdManager.initHiAdSdk(
                        params.hisavanaAppId, params.adTestDevice, params.isDebug
                    )
                    "HiSavana 广告SDK初始化完成"
                }

                val sdkMb = async {
                    // 初始化非标广告相关
                    MonopolyAdManager.init(params.adConfigUrl)
                    "包断广告初始化完成"
                }

                AdLogger.logAdInit(
                    " ================= ${getClassTag()} --> initAd() --> ${sdkMb.await()} -- ${sdkHi.await()} ================= ",
                    level = Log.ERROR
                )
            }.getOrElse {
                it.printStackTrace()
                AdLogger.logAdInit(
                    "${getClassTag()} --> initAd() --> 初始化异常 --> it = $it", level = Log.ERROR
                )
            }
        }
    }

    suspend fun initNonAd(adConfigUrl: String) {
        withContext(Dispatchers.IO) {
            // 初始化非标广告相关
            MonopolyAdManager.init(adConfigUrl)
        }
    }
}