package com.transsion.ad.ps.attribution

import com.blankj.utilcode.util.Utils
import com.transsion.ad.db.MbAdDatabase
import com.transsion.ad.db.pslink.AttributionPoint
import com.transsion.ad.db.pslink.AttributionPoint.AttributionType
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * @author shmizhangxinbing
 * @date : 2025/7/14 20:58
 * @description:
 */
open class BaseAttributionProvider {

    companion object {

        /**
         * 用于 上报失败之后 重试的时间间隔
         */
        const val SPACING_INTERVAL = 5000L
    }

    private val pointDao by lazy {
        MbAdDatabase.getInstance(Utils.getApp()).attributionPointDao()
    }

    protected fun getClassTag(): String = javaClass.simpleName


    /**
     * 查询是否存在指定 psId 和 type 的记录
     */
    protected suspend fun isAttributionPointExists(
        psId: String?, type: AttributionType, reportUrl: String?
    ): Boolean {
        return withContext(Dispatchers.IO) {
            pointDao.isAttributionPointExists(psId = psId, type = type, reportUrl = reportUrl)
        }
    }

    /**
     * 插入数据
     */
    protected suspend fun insert(reportUrl: String, type: AttributionType, psId: String) {
        withContext(Dispatchers.IO) {
            pointDao.insertAttributionPoint(
                AttributionPoint(
                    reportUrl = reportUrl,
                    type = type,
                    updateTimestamp = System.currentTimeMillis() - (SPACING_INTERVAL * 2),
                    psId = psId
                )
            )
        }
    }

    /**
     * 删除数据
     */
    protected suspend fun delete(attributionPoint: AttributionPoint) {
        withContext(Dispatchers.IO) {
            pointDao.deleteAttributionPoint(attributionPoint)
        }
    }

    /**
     * 更新数据
     */
    protected suspend fun update(attributionPoint: AttributionPoint) {
        withContext(Dispatchers.IO) {
            attributionPoint.updateTimestamp = System.currentTimeMillis()
            pointDao.updateAttributionPoint(attributionPoint)
        }
    }

    /**
     * 获取一个最优数据
     */
    protected suspend fun getAttributionPoint(): AttributionPoint? {
        return MbAdDatabase.getInstance(Utils.getApp()).attributionPointDao()
            .getTopAttributionPoints(
                currentTimestamp = System.currentTimeMillis(), timeThreshold = SPACING_INTERVAL
            )
    }

}