package com.transsion.ad.strategy

import android.content.Intent
import android.net.Uri
import android.text.TextUtils
import android.util.Log
import androidx.annotation.MainThread
import com.blankj.utilcode.util.ActivityUtils
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.ToastUtils
import com.blankj.utilcode.util.Utils
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.ad.MBAd
import com.transsion.ad.R
import com.transsion.ad.db.pslink.AttributionPoint.AttributionType
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.monopoly.plan.AdPlanUtil
import com.transsion.ad.ps.PsLinkUtils
import com.transsion.ad.ps.attribution.AttributionProduceManager
import com.transsion.ad.ps.model.PSAdTypeEnum
import com.transsion.ad.ps.model.RecommendInfo

/**
 * @author: zhangxinbing
 * @date : 2025/2/18 14:56
 * @description: 广告点击统一处理
 */
object AdClickManager {

    interface AdClickListener {
        fun onAdClick(adPlan: AdPlans?)
    }

    private val clickListenerList = mutableListOf<AdClickListener>()

    @MainThread
    fun addListener(listener: AdClickListener?) {
        if (listener == null) {
            return
        }
        if (clickListenerList.contains(listener).not()) {
            clickListenerList.add(listener)
        }
    }

    @MainThread
    fun removeListener(listener: AdClickListener?) {
        clickListenerList.remove(listener)
    }

    fun onClick(adPlan: AdPlans?) {
        clickListenerList.forEach {
            it.onAdClick(adPlan)
        }
    }

    private fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    /**
     * 竞价广告点击事件处理
     */
    fun onBiddingAdClick(
        adPlan: AdPlans? = null,
        overridePendingTransition: AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum? = null,
        logTag: String,
        sceneId: String
    ) {
        if (adPlan == null) {
            AdLogger.logSdk(
                tag = logTag,
                msg = "${getClassTag()} --> onBiddingAdClick() --> sceneId = $sceneId --> adPlan == null",
                level = Log.ERROR
            )
            return
        }

        // 获取素材对象
        val adMaterial = AdPlanUtil.getAdMaterial(adPlan)
        if (adMaterial == null) {
            AdLogger.logSdk(
                tag = logTag,
                msg = "${getClassTag()} --> onBiddingAdClick() --> sceneId = $sceneId --> adMaterial == null",
                level = Log.ERROR
            )
            return
        }

        // 广告点击回调 全局监听
        onClick(adPlan)

        // 输出点击事件信息
        //val gpLink = "https://play.google.com/store/apps/details?id=com.game.rescueagent&hl=zh"
        val deeplink = adMaterial.deeplink
        val h5Url = adMaterial.h5Link
        val adSource = adPlan.adSource
        AdLogger.logSdk(
            tag = logTag,
            msg = "${getClassTag()} --> onBiddingAdClick() --> sceneId = $sceneId --> name = ${adPlan.name} --> adSource = $adSource --> deeplink = $deeplink --> h5Url = $h5Url"
        )

        // AdSourceMBPS
        if (isMbPsAdPlan(adPlan)) {
            return
        }

        // AdSourcePS
        if (isPsAdPlan(adPlan)) {
            return
        }

        // 其他类型跳转
        // deeplink 打开失败使用落地页
        if (jumpDeeplink(deeplink).not()) {
            jumpH5(h5Url, overridePendingTransition)
        }
    }

    /**
     * deeplink跳转 --> 跳转失败看看是否配置了H5 --> 跳转H5
     * val deeplink = "http://play.google.com/store/apps/details?id=com.google.android.apps.maps"
     */
    fun jumpDeeplink(deeplink: String?): Boolean {

        if (TextUtils.isEmpty(deeplink)) {
            return false
        }

        return kotlin.runCatching {
            val contentUri = Uri.parse(deeplink)
            val intent = Intent()
            intent.data = contentUri
            intent.action = Intent.ACTION_VIEW
            intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
            Utils.getApp().startActivity(intent)
            true
        }.getOrElse {
            AdLogger.logCommon(msg = "${getClassTag()} --> jumpDeeplink() --> e = $it")
            false
        }
    }

    /**
     * AdSourceMBPS 处理
     *
     * 判断当前计划是否是PS_MB类型的包断广告
     */
    private fun isMbPsAdPlan(adPlan: AdPlans): Boolean {
        // 获取素材对象
        val adMaterial = AdPlanUtil.getAdMaterial(adPlan)

        // AdSourceMBPS 包断广告内置PS类型
        if (AdPlanSourceManager.isMbPsAdPlan(adPlan)) {
            val psLink = adMaterial?.psLink ?: ""
            val recommendInfo = GsonUtils.fromJson(psLink, RecommendInfo::class.java)
            PsLinkUtils.startPsActivity(
                itemInfo = recommendInfo, true, scene = adPlan.extAdSlot.toString()
            )
            return true
        } else {
            return false
        }
    }

    /**
     * PS 虚拟计划点击事件处理
     */
    private fun isPsAdPlan(adPlan: AdPlans): Boolean {
        // 获取素材对象
        val adMaterial = AdPlanUtil.getAdMaterial(adPlan)
        var deeplink = adMaterial?.deeplink

        // AdSourcePS
        if (AdPlanSourceManager.isPsAdPlan(adPlan)) {
            val psRecommendInfo = adMaterial?.psRecommendInfo
            val gpLink = psRecommendInfo?.gpLink
            deeplink = psRecommendInfo?.deepLink
            if (TextUtils.isEmpty(deeplink)) {
                deeplink = gpLink // deeplink 不存在 使用 gpLink
            }

            // 根据单子类型判断
            val adType = psRecommendInfo?.adType
            when (adType) {
                PSAdTypeEnum.PS_AD_TYPE_USER_RETENTION_1.value -> { // 拉活
                    // 打开失败需要兜底方案
                    if (jumpDeeplink(deeplink).not()) {
                        if (PsLinkUtils.isSupportPsLink()) {
                            PsLinkUtils.startPsActivity(
                                itemInfo = psRecommendInfo,
                                true,
                                scene = adPlan.extAdSlot.toString()
                            )
                        } else {
                            jumpDeeplink(gpLink)
                            AttributionProduceManager.onBiddingBuyOutClick(
                                adPlan, AttributionType.GP_CLICK
                            )
                        }
                    } else {
                        val type = if (TextUtils.isEmpty(deeplink)) {
                            AttributionType.GP_CLICK
                        } else {
                            AttributionType.CLICK
                        }
                        AttributionProduceManager.onBiddingBuyOutClick(adPlan, type = type)
                    }
                }

                PSAdTypeEnum.PS_AD_TYPE_USER_ACQUISITION_0.value -> { // 拉新
                    if (PsLinkUtils.isSupportPsLink()) {
                        PsLinkUtils.startPsActivity(
                            itemInfo = psRecommendInfo, true, scene = adPlan.extAdSlot.toString()
                        )
                    } else {
                        // 归因埋点
                        jumpDeeplink(gpLink) // 跳转
                        AttributionProduceManager.onBiddingBuyOutClick(
                            adPlan, AttributionType.GP_CLICK
                        ) // 点击归因
                    }
                }
            }
            return true
        } else {
            return false
        }
    }

    /**
     * 区分是否站内打开 点击之后暂停播放
     */
    private fun jumpH5(
        h5Url: String?,
        overridePendingTransition: AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum? = null,
        logTag: String? = ""
    ) {
        // 无网络判断
        if (NetworkUtil.hasCapabilityAsync().not() || NetworkUtil.isNetworkConnected(Utils.getApp())
                .not()
        ) {
            val text = ActivityUtils.getTopActivity()?.getString(R.string.ad_no_network)
            ToastUtils.showShort(text)
            return
        }

        // 是否需要通过系统浏览器打开落地页
        if (AdUrlParameterManager.isOpenedInSystemBrowser(h5Url)) {
            val uri = Uri.parse(h5Url)
            val intent = Intent(Intent.ACTION_VIEW, uri)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            Utils.getApp().startActivity(intent)
            return
        }

        // 宏替换处理
        val newUrl = AdUrlParameterManager.replaceParameter(url = h5Url ?: "", logTag = logTag)

        // TODO 2025-08-29
        // 跳转落地页
//        AdWebActivity.startActivity(
//            url = newUrl, overridePendingTransition = overridePendingTransition
//        )

        // 交给媒体自行处理
        MBAd.getAdInitParams()?.businessBridge?.startWebActivity(
            h5Url = newUrl, overridePendingTransition = overridePendingTransition
        )
    }


    // =============================================================================================


    /**
     * 广告点击跳转处理
     */
    @Deprecated("onBiddingAdClick")
    fun adClick(deeplink: String?, h5Url: String?, adPlan: AdPlans? = null) {
        AdLogger.logCommon(msg = "AdClickManager --> adClick() --> deeplink = $deeplink --> h5Url = $h5Url")

        // PS_MB类型的包断广告
        if (startPsActivity(adPlan)) {
            return
        }

        val gpLink = adPlan?.adMaterialList?.get(0)?.psRecommendInfo?.gpLink
//        val gpLink = "https://play.google.com/store/apps/details?id=com.game.rescueagent&hl=zh"
        if (AdPlanSourceManager.isPsAdPlan(adPlan)) {
            //不支持ps半屏 + 有gp链接 直接跳转gp
            if (!PsLinkUtils.isSupportPsLink() && gpLink?.contains("play.google.com") == true) {
                jumpDeeplink(gpLink)
            } else {
                PsLinkUtils.startPsActivity(
                    itemInfo = adPlan?.adMaterialList?.get(0)?.psRecommendInfo,
                    true,
                    scene = adPlan?.extAdSlot.toString()
                )
            }
        } else {
            if (jumpDeeplink(deeplink).not()) {
                jumpH5(h5Url)
            }
        }

    }

    /**
     * 打开PS半屏
     */
    @Deprecated("onBiddingAdClick")
    private fun startPsActivity(adPlan: AdPlans?): Boolean {
        return kotlin.runCatching {
            // 1. 先判断当前计划是否是PS_MB类型的包断广告
            if (AdPlanSourceManager.isMbPsAdPlan(adPlan).not()) {
                return false
            }

            // 2. 在判断PSLink 字段是否是空的
            val adMaterialList = adPlan?.adMaterialList
            if (adMaterialList.isNullOrEmpty()) {
                return false
            }
            val adMaterial = adMaterialList[0]
            val psLink = adMaterial.psLink
            if (TextUtils.isEmpty(psLink)) {
                return false
            }

            // 3. 解析 psLink JSON 字符串
            val recommendInfo = GsonUtils.fromJson(psLink, RecommendInfo::class.java)
            //val recommendInfo = GsonUtils.fromJson(gson, RecommendInfo::class.java)

            // 4. 打开半屏
            PsLinkUtils.startPsActivity(
                itemInfo = recommendInfo, true, scene = adPlan.extAdSlot.toString()
            )
            true
        }.getOrElse {
            false
        }
    }

    private val gson =
        "{\n" + "            \"adSource\": \"ps_candidate\",\n" + "            \"buttonText\": \"Go\",\n" + "            \"categoryCode\": \"1896ff1657b2408daeb264ba6d370bf2\",\n" + "            \"categoryName\": \"Finance\",\n" + "            \"detail\": {\n" + "                \"downloadCount\": \"27073080\",\n" + "                \"iconUrl\": \"https://cdn2.palmplaystore.com/static/59/42d6df6edbe84bdd926fab309aa48261-V02222_512~512.webp?blurhash=LVRpFBkB%5E%23j%5BbHjaoLfQ%7ESa%7C9Jjt\",\n" + "                \"img0\": \"https://cdn2.palmplaystore.com/static/889/0a4a8ed888f04551bcef55e4d31187ff-V02222_288~512.webp?blurhash=LWC%5D6r0f%2BYRO*JkWXos.5q%3FGNGXn\",\n" + "                \"img1\": \"https://cdn2.palmplaystore.com/static/881/b88eab70ee8e445a836225f7fd7c7180-V02222_288~512.webp?blurhash=LqHM4j%7EpXnRj4oI%5Bxtx%5DNeM%7BShae\",\n" + "                \"img2\": \"https://cdn2.palmplaystore.com/static/791/41db2f5cea9549f584c0bae3a7b1b1c9-V02222_288~512.webp?blurhash=LQFj%24.WU4mS7*woHMcRll.tQR%3BR-\",\n" + "                \"img3\": \"https://cdn2.palmplaystore.com/static/181/e99237f24ede40f4a9fd56a08a1c30a5-V02222_288~512.webp?blurhash=LE9ums%7D5Uu9b4p9IPBxZm%25kryDvd\",\n" + "                \"img4\": \"https://cdn2.palmplaystore.com/static/574/2bddbb51c7874e7a906ceb55a75b9124-V02222_288~512.webp?blurhash=LVGTa6n301RP3ERP_3n%23PXo%7DxBW%3D\",\n" + "                \"isOffer\": 1,\n" + "                \"itemID\": \"s_b7801ad36c0a0e4c7fc1f1e72bb04b2d\",\n" + "                \"lan\": \"EN\",\n" + "                \"name\": \"Moniepoint Personal Banking\",\n" + "                \"packageName\": \"com.moniepoint.personal\",\n" + "                \"safeTagList\": [\n" + "                    {\n" + "                        \"icon\": \"https://cdn2.palmplaystore.com/static/673/fa007ff1d5c14939b7d52a5d87247b5f.webp\",\n" + "                        \"id\": 100033,\n" + "                        \"isDeleted\": 0,\n" + "                        \"name\": \"Free\",\n" + "                        \"priority\": 90,\n" + "                        \"type\": 4\n" + "                    }\n" + "                ],\n" + "                \"safetyStyle\": 1,\n" + "                \"score\": \"4.0\",\n" + "                \"screenshotMode\": [\n" + "                    0,\n" + "                    0,\n" + "                    0,\n" + "                    0,\n" + "                    0\n" + "                ],\n" + "                \"simpleDescription\": \"Send money & airtime, request a debit card & manage your savings from one place\",\n" + "                \"sourceSize\": \"71526348\",\n" + "                \"star\": \"4\",\n" + "                \"tagList\": [],\n" + "                \"versionCode\": \"2068\"\n" + "            },\n" + "            \"downloadCount\": 27061226,\n" + "            \"exists\": null,\n" + "            \"gpLink\": \"https://play.google.com/store/apps/details?id=com.moniepoint.personal&hl=en\",\n" + "            \"iconUrl\": \"https://cdn2.palmplaystore.com/static/59/42d6df6edbe84bdd926fab309aa48261-V02222_512~512.webp?blurhash=LVRpFBkB%5E%23j%5BbHjaoLfQ%7ESa%7C9Jjt\",\n" + "            \"id\": 580,\n" + "            \"isOffer\": 1,\n" + "            \"itemID\": \"s_b7801ad36c0a0e4c7fc1f1e72bb04b2d\",\n" + "            \"name\": null,\n" + "            \"offerDesc\": \"\",\n" + "            \"packageName\": \"com.moniepoint.personal\",\n" + "            \"planName\": \"Moniepoint上线测试20250313-MB信息流竖屏\",\n" + "            \"showContent\": \"https://cdn2.palmplaystore.com/static/403/f2d43eb546cb4cd9ad61fe9adec150d4.webp?blurhash=LXE%7B%5EdOtxrR401rpIVtl%7EUtRkCaK\",\n" + "            \"showType\": 9,\n" + "            \"simpleDescription\": \"Send money & airtime, request a debit card & manage your savings from one place\",\n" + "            \"size\": 71526348,\n" + "            \"star\": \"4.0\",\n" + "            \"verifyGoogle\": \"F\",\n" + "            \"versionCode\": 2068,\n" + "            \"versionName\": \"1.23.0\"\n" + "        }"

}