package com.transsion.ad.strategy

import android.net.Uri
import android.text.TextUtils
import com.cloud.sdk.commonutil.util.DeviceUtil
import com.transsion.ad.log.AdLogger
import com.transsion.ad.scene.SceneGlobalConfig

/**
 * @author: zhangxinbing
 * @date : 2025/2/17 20:05
 * @description: 广告落地页字段替换
 */
object AdUrlParameterManager {

    data class UrlParameterReplaceEntity(val key: String, val value: String, val type: String)

    enum class UrlParameterReplaceEnum(val value: String) {
        GAID("GAID"), CUSTOM_USER_ID("CUSTOM_USER_ID")
    }

    /**
     * 字段替换
     */
    fun replaceParameter(url: String, logTag: String? = ""): String {
        return kotlin.runCatching {

            if (TextUtils.isEmpty(url)) {
                return url
            }

            // 下发配置
            val replaceList = SceneGlobalConfig.getUrlParameterReplace()
            if (replaceList.isEmpty()) {
                return url
            }

            // 解析 URL
            val uri: Uri = Uri.parse(url)
            // 获取所有的参数键
            val queryParameterNames = uri.queryParameterNames

            // 替换字段
            var newUrl = url
            replaceList.forEach {
                // 根据条件替换
                if (queryParameterNames.contains(it.key) && uri.getQueryParameter(it.key) == it.value) {
                    if (it.type == UrlParameterReplaceEnum.GAID.value) {
                        newUrl = newUrl.replace(it.value, DeviceUtil.getGAId())
                    }
                    if (it.type == UrlParameterReplaceEnum.CUSTOM_USER_ID.value) {
                        newUrl = newUrl.replace(it.value, "custom_use_id_ashdjhv1234dhb")
                    }
                }
            }

            // 返回处理之后URL
            AdLogger.logSdk(
                tag = logTag ?: "",
                msg = "AdUrlParameterReplaceManager --> replaceParameter() --> url = $url --> replaceList = $replaceList --> newUrl = $newUrl"
            )
            return newUrl
        }.getOrElse {
            url
        }
    }


    /**
     * 判断配置的链接是否需要通过系统浏览器打开
     */
    fun isOpenedInSystemBrowser(url: String?): Boolean {
        return kotlin.runCatching {

            // 检查URL是否为空或空字符串
            if (url.isNullOrEmpty()) {
                return false
            }

            // 使用Uri解析URL
            val uri = Uri.parse(url)
            // 获取字段值并判断是否为"true"
            uri.getQueryParameter("mb_system_browser")?.equals("true", ignoreCase = true) == true
        }.getOrElse {
            false
        }
    }
}