package com.transsion.ad.strategy

import android.annotation.SuppressLint
import android.os.Handler
import android.os.Looper
import android.os.SystemClock
import android.text.TextUtils
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.blankj.utilcode.util.ActivityUtils
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.bidding.interstitial.BiddingInterstitialManager
import com.transsion.ad.config.FullSearchModel
import com.transsion.ad.config.HotSceneManager
import com.transsion.ad.log.AdLogger
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.scene.SceneStorage
import com.transsion.ad.util.ScreenUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * @author shmizhangxinbing
 * @date : 2025/8/4 17:03
 * @description:
 */
object HotSplashManager : BiddingTAdditionalListener() {

    /**
     * System.currentTimeMillis()
     * 含义：返回系统的“当前时间”，即自1970年1月1日00:00:00 UTC（标准时间）以来的毫秒数，也就是我们常说的“时间戳”。
     * 特点：
     * 受系统时间影响。如果用户手动修改了时间或者网络自动校准了时间，这个值会发生变化。
     * 适合用于记录真实的日历时间，比如打卡、日志时间等。
     * 不适合用来做时间间隔的统计（容易被用户修改系统时间影响）。
     *
     * SystemClock.elapsedRealtime()
     * 含义：返回系统“开机以来”经过的时间（包括休眠时间），单位为毫秒。
     * 特点：
     * 不受系统时间调整的影响，用户修改时间也不会变。
     * 适合用于统计程序运行时间、计时、性能监控等场景。
     * 不是实际的日历时间，只是一个相对的时间戳。
     */
    private var inBackgroundTime = 0L   // 在后台的累计时间
    private var inBackground = false    // 当前是否是后台
    private fun getClassTag(): String = javaClass.simpleName

    @SuppressLint("StaticFieldLeak")
    private var interstitialManager: BiddingInterstitialManager? = null
    private val mHandler = Handler(Looper.getMainLooper())


    // =============================================================================================


    /**
     * App 进入前台
     */
    fun onStart() {
        AdLogger.logSdk(
            tag = AdLogger.TAG_AD_INTERSTITIAL,
            msg = "${getClassTag()} --> onStop() --> App进入前台",
            writeToFile = false
        )

        inBackground = false
        // 加载并展示广告
        mHandler.removeCallbacksAndMessages(null)
        mHandler.postDelayed({
            loadHotSplashAd()
        }, 300)
    }

    /**
     * App 进入后台
     */
    fun onStop() {
        AdLogger.logSdk(
            tag = AdLogger.TAG_AD_INTERSTITIAL,
            msg = "${getClassTag()} --> onStop() --> App进入后台",
            writeToFile = false
        )

        inBackgroundTime = SystemClock.elapsedRealtime()
        inBackground = true
        mHandler.removeCallbacksAndMessages(null)

        // 包断默认直接可用，这里仅需要触发Hi程序化即可
        BiddingInterstitialManager.preLoadAd(HotSceneManager.getHotSceneId())
    }


    // ==================================== 广告回调 =================================================


    private fun resetAd() {
        interstitialManager?.destroy()
        interstitialManager = null
    }

    override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingLoad(maxEcpmObject)
        ActivityUtils.getTopActivity()?.let { topAct ->
            (topAct as? AppCompatActivity)?.let {
                it.lifecycleScope.launch {
                    withContext(Dispatchers.Main) {
                        val showMemberEntrance = SceneCommonConfig.getIsShowMemberEntrance(HotSceneManager.getHotSceneId())
                        interstitialManager?.startAdActivity(activity = topAct,maxEcpmObject, isShowMemberBtn = showMemberEntrance)
                    }
                }
            }
        }
    }

    override fun onBiddingWrapperAdClose(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingWrapperAdClose(maxEcpmObject)
        resetAd()
    }

    override fun onBiddingError(p0: TAdErrorCode?) {
        super.onBiddingError(p0)
        resetAd()
    }

    override fun onBiddingWrapperAdShowError(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingWrapperAdShowError(maxEcpmObject)
        resetAd()
    }


    // =============================================================================================


    /**
     * 加载并且展示热启动开屏广告
     */
    private fun loadHotSplashAd() {

        val msg = isIntercept()
        if (TextUtils.isEmpty(msg).not()) {
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_INTERSTITIAL,
                msg = "${getClassTag()} --> loadHotSplashAd() --> msg = $msg",
                level = Log.ERROR,
                writeToFile = false
            )
            return
        }

        if (interstitialManager != null) {
//            AdLogger.logSdk(
//                tag = AdLogger.TAG_AD_INTERSTITIAL,
//                msg = "${getClassTag()} --> loadHotSplashAd() --> 广告已存在等待回收之后重新创建",
//                writeToFile = false
//            )
            return
        }

        CoroutineScope(Dispatchers.IO).launch {
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_INTERSTITIAL,
                msg = "${getClassTag()} --> loadHotSplashAd() --> App 进入前台符合条件 --> 准备展示开屏广告",
                writeToFile = false
            )

            if (interstitialManager == null) {
                interstitialManager = BiddingInterstitialManager()
                interstitialManager?.setSceneId(HotSceneManager.getHotSceneId())
                    ?.setListener(this@HotSplashManager)
            }
            interstitialManager?.loadAd()
        }
    }


    // =============================================================================================


    /**
     * 检查展示条件是否满足
     */
    private fun isIntercept(): String {
        // 处理App在锁屏状态下被拉活
        val topAct = ActivityUtils.getTopActivity()
        topAct ?: return "topAct is null"
        if (topAct.isDestroyed) {
            return "topAct?.isDestroyed == true"
        }

        val isFakeLaunch =
            ScreenUtils.isScreenActive(topAct).not() || ScreenUtils.isScreenLocked(topAct)
        if (isFakeLaunch) {
            return "处理App在锁屏状态下被拉活"
        }

        // 全搜的时候热启动不展示热启动广告
        if (FullSearchModel.isSkipHotSplash()) {
            // 重置数据
            FullSearchModel.setSkipHotSplash(isSkip = false)
            return "全搜的时候热启动不展示热启动广告"
        }

        // 在后台累计时间判断
        if (inBackgroundTime <= 0) {
            return "inBackgroundTime <= 0"
        }

        // 获取远端配置信息
        val time = SystemClock.elapsedRealtime() - inBackgroundTime
        val sceneConfig = SceneStorage.getSceneConfig(HotSceneManager.getHotSceneId())
        val hotSplashInterval =
            (sceneConfig?.get(SceneCommonConfig.SCENE_CONFIG_KEY_HOT_SPLASH_INTERVAL)?.asInt
                ?: 1) * 60 * 1000
        if (time < hotSplashInterval) {
            return "不满足最小间隔时间 -- background_time = $time --- TIME_IN_THE_BACKGROUND = $hotSplashInterval"
        }

        return ""
    }
}