package com.transsion.ad.strategy

import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.MbAdPlansBean

/**
 * @author shmizhangxinbing
 * @date : 2025/9/10 11:50
 * @description: 新用户屏蔽策略
 */
object NewUserShieldStrategy {

    /**
     * MMKV 键值对
     */
    private const val USER_CREATE_TIME = "userCreateTime"
    private const val NEW_USER_AD_SHIELD_HOUR = "newUserAdShieldHour"

    private var isDataUpdated = true  // 解决没次读取MMKV
    private var userCreateTime: Long = 0L // 新用户创建时间（毫秒）
    private var newUserAdShieldHour: Int = 0 // 新用户屏蔽时间（小时）


    // =============================================================================================


    /**
     * 获取 userCreateTime
     */
    fun getUserCreateTime(): Long {
        if (isDataUpdated) {
            userCreateTime = AdMmkv.getAdMMKV().getLong(USER_CREATE_TIME, 0L)
        }
        return userCreateTime
    }

    /**
     * 获取 newUserAdShieldHour
     */
    fun getNewUserAdShieldHour(): Int {
        if (isDataUpdated) {
            newUserAdShieldHour = AdMmkv.getAdMMKV().getInt(NEW_USER_AD_SHIELD_HOUR, 0)
        }
        return newUserAdShieldHour
    }

    /**
     * 保存新用户屏蔽数据
     */
    fun saveNewUserShieldData(mbAdPlansBean: MbAdPlansBean?) {
        AdMmkv.getAdMMKV().putLong(USER_CREATE_TIME, mbAdPlansBean?.userCreateTime ?: 0L)
        AdMmkv.getAdMMKV().putInt(NEW_USER_AD_SHIELD_HOUR, mbAdPlansBean?.newUserAdShieldHour ?: 0)
        // App使用过程中也会更新配置，所以这里标记为已更新，下次读取时重新获取数据
        isDataUpdated = true
        AdLogger.logAdInit(
            msg = "NewUserShieldStrategy --> saveNewUserShieldData() --> 保存新用户屏蔽数据 --> userCreateTime = ${mbAdPlansBean?.userCreateTime}, newUserAdShieldHour = ${mbAdPlansBean?.newUserAdShieldHour}",
            writeToFile = false
        )
    }

    /**
     * 是否是新用户
     */
    fun isNewUser(): Boolean {
        // 如果数据已更新，则重新获取数据，否则使用缓存数据
        if (isDataUpdated) {
            userCreateTime = AdMmkv.getAdMMKV().getLong(USER_CREATE_TIME, 0L)
            newUserAdShieldHour = AdMmkv.getAdMMKV().getInt(NEW_USER_AD_SHIELD_HOUR, 0)
        }
        isDataUpdated = false // 标记数据已读取，下次不再重新获取

        // 屏蔽时间设置为0，则不屏蔽新用户，广告展示
        if (newUserAdShieldHour == 0) {
            return false
        }

        // 屏蔽时间不为0，则判断当前时间和创建时间的差值是否大于等于屏蔽时间
        val currentTime = System.currentTimeMillis() // 当前时间（毫秒）
        val timeDiff = currentTime - userCreateTime // 时间差（毫秒）

        // 将屏蔽时间转换为毫秒进行比较 注意：newUserAdShieldTime 单位小时
        val newUserShieldTimeInMillis = newUserAdShieldHour.toLong() * 3600 * 1000 // 将小时转换为毫秒
        return timeDiff < newUserShieldTimeInMillis
    }

}