package com.transsion.ad.view

import android.content.Context
import android.util.AttributeSet
import android.view.MotionEvent
import android.widget.FrameLayout

/**
 * @author shmizhangxinbing
 * @date : 2025/7/22 12:12
 * @description: 滑动监测
 */
class ScrollDetectFrameLayout @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    // 定义滑动状态监听器接口
    interface OnScrollListener {
        fun onScrollStart() // 滑动开始
        fun onScroll(deltaX: Float, deltaY: Float, direction: ScrollDirection) // 滑动中
        fun onScrollEnd() // 滑动结束
    }

    // 定义滑动方向枚举
    enum class ScrollDirection {
        UP, DOWN
    }

    // 滑动监听器
    private var scrollListener: OnScrollListener? = null

    // 记录触摸点和滑动状态
    private var lastX = 0f
    private var lastY = 0f
    private var isScrolling = false

    private val defaultMinMovie: Int = 10

    // 设置滑动监听器
    fun setOnScrollListener(listener: OnScrollListener) {
        scrollListener = listener
    }

    override fun dispatchTouchEvent(event: MotionEvent?): Boolean {
        when (event?.action) {
            MotionEvent.ACTION_DOWN -> {
                // 记录初始触摸点
                lastX = event.x
                lastY = event.y
                isScrolling = false
            }

            MotionEvent.ACTION_MOVE -> {
                val currentX = event.x
                val currentY = event.y

                // 计算滑动的偏移量
                val deltaX = currentX - lastX
                val deltaY = currentY - lastY

                // 只需要关注Y轴滑动
                // 判断是否开始滑动
                //if (!isScrolling && (kotlin.math.abs(deltaX) > defaultMinMovie || kotlin.math.abs(deltaY) > defaultMinMovie)) {
                if (!isScrolling && (kotlin.math.abs(deltaY) > defaultMinMovie)) {
                    isScrolling = true
                    scrollListener?.onScrollStart() // 通知滑动开始
                }

                // 如果正在滑动，通知滑动的偏移量和方向
                if (isScrolling) {
                    val direction = if (deltaY > 0) ScrollDirection.DOWN else ScrollDirection.UP
                    scrollListener?.onScroll(deltaX, deltaY, direction)
                }

                // 更新触摸点
                lastX = currentX
                lastY = currentY
            }

            MotionEvent.ACTION_UP, MotionEvent.ACTION_CANCEL -> {
                // 滑动结束
                if (isScrolling) {
                    scrollListener?.onScrollEnd()
                }
                isScrolling = false
            }
        }
        return super.dispatchTouchEvent(event)
    }
}
