package com.transsion.ad.bidding.base

import android.annotation.SuppressLint
import android.content.Intent
import android.content.pm.ActivityInfo
import android.graphics.drawable.Drawable
import android.os.Build
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.os.Parcelable
import android.os.SystemClock
import android.text.TextUtils
import android.util.Log
import android.view.TextureView
import android.view.View
import android.view.ViewGroup.LayoutParams
import android.widget.FrameLayout
import androidx.activity.OnBackPressedCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.AppCompatImageView
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.load.resource.gif.GifDrawable
import com.bumptech.glide.request.RequestListener
import com.gyf.immersionbar.ImmersionBar
import com.transsion.ad.MBAd
import com.transsion.ad.R
import com.transsion.ad.bidding.gemini.AbsBiddingBuyOutGemini
import com.transsion.ad.config.MemberBusinessSwitch
import com.transsion.ad.log.AdLogger
import com.transsion.ad.log.ILog
import com.transsion.ad.monopoly.model.AdMaterialList
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.strategy.AdTagManager
import com.transsion.ad.strategy.MeasureManager
import com.transsion.ad.util.DebounceClickListener
import com.transsion.player.MediaSource
import com.transsion.player.config.VodConfig
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.ORPlayer
import com.transsion.player.orplayer.PlayError
import java.util.concurrent.atomic.AtomicBoolean


/**
 * @author shmizhangxinbing
 * @date : 2025/6/3 15:12
 * @description: 广告Activity 封装,   处理广告的曝光、埋点
 */
abstract class AbsBiddingActivity : AppCompatActivity(), MeasureManager.ViewVisibilityListener,
    ILog, IPlayerListener {

    /**
     * 传递过来的数据
     */
    private var mAdPlan: AdPlans? = null   // 广告计划
    private var maxEcpmObject: BiddingIntermediateMaterialBean? = null   // 广告计划
    private var mAdMaterialList: AdMaterialList? = null
    private var mAppLayoutId: Int = 0  // App应用信息展示
    private var mSceneId: String? = null   // 场景ID

    /**
     * 标记数据
     */
    private var displayTimestamp = 0L // 广告开始展示的时间 时间戳 毫秒
    private var isReportAdDisplay: AtomicBoolean = AtomicBoolean(false) // 是否已经上报过一次曝光了

    /*** 广告素材是否已经展示*/
    private var isAdMaterialShow: AtomicBoolean = AtomicBoolean(false)

    //protected var isCanFinish: Boolean = false // 当前界面是否可以销毁,只有倒计时结束才可以销毁界面
    private var mIsMute: Boolean = true // 是否打开视频的声音,默认关闭

    /**
     * 倒计时
     */
    private val mHandler = Handler(Looper.getMainLooper()) // 广告倒计时 主线程处理倒计时
    private var countDownTimes: Int = 0 // 倒计时时间

    /**
     * 媒体展示相关
     */
    private var mOrPlayer: ORPlayer? = null // 视频播放器
    private fun getOrPlayer(): ORPlayer? = mOrPlayer

    /**
     * 当前广告是否在屏幕中
     */
    private var isInScreen: Boolean = false


    // =================================== 生命周期方法 ==============================================


    override fun onCreate(savedInstanceState: Bundle?) {
        // 广告默认竖屏展示
        setRequestedOrientation()
        super.onCreate(savedInstanceState)

        // 广告需要 全屏展示
        val immersionBar = ImmersionBar.with(this)
        //immersionBar.transparentBar()
        immersionBar.statusBarAlpha(0f)
        withConfig(immersionBar)
        immersionBar.init()

        setContentView(getSubContentView())

        // 全屏显示 必须在setContentViw之后设置全屏。否则不会生效。
        window.setLayout(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)
        //设置点击外部空白处可以关闭Activity true:可以关闭 false:不能关闭
        this.setFinishOnTouchOutside(false)

        // 初始化数据
        initExtData()

        // 数据校验
        if (getAdPlan() != null && TextUtils.isEmpty(getSceneId()).not()) {
            onLog(Log.DEBUG, "onCreate() --> 开始展示广告", writeToFile = false)
            // 添加到曝光测量
            // 倒计时组件
            countDownTimes = getCountDownTimes()
            MeasureManager.addSession(this)
            showAdInformation()
            showMedia()
            onBackPressedDispatcher()

            // 统一处理是否展示 会员按钮
            val isShowMemberBtn =
                intent?.getBooleanExtra(AbsBiddingBuyOutGemini.EXT_IS_SHOW_MEMBER_BTN, true)
            if (isShowMemberBtn?.not() == true) {
                getMemberContainer()?.visibility = View.GONE
            }

        } else {
            onShowErrorCallback()
        }
    }

    override fun onResume() {
        super.onResume()
        //onLog(msg = "onResume() --> 重新进入页面的时候需要校验一下，当前是否开通了会员需要关闭广告页面")
        if (MemberBusinessSwitch.isSkipShowAd()) {
            adRewarded()
            finish()
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        mHandler.removeCallbacksAndMessages(null)
        MeasureManager.removeSession(this) // View 监测移除
        getOrPlayer()?.release() // 播放器释放
        // 上报一次累计展示时间
        reportShowTime()
        adClose() // 广告关闭回调
    }


    // ================================= 页面曝光监测 ================================================


    /**
     * 需要曝光监测的View
     */
    override fun getVisibilityView(): View? {
        return getMediaContain()
    }

    override fun onVisibilityChanged(isVisible: Boolean) {

        // 从不可见到可见都需要回调
        if (isInScreen.not() && isVisible) {
            adResume()
        }
        isInScreen = isVisible

        if (isVisible) {
            // 开始展示的时间
            if (displayTimestamp == 0L) {
                displayTimestamp = SystemClock.elapsedRealtime()
                startCountDown() // 可见的时候开始计时
            }

            // 视频状态处理
            if (getOrPlayer()?.isPlaying() == false) {
                getOrPlayer()?.play()
            }

            // 曝光上报
            if (isReportAdDisplay.get().not() && isAdMaterialShow.get()) {
                isReportAdDisplay.set(true)
                display()
            }
        } else {
            getOrPlayer()?.pause()
            // 播放时长上报
            reportShowTime()
            pauseCountDown() // 不可见的时候暂停计时
        }
    }


    // ================================= 多媒体播放展示 ===============================================


    override fun onPlayError(errorInfo: PlayError, mediaSource: MediaSource?) {
        super.onPlayError(errorInfo, mediaSource)
        onLog(
            Log.ERROR,
            "onPlayError() --> errorInfo = $errorInfo -- sceneId = ${getSceneId()} --> url = ${getAdMaterialList()?.video?.url} --> path = ${getAdMaterialList()?.video?.path}"
        )
        onShowErrorCallback()
    }

    override fun onPrepare(mediaSource: MediaSource?) {
        super.onPrepare(mediaSource)
        getOrPlayer()?.play()
    }

    override fun onRenderFirstFrame() {
        super.onRenderFirstFrame()
        isAdMaterialShow.set(true)
    }

    override fun onVideoPause(mediaSource: MediaSource?) {
        super.onVideoPause(mediaSource)
    }

    override fun onVideoStart(mediaSource: MediaSource?) {
        super.onVideoStart(mediaSource)
    }

    override fun onCompletion(mediaSource: MediaSource?) {
        super.onCompletion(mediaSource)
        //pauseCountDown()
        //isCanFinish = true
        //refreshCountDownView(0)¬
        // 这里是为了处理 视频小于倒计时时间的情况
        //onAdCompletion()
    }


    // ===================================== 子类需要实现的方法 =======================================

    override fun getSceneId(): String = mSceneId ?: ""

    abstract fun getAdType(): Int                       // 获取广告的类型
    abstract fun getSubContentView(): View?             // 获取广告View
    abstract fun showAdInformation()                    // 展示广告详情
    abstract fun getMediaContain(): FrameLayout?        // 获取多媒体容器
    abstract fun getVolumeContainer(): FrameLayout?     // 音量控件容器
    abstract fun getCountDownTimes(): Int               // 获取倒计时时间
    abstract fun refreshCountDownView(i: Int)           // 倒计时刷新
    abstract fun onAdCompletion()                       // 倒计时结束
    abstract fun showCloseAdDialog()                    // 页面关闭提示弹窗
    abstract fun getAdIcon(): View?                     // AdTagView

    abstract fun getMemberContainer(): View?            // 会员入口容器


    // =============================================================================================


    open fun onImageAdd(imageView: AppCompatImageView?, width: Int?, height: Int?) {}

    open fun getLooping(): Boolean = true               // 是否需要循环播放
    open fun getMute(): Boolean = true                  // 默认静音

    open fun withConfig(with: ImmersionBar) {}

    /**
     * 关闭广告页面
     */
    fun finishAdActivity() {
        finish()
    }

    /**
     * 会员入口
     */
    fun startMemberActivity() {
        MBAd.getAdInitParams()?.businessBridge?.startMemberActivity(getSceneId())
    }

    @SuppressLint("SourceLockedOrientationActivity")
    open fun setRequestedOrientation() {
        if (Build.VERSION.SDK_INT != Build.VERSION_CODES.O) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        }
    }

    /**
     * 展示和关闭退出激励视频弹窗
     * 1. 视频暂停
     * 2. 倒计时暂停
     */
    fun setShowCloseAdDialog(isShow: Boolean) {
        if (isShow) {
            getOrPlayer()?.pause()
            pauseCountDown()
        } else {
            getOrPlayer()?.play()
            startCountDown()
        }
    }

    /**
     * 获取延迟关闭时间
     */
    fun getDelayCloseTime(): Int {
        return SceneCommonConfig.getDelayCloseTime(getSceneId())
    }


    // =============================================================================================


    fun getAppLayoutId(): Int = mAppLayoutId
    fun getAdPlan(): AdPlans? = mAdPlan
    fun getAdMaterialList(): AdMaterialList? = mAdMaterialList


    // ==================================== 埋点处理 =================================================


    /***广告重新进入屏幕*/
    fun adResume() {
        // 创建一个Intent，指定广播的动作
        val intent = Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_RESUME))
        // 添加额外的数据（如果需要）
        intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
        LocalBroadcastManager.getInstance(this).sendBroadcast(intent)
    }

    /*** 广告展示埋点*/
    fun display() {
        // 创建一个Intent，指定广播的动作
        val intent = Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_DISPLAY))
        // 添加额外的数据（如果需要）
        intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
        LocalBroadcastManager.getInstance(this).sendBroadcast(intent)
    }

    /*** 广告点击埋点*/
    fun adClick() {
        // 创建一个Intent，指定广播的动作
        val intent = Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_CLICK))
        // 添加额外的数据（如果需要）
        intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
        LocalBroadcastManager.getInstance(this).sendBroadcast(intent)
    }

    /*** 上报累计展示埋点*/
    private fun reportShowTime() {
        if (displayTimestamp > 0) {
            // 上报曝光时长
            // 创建一个Intent，指定广播的动作
            val intent =
                Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_DISPLAY_TIMESTAMP))
            // 添加额外的数据（如果需要）
            intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
            intent.putExtra(
                AbsBiddingBuyOutGemini.EXT_DISPLAY_TIMESTAMP,
                SystemClock.elapsedRealtime() - displayTimestamp
            )
            LocalBroadcastManager.getInstance(this).sendBroadcast(intent)
            displayTimestamp = 0
        }
    }

    /*** 获取奖励*/
    fun adRewarded() {
        // 创建一个Intent，指定广播的动作
        val intent = Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_REWARDED))
        // 添加额外的数据（如果需要）
        intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
        LocalBroadcastManager.getInstance(this).sendBroadcast(intent)
    }

    /*** 广告页面关闭埋点*/
    private fun adClose() {
        // 创建一个Intent，指定广播的动作
        val intent = Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_CLOSE))
        // 添加额外的数据（如果需要）
        intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
        LocalBroadcastManager.getInstance(this).sendBroadcast(intent)
    }

    /*** 展示失败回调*/
    private fun onShowErrorCallback() {
        getMediaContain()?.post {
            // 创建一个Intent，指定广播的动作
            val intent = Intent(intent.getStringExtra(AbsBiddingBuyOutGemini.EXT_ACTION_SHOW_ERROR))
            // 添加额外的数据（如果需要）
            intent.putExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT, maxEcpmObject)
            LocalBroadcastManager.getInstance(this).sendBroadcast(intent)

            // 展示失败，关闭当前
            finishAdActivity()
        }
    }


    // ================================== 数据UI 初始化 ==============================================


    /*** 初始化 数据*/
    private fun initExtData() {
        runCatching {
            // 解析参数
            mSceneId = intent?.getStringExtra(AbsBiddingBuyOutGemini.EXT_SCENE_ID)
            mAppLayoutId = intent?.getIntExtra(AbsBiddingBuyOutGemini.EXT_APP_LAYOUT_ID, 0) ?: 0

            runCatching {
                // 处理 FM Parcelable 异常
                val rawExtras = intent?.extras
                rawExtras?.keySet()?.forEach {
                    val value = rawExtras.get(it)
                    onLog(
                        level = Log.ERROR,
                        "处理 FM Parcelable 异常 --> Bundle key = $it, valueClass = ${value?.javaClass?.name}",
                        writeToFile = false
                    )
                }
            }

            // 接受传递过来的数据
            maxEcpmObject = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                intent?.getParcelableExtra(
                    AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT,
                    BiddingIntermediateMaterialBean::class.java
                )
            } else {
                intent?.getParcelableExtra(AbsBiddingBuyOutGemini.EXT_AD_MAX_ECPM_OBJECT)
            }
            mAdPlan = maxEcpmObject?.plans
            // 目前规定一个计划只有一个素材
            if (mAdPlan?.adMaterialList?.isNotEmpty() == true) {
                mAdMaterialList = mAdPlan?.adMaterialList?.get(0)
            }
        }.getOrElse {
            onLog(level = Log.ERROR, "initExtData() --> it = ${it.message}")
        }
    }

    /*** 页面销毁统一拦截处理*/
    private fun onBackPressedDispatcher() {
        onBackPressedDispatcher.addCallback(this, object : OnBackPressedCallback(true) {
            override fun handleOnBackPressed() {
                // 插屏、激励视频、开屏 不可以通过 系统返回事件处理
            }
        })
    }

    /*** 多媒体数据展示*/
    private fun showMedia() {
        // 展示媒体内容
        if (getAdMaterialList()?.type == AdMaterialList.NON_AD_TYPE_TEXT) {
            showImage()
        } else {
            playVideo()
        }

        // 点击事件处理
        getMediaContain()?.setOnClickListener(DebounceClickListener {
            adClick()
        })

        // 声音开关
        showVolume()

        // AD 标识展示处理
        // 广告 Logo 展示
        getAdIcon()?.apply {
            AdTagManager.handleForAdTagView(this)
        }
    }

    /*** 设置播放器 静音状态*/
    private fun setMute(): Boolean {
        mIsMute = mIsMute.not()
        getOrPlayer()?.setMute(mIsMute)
        return mIsMute
    }

    /*** 声音开关*/
    private fun showVolume() {
        getVolumeContainer()?.apply {
            visibility = if (getAdMaterialList()?.type == AdMaterialList.NON_AD_TYPE_TEXT) {
                View.GONE
            } else {
                val volumeImageView = AppCompatImageView(this@AbsBiddingActivity)
                volumeImageView.setImageResource(R.mipmap.ad_volumeoff)
                removeAllViews()
                addView(volumeImageView, LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)
                setOnClickListener {
                    if (setMute()) {
                        volumeImageView.setImageResource(R.mipmap.ad_volumeoff)
                    } else {
                        volumeImageView.setImageResource(R.mipmap.ad_volumeon)
                    }
                }
                View.VISIBLE
            }
        }
    }

    /*** 图片广告*/
    private fun showImage() {
        // 如果本地地址是空的，那就不使用线上地址在线加载
        getMediaContain()?.let {
            val imageView = AppCompatImageView(this@AbsBiddingActivity)
            imageView.layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT
            )

            getAdMaterialList()?.image?.let { image ->
                val imagePath = image.path
                val imageUrl = image.url
                val loadUrl = if (TextUtils.isEmpty(imagePath)) imageUrl else imagePath

                // 监听图片加载状态
                Glide.with(this).load(loadUrl).listener(object : RequestListener<Drawable?> {
                    override fun onLoadFailed(
                        e: GlideException?,
                        model: Any?,
                        target: com.bumptech.glide.request.target.Target<Drawable?>?,
                        isFirstResource: Boolean,
                    ): Boolean {
                        // 加载失败
                        onLog(
                            level = Log.ERROR,
                            msg = "showImage() --> 图片加载失败 --> e = ${e?.message}"
                        )
                        onShowErrorCallback()
                        return false
                    }

                    override fun onResourceReady(
                        resource: Drawable?,
                        model: Any?,
                        target: com.bumptech.glide.request.target.Target<Drawable?>?,
                        dataSource: DataSource?,
                        isFirstResource: Boolean,
                    ): Boolean {
                        // 加载成功
                        // 判断是否为GIF，如果是，启动动画
                        if (resource is GifDrawable) {
                            resource.start()
                        }
                        isAdMaterialShow.set(true)
                        // 返回 false，让Glide继续处理图片显示
                        return false
                    }
                }).into(imageView)

                // 无论图片加载成功与否 通知容器更新大小
                onImageAdd(imageView, width = image.width, height = image.height)
            }

            it.removeAllViews() // 如果只想显示当前图片，建议先清除
            it.addView(imageView)
        }
    }

    /*** 视频广告*/
    private fun playVideo() {
        getMediaContain()?.let {
            val textureView = TextureView(this@AbsBiddingActivity)
            // 创建播放器
            mOrPlayer = ORPlayer.Builder(this)
                .vodConfig(VodConfig(clearFrameWhenStop = true, openAudioFocus = false)).builder()
                .apply {
                    setMute(getMute())
                    setTextureView(textureView)
                    setLooping(getLooping()) // 插屏需要循环播放，激励视频不需要循环播放
                    setPlayerListener(this@AbsBiddingActivity)
                }

            // 开始播放
            val url = if (TextUtils.isEmpty(getAdMaterialList()?.video?.path)) {
                getAdMaterialList()?.video?.url ?: ""
            } else {
                getAdMaterialList()?.video?.path ?: ""
            }
            mOrPlayer?.setDataSource(MediaSource(url, url))
            mOrPlayer?.prepare()

            it.addView(
                textureView, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)
            )
        }
    }


    // ======================================= 倒计时处理 ============================================


    /**
     * 倒计时任务
     */
    private val countdownRunnable = Runnable {
        refreshCountDownView(countDownTimes)
        onLog(
            level = Log.DEBUG,
            msg = "countdownRunnable() --> times = ${getCountDownTimes()} --> i = $countDownTimes",
            writeToFile = false
        )
        if (countDownTimes == 0) {
            onAdCompletion()
        }
        countDownTimes--
        startCountDown()
    }

    /**
     * 开始倒计时
     */
    private fun startCountDown() {
        pauseCountDown()
        if (countDownTimes < 0) {
            //isCanFinish = true
        } else {
            mHandler.postDelayed(countdownRunnable, 1000)
        }
    }

    private fun pauseCountDown() {
        mHandler.removeCallbacksAndMessages(null)
    }
}