package com.transsion.ad.log

import android.util.Log
import com.tn.lib.logger.MBLogger

/**
 * @author shmizhangxinbing
 * @date : 2025/8/19 11:54
 * @description:
 *
 * Verbose（冗长）：Log.v(TAG, "Your message");      - 用于记录最详细的信息，通常用于调试。
 * Debug（调试）：Log.d(TAG, "Your message");        - 用于调试信息，记录应用的运行状态。
 * Info（信息）：Log.i(TAG, "Your message");         - 用于记录一般信息，表示应用的正常运行状态。
 * Warn（警告）：Log.w(TAG, "Your message");         - 用于记录潜在问题的警告信息。
 * Error（错误）：Log.e(TAG, "Your message");        - 用于记录错误信息，表示发生了错误。
 * Assert（断言）：Log.wtf(TAG, "Your message");     - 用于记录程序中的严重问题，通常是不可恢复的错误。
 */
object AdLogger {

    /**
     * SDK 内部日志、业务日志、SDK初始化日志
     */
    const val TAG_BASE = "ad_" // base
    const val TAG_INIT = TAG_BASE + "init" // 初始化相关

    /**
     * 下面是各个业务线 tag
     */
    const val TAG_AD_PS = TAG_BASE + "ps"               // PS 相关的
    const val TAG_AD_BANNER = TAG_BASE + "b"            // banner
    const val TAG_AD_SPLASH = TAG_BASE + "s"            // 开屏
    const val TAG_AD_INTERSTITIAL = TAG_BASE + "int"    // 插屏
    const val TAG_AD_VIDEO = TAG_BASE + "v"             // 激励视频
    const val TAG_AD_NATIVE = TAG_BASE + "n"            // native
    const val TAG_AD_IRREGULAR = TAG_BASE + "fb"        // 非标 简化版本Native
    const val TAG_AD_ICON = TAG_BASE + "ic"             // icon


    // ==================================== 日志输出 ================================================


    /**
     * 日志输出
     *
     * 替换日志输出
     */
    private fun log(level: Int, tag: String, msg: String, writeToFile: Boolean = true) {
        when (level) {
            Log.ERROR -> {
                MBLogger.e(tag, msg, writeToFile = writeToFile)
            }

            Log.WARN -> {
                MBLogger.w(tag, msg, writeToFile = writeToFile)
            }

            Log.DEBUG -> {
                MBLogger.d(tag, msg, writeToFile = writeToFile)
            }

            Log.INFO -> {
                MBLogger.i(tag, msg, writeToFile = writeToFile)
            }
        }
    }


    // =============================================================================================


    /**
     * 通用日志输出
     *
     * 广告一定会输出
     */
    fun logCommon(msg: String, level: Int = Log.DEBUG, writeToFile: Boolean = true) {
        log(level = level, tag = TAG_BASE, msg = msg, writeToFile = writeToFile)
    }

    /**
     * 广告初始化
     */
    fun logAdInit(msg: String, level: Int = Log.INFO, writeToFile: Boolean = true) {
        log(level = level, tag = TAG_INIT, msg = msg, writeToFile = writeToFile)
    }

    /**
     * SDK -- 通用 -- ad_sdk_
     */
    fun logSdk(tag: String, msg: String, level: Int = Log.DEBUG, writeToFile: Boolean = true) {
        log(level = level, tag = tag, msg = msg, writeToFile = writeToFile)
    }

    /**
     * PS 相关业务
     */
    fun logPS(msg: String, level: Int = Log.DEBUG, writeToFile: Boolean = true) {
        log(level = level, tag = TAG_AD_PS, msg = msg, writeToFile = writeToFile)
    }

    /***
     * SDK -- Irregular(非标) -- ad_sdk_fb
     */
    fun logSdkIrregular(msg: String, level: Int = Log.DEBUG, writeToFile: Boolean = true) {
        log(level = level, tag = TAG_AD_IRREGULAR, msg = msg, writeToFile = writeToFile)
    }

    /**
     * Icon
     */
    fun logIcon(msg: String, level: Int = Log.DEBUG, writeToFile: Boolean = true) {
        log(level = level, tag = TAG_AD_ICON, msg = msg, writeToFile = writeToFile)
    }

    /**
     * 业务可以调用
     */
    fun logBusiness(
        tag: String = TAG_BASE,
        level: Int = Log.DEBUG,
        writeToFile: Boolean = false,
        msg: String,
    ) {
        log(level = level, tag = tag, msg = msg, writeToFile = writeToFile)
    }

}