package com.transsion.ad.util

import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.text.SpannableString
import android.text.style.ForegroundColorSpan
import android.text.style.ReplacementSpan
import com.blankj.utilcode.util.SizeUtils

/**
 * @param lineWidthDp 分隔线宽度（默认1dp）
 * @param marginVerticalDp 分隔线上下边距（默认4dp）
 * @param leftTextColor 左侧文本颜色（可选，不传则不设置）
 * @param rightTextColor 右侧文本颜色（可选，不传则不设置）
 */
data class TextWithDividerConfig(
    val lineWidthDp: Int = 1,
    val lineColor: Int = Color.LTGRAY,
    val marginVerticalDp: Int = 4,
    val leftTextColor: Int? = null,
    val rightTextColor: Int? = null
)

object TextWithDividerHelper {

    /**
     * @param leftText 左侧文本内容
     * @param rightText 右侧文本内容
     * @param config 配置参数
     * @return 配置好的SpannableString
     */
    fun create(
        leftText: String,
        rightText: String,
        config: TextWithDividerConfig = TextWithDividerConfig()
    ): SpannableString {
        val separator = if (leftText.isNotBlank() && rightText.isNotBlank()) " " else ""
        val combinedText = "$leftText$separator$rightText"
        val spannableString = SpannableString(combinedText)

        val linePosition = leftText.length
        spannableString.setSpan(
            VerticalLineSpan(config),
            linePosition,
            linePosition + 1,
            SpannableString.SPAN_INCLUSIVE_EXCLUSIVE
        )

        config.leftTextColor?.let { color ->
            spannableString.setSpan(
                ForegroundColorSpan(color),
                0,
                leftText.length,
                SpannableString.SPAN_INCLUSIVE_EXCLUSIVE
            )
        }

        config.rightTextColor?.let { color ->
            val rightTextStart = leftText.length + 1
            val rightTextEnd = rightTextStart + rightText.length
            spannableString.setSpan(
                ForegroundColorSpan(color),
                rightTextStart,
                rightTextEnd,
                SpannableString.SPAN_INCLUSIVE_EXCLUSIVE
            )
        }

        return spannableString
    }
}

class VerticalLineSpan(
    private val config: TextWithDividerConfig = TextWithDividerConfig()
) : ReplacementSpan() {
    private val marginVertical = SizeUtils.dp2px(config.marginVerticalDp.toFloat())
    private val lineWidth = SizeUtils.dp2px(config.lineWidthDp.toFloat())

    override fun getSize(
        paint: Paint,
        text: CharSequence?,
        start: Int,
        end: Int,
        fm: Paint.FontMetricsInt?
    ): Int {
        return lineWidth
    }

    override fun draw(
        canvas: Canvas,
        text: CharSequence?,
        start: Int,
        end: Int,
        x: Float,
        top: Int,
        y: Int,
        bottom: Int,
        paint: Paint
    ) {
        val originalColor = paint.color
        val originalStrokeWidth = paint.strokeWidth
        paint.color = config.lineColor
        paint.strokeWidth = lineWidth.toFloat()

        canvas.drawLine(x, top.toFloat() + marginVertical,
            x, bottom.toFloat() - marginVertical, paint)

        paint.color = originalColor
        paint.strokeWidth = originalStrokeWidth
    }
}