package com.transsion.ad.bidding.base

import android.app.Activity
import android.content.Context
import android.os.Handler
import android.os.Looper
import android.text.TextUtils
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.hisavana.common.bean.AdditionalInfo
import com.hisavana.common.bean.TAdErrorCode
import com.hisavana.common.bean.TAdNativeInfo
import com.transsion.ad.MBAd
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.gemini.AbsBiddingBuyOutGemini
import com.transsion.ad.monopoly.intercept.NonAdShowedTimesManager
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdShowLevel
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.monopoly.model.MbAdType
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.monopoly.plan.AdPlanUtil
import com.transsion.ad.report.AdReportProvider
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.scene.SceneOnOff
import com.transsion.ad.config.TestConfig
import com.transsion.ad.log.ILog
import com.transsion.ad.ps.PsDbManager
import com.transsion.ad.ps.PsOfferProvider
import com.transsion.ad.ps.attribution.AttributionProduceManager
import com.transsion.ad.report.BiddingStateEnum
import com.transsion.ad.strategy.AdClickManager
import com.transsion.ad.strategy.AdOverridePendingTransitionManager
import com.transsion.ad.strategy.AdUrlParameterManager
import com.transsion.ad.strategy.DisplayIntervalTimeManager
import com.transsion.ad.strategy.NewUserShieldStrategy
import com.transsion.ad.util.RandomUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/5/27 19:21
 * @description: 竞价逻辑封装
 *
 * loadAd() --> 前置条件判断 --> 触发埋点 --> 读取所有广告计划 --> 找出可用计划 --> 竞价 --> 回调
 */
abstract class AbsAdBidding : BiddingTAdditionalListener(), ILog {

    /**
     * 传递进来的数据
     */
    private var mSceneId: String? = null // 场景调用的时候传进来的场景ID
    private var mSceneSubId: String? = null // 场景调用的时候传进来的场景ID
    private var mCtxMap: Map<String, Any>? = null// 上下文信息
    private var mListener: BiddingTAdditionalListener? = null // 广告监听回调 加载的场景提供
    private var mContext: Context? = null // 上下文
    private var mLayoutId: Int? = null    // 自定义布局
    private var mAdOverridePendingTransitionEnum: AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum? =
        null // 广告Activity页面打开样式
    private var mFetchCount: Int = 1

    /*** 是否正在加载中*/
    private val isLoading: AtomicBoolean = AtomicBoolean(false)

    /*** 埋点使用 链路关系*/
    private var triggerId: String = ""
    protected fun getTriggerId(): String = triggerId

    /*** 竞价 中间产物数据*/
    private val mBiddingHandler: Handler = Handler(Looper.getMainLooper()) // 竞价计时
    private var biddingPlanList: MutableList<BiddingIntermediateMaterialBean>? = null // 符合条件的计划
    private var maxEcpmObject: BiddingIntermediateMaterialBean? = null // 竞价胜出对象

    /*** 广告相关回调通过UI线程回调*/
    private val mUIHandler: Handler = Handler(Looper.getMainLooper())


    // ===================================== 子类需要重写的API ========================================


    /*** 获取广告类型*/
    abstract fun getAdType(): Int

    /*** Activity 广告必须实现*/
    abstract fun getGemini(): AbsBiddingBuyOutGemini?

    /*** 添加HiSavana Provider，Hi程序化作为广告源提供广告*/
    abstract fun addHiSavanaProvider(
        biddingPlan: MutableList<BiddingIntermediateMaterialBean>,
        adPlans: AdPlans,
        fetchCount: Int,
    )


    // ======================================== get set ============================================
    // ======================================== get set ============================================
    // ======================================== get set ============================================


    override fun getSceneId(): String = mSceneId ?: ""
    protected fun getSceneSubId(): String? = mSceneSubId
    fun getContext(): Context? = mContext
    fun getListener(): BiddingTAdditionalListener? = mListener
    fun getCtxMap(): Map<String, Any> = mCtxMap ?: emptyMap()
    fun getLayoutId(): Int? = mLayoutId
    private fun getAdOverridePendingTransition(): AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum? {
        return mAdOverridePendingTransitionEnum
    }

    fun setSceneId(sceneId: String?): AbsAdBidding {
        mSceneId = sceneId
        return this@AbsAdBidding
    }

    fun setSceneSubId(sceneSubId: String?): AbsAdBidding {
        mSceneSubId = sceneSubId
        return this@AbsAdBidding
    }

    fun setCtxMap(ctxMap: Map<String, Any>?): AbsAdBidding {
        mCtxMap = ctxMap
        return this@AbsAdBidding
    }

    fun setListener(listener: BiddingTAdditionalListener?): AbsAdBidding {
        mListener = listener
        return this@AbsAdBidding
    }

    fun setContext(context: Context): AbsAdBidding {
        mContext = context
        return this@AbsAdBidding
    }

    fun setAppLayoutId(layoutId: Int?): AbsAdBidding {
        mLayoutId = layoutId
        return this@AbsAdBidding
    }

    fun setAdOverridePendingTransition(adOverridePendingTransitionEnum: AdOverridePendingTransitionManager.AdOverridePendingTransitionEnum?): AbsAdBidding {
        mAdOverridePendingTransitionEnum = adOverridePendingTransitionEnum
        return this@AbsAdBidding
    }


    // =============================================================================================


    /*** 获取竞价成功的对象*/
    fun getMaxEcpmObject(): BiddingIntermediateMaterialBean? {
        return maxEcpmObject
    }

    /*** 参与竞价的广告计划列表*/
    fun getBiddingPlanList(): MutableList<BiddingIntermediateMaterialBean>? = biddingPlanList

    /**
     * 展示广告
     *
     * 这个方法仅用于 激励视频、插屏广告展示的调用
     *
     * Activity 广告统一打开方法
     */
    fun startAdActivity(
        activity: Activity?,
        maxEcpmObject: BiddingIntermediateMaterialBean?,
        isShowMemberBtn: Boolean = true, // 是否展示会员按钮
    ) {
        if (InterceptUtil.isInterceptAdShowed()) {
            onBiddingWrapperAdShowError(maxEcpmObject)
            return
        }

        // 拦截广告展示逻辑，防止广告重复曝光
        // 展示失败和关闭的时候重置数据
        InterceptUtil.onAdShow()

        (activity as? AppCompatActivity)?.lifecycleScope?.launch {
            val hiProvider = maxEcpmObject?.hiSavanaInterceptProvider
            if (hiProvider == null) {
                if (null != maxEcpmObject?.plans) {
                    activity.let {
                        getGemini()?.setListener(this@AbsAdBidding)?.registerReceiver()
                            ?.setSceneId(getSceneId())?.setAppLayoutId(getLayoutId())
                            ?.isShowMemberBtn(isShowMemberBtn)
                            ?.startActivity(it, getSceneId(), maxEcpmObject) ?: run {
                            onBiddingWrapperAdShowError(maxEcpmObject)
                        }
                    }
                } else {
                    // 理论上来说这里是不会调用的，不排除API调用错误在展示之前调用了 destroy()
                    onBiddingWrapperAdShowError(maxEcpmObject)
                }
            } else {
                hiProvider.setListener(this@AbsAdBidding)
                hiProvider.showAd(activity, getLayoutId(), getSceneId())
            }
        } ?: run {
            onBiddingWrapperAdShowError(maxEcpmObject)
        }
    }

    open fun checkContextNonNull(): Boolean = false // 是否需要检查context是否为空

    /*** 资源回收*/
    open fun destroy() {
        setListener(null)
        setCtxMap(null)
        getMaxEcpmObject()?.hiSavanaInterceptProvider?.destroy(this@AbsAdBidding) // 激励视频、插屏广告移除监听
        mBiddingHandler.removeCallbacksAndMessages(null)
        maxEcpmObject = null

        getGemini()?.destroy() // 子类对象释放

        if (getAdType() != MbAdType.MB_AD_TYPE_NATIVE) {
            onLog(Log.DEBUG, "destroy() --> 资源回收", writeToFile = false)
        }
        setSceneId(null)
    }


    // =================================== 广告回调 ==================================================
    // =================================== 广告回调 ==================================================
    // =================================== 广告回调 ==================================================


    /**
     * 统一处理广告 加载展示 回调
     */
    override fun onBiddingError(p0: TAdErrorCode?) {
        super.onBiddingError(p0)
        onLog(
            level = Log.ERROR, msg = "onBiddingError() --> errorMessage = ${p0?.errorMessage}"
        )
        // 重置状态
        isLoading.set(false)
        mUIHandler.post {
            getListener()?.onBiddingError(p0) // 传递出去
        }
    }

    override fun onBiddingLoad(maxEcpmObject: List<BiddingIntermediateMaterialBean>?) {
        super.onBiddingLoad(maxEcpmObject)

        onLog(
            level = Log.DEBUG,
            msg = "onBiddingLoad() --> 竞价完成 --> maxEcpmObject.size = ${maxEcpmObject?.size}",
            writeToFile = false
        )

        // 竞价成功上报
        maxEcpmObject?.forEach {
            // 中转
            it.sceneSubId = getSceneSubId()
            it.sceneId = getSceneId()

            AdReportProvider.biddingReport(
                triggerId = getTriggerId(),
                sceneId = getSceneId(),
                adType = getAdType(),
                msg = "竞价成功 --> ecpmList = $it , ecpm = ${it.ecpm} , plans?.id = ${it.plans?.id} , plans?.name = ${it.plans?.name}",
                result = BiddingStateEnum.BIDDING_REPORT_BIDDING_SUCCESS,
                ecpmList = null,
                ecpm = "${it.ecpm}",
                planId = "${it.plans?.id}",
                planName = "${it.plans?.name}",
                sceneSubId = getSceneSubId()
            )
        }

        isLoading.set(false)
        mUIHandler.post {
            getListener()?.onBiddingLoad(maxEcpmObject) // 传递出去
        }
    }

    override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingLoad(maxEcpmObject)

        // 保存在本地
        this.maxEcpmObject = maxEcpmObject

        // 获取所有的ecpm
        val ecpmList = mutableListOf<String?>()
        getBiddingPlanList()?.forEach {
            ecpmList.add(it.ecpm?.toString())
        }
        onLog(
            level = Log.DEBUG,
            msg = "onBiddingLoad() --> 竞价完成 --> ecpmList = $ecpmList --> ecpm = ${maxEcpmObject?.ecpm} --> plans?.id = ${maxEcpmObject?.plans?.id} --> plans?.name = ${maxEcpmObject?.plans?.name}",
            writeToFile = false
        )

        // 竞价成功上报
        AdReportProvider.biddingReport(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adType = getAdType(),
            msg = "竞价成功 --> ecpmList = $ecpmList , ecpm = ${maxEcpmObject?.ecpm} , plans?.id = ${maxEcpmObject?.plans?.id} , plans?.name = ${maxEcpmObject?.plans?.name}",
            result = BiddingStateEnum.BIDDING_REPORT_BIDDING_SUCCESS,
            ecpmList = ecpmList.toString(),
            ecpm = "${maxEcpmObject?.ecpm}",
            planId = "${maxEcpmObject?.plans?.id}",
            planName = "${maxEcpmObject?.plans?.name}",
            sceneSubId = getSceneSubId()
        )

        isLoading.set(false)
        mUIHandler.post {
            getListener()?.onBiddingLoad(maxEcpmObject) // 传递出去
        }
    }

    /**
     * Hi程序化广告回调 埋点
     *
     * SSP的回到不走这里，BiddingNativeManager自己实现
     */
    override fun onShow(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onShow(p0, p1)
        // Native 广告会使用一个TNativeAd，所以回多次回调，这里需要判断一下是不是当前展示的广告
        if (getAdType() == MbAdType.MB_AD_TYPE_NATIVE && TextUtils.equals(
                getMaxEcpmObject()?.nativeInfo?.adId, p0?.adId
            ).not()
        ) {
            return
        }
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = getMaxEcpmObject()?.plans?.id,
            adSource = p1.source,
            adId = p1.codeSeatId,
            adType = getAdType(),
            isAdShowFinal = false,
            psId = null,
            bidEcpmCent = getMaxEcpmObject()?.ecpm,
            ecpmCent = null,
            sceneSubId = getSceneSubId()
        )
        onBiddingWrapperAdDisplay(getMaxEcpmObject())
    }

    override fun onShowError(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onShowError(p0, p1)
        onLog(
            level = Log.ERROR,
            msg = "onShowError() --> errorMessage = ${p0?.errorMessage} --> placementId = ${p1.placementId}"
        )
        onBiddingWrapperAdShowError(getMaxEcpmObject())
    }

    override fun onClick(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onClick(p0, p1)
        // Native 广告会使用一个TNativeAd，所以回多次回调，这里需要判断一下是不是当前展示的广告
        if (getAdType() == MbAdType.MB_AD_TYPE_NATIVE && TextUtils.equals(
                getMaxEcpmObject()?.nativeInfo?.adId, p0?.adId
            ).not()
        ) {
            return
        }
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = getMaxEcpmObject()?.plans?.id,
            adSource = p1.source,
            adId = p1.codeSeatId,
            adType = getAdType(),
            isAdShowFinal = false,
            psId = null,
            bidEcpmCent = getMaxEcpmObject()?.ecpm,
            ecpmCent = null,
            sceneSubId = getSceneSubId()
        )
        onBiddingWrapperAdClick(getMaxEcpmObject())
    }

    override fun onRewarded() {
        super.onRewarded()
        onBiddingWrapperAdRewarded(getMaxEcpmObject())
    }

    override fun onClosed(p0: Int) {
        super.onClosed(p0)
        onBiddingWrapperAdClose(getMaxEcpmObject())
    }

    /**
     * 广告回复可见回调
     */
    override fun onBiddingBuyOutResume(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingBuyOutResume(maxEcpmObject)
        // 广告展示之后判断是否需要提前预加载落地页
        val h5LinkPreload = maxEcpmObject?.plans?.h5LinkPreload ?: false
        val h5Link = AdPlanUtil.getAdMaterial(maxEcpmObject?.plans)?.h5Link
        if (h5LinkPreload && TextUtils.isEmpty(h5Link).not()) {
            // h5Link 需要宏替换
            val newUrl =
                AdUrlParameterManager.replaceParameter(url = h5Link ?: "", logTag = "on Ad show")
            mBiddingHandler.post {
                MBAd.getAdInitParams()?.businessBridge?.preLoadWebview(newUrl)
            }
        }
    }

    /**
     * BuyOut 广告回调 埋点
     */
    override fun onBiddingBuyOutDisplay(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingBuyOutDisplay(maxEcpmObject)
        // 标记当前广告已经曝光了
        maxEcpmObject?.isExpend = true

        // PS直投归因处理
        AttributionProduceManager.onBiddingBuyOutDisplay(maxEcpmObject?.plans)
        // 广告埋点上报
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = maxEcpmObject?.plans?.id,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = AdPlanUtil.getAdMaterial(maxEcpmObject?.plans)?.id,
            adType = getAdType(),
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(maxEcpmObject?.plans),
            psId = AdPlanUtil.getPsId(maxEcpmObject?.plans),
            psPackageName = AdPlanUtil.getPsPackageName(maxEcpmObject?.plans),
            bidEcpmCent = maxEcpmObject?.plans?.bidEcpmCent,
            ecpmCent = maxEcpmObject?.plans?.ecpmCent,
            sceneSubId = getSceneSubId()
        )
        // PS Offer 处理
        PsDbManager.onBiddingBuyOutDisplay(maxEcpmObject?.plans)
        // 回调给媒体
        onBiddingWrapperAdDisplay(maxEcpmObject)
    }

    override fun onBiddingBuyOutShowError(
        p0: TAdErrorCode?,
        maxEcpmObject: BiddingIntermediateMaterialBean?,
    ) {
        super.onBiddingBuyOutShowError(p0, maxEcpmObject)
        onBiddingWrapperAdShowError(maxEcpmObject)
    }

    override fun onBiddingBuyOutClick(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingBuyOutClick(maxEcpmObject)
        // PS直投归因处理 特殊处理
        // AttributionProduceManager.onBiddingBuyOutClick(plans)
        // 广告埋点上报
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = maxEcpmObject?.plans?.id,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = AdPlanUtil.getAdMaterial(maxEcpmObject?.plans)?.id,
            adType = getAdType(),
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(maxEcpmObject?.plans),
            psId = AdPlanUtil.getPsId(maxEcpmObject?.plans),
            psPackageName = AdPlanUtil.getPsPackageName(maxEcpmObject?.plans),
            bidEcpmCent = maxEcpmObject?.plans?.bidEcpmCent,
            ecpmCent = maxEcpmObject?.plans?.ecpmCent,
            sceneSubId = getSceneSubId()
        )
        // 点击事件统一处理
        AdClickManager.onBiddingAdClick(
            adPlan = maxEcpmObject?.plans,
            overridePendingTransition = getAdOverridePendingTransition(),
            logTag = getLogTag(),
            sceneId = getSceneId()
        )
        // PS Offer 处理
        PsDbManager.onBiddingBuyOutClick(maxEcpmObject?.plans)
        onBiddingWrapperAdClick(maxEcpmObject)
    }

    override fun onBiddingBuyOutRewarded(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingBuyOutRewarded(maxEcpmObject)
        onBiddingWrapperAdRewarded(maxEcpmObject)
    }

    override fun onBiddingBuyOutDisplayTimestamp(
        maxEcpmObject: BiddingIntermediateMaterialBean?,
        displayTimestamp: Long,
    ) {
        super.onBiddingBuyOutDisplayTimestamp(maxEcpmObject, displayTimestamp)
        onLog(
            msg = "onBiddingBuyOutDisplayTimestamp() --> name = ${maxEcpmObject?.plans?.name} --> displayTimestamp = $displayTimestamp",
            writeToFile = false
        )
        AdReportProvider.adShowTime(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = maxEcpmObject?.plans?.id,
            displayTime = displayTimestamp,
            adId = AdPlanUtil.getAdMaterial(maxEcpmObject?.plans)?.id,
            adType = getAdType(),
            isAdShowFinal = MbAdShowLevel.isAdShowLevel(maxEcpmObject?.plans),
            sceneSubId = getSceneSubId()
        )
        mUIHandler.post {
            getListener()?.onBiddingBuyOutDisplayTimestamp(maxEcpmObject, displayTimestamp)
        }
    }

    override fun onBiddingBuyOutClose(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingBuyOutClose(maxEcpmObject)
        onBiddingWrapperAdClose(maxEcpmObject)
    }

    /**
     * Wrapper 层通知
     */
    override fun onBiddingWrapperAdDisplay(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        //super.onBiddingWrapperAdDisplay(adSource)
        onLog(
            msg = "onBiddingWrapperAdDisplay() --> name = ${maxEcpmObject?.plans?.name}",
            writeToFile = false
        )

        // 现在所有的广告都走计划，所以广告展示需要将对应的计划展示数自增
        // 统一处理 广告计划展示数自增
        NonAdShowedTimesManager.saveShowedTimes(maxEcpmObject?.plans)

        // 广告展示时间保存
        DisplayIntervalTimeManager.saveShowTime(getSceneId())

        // 回调给媒体
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdDisplay(maxEcpmObject)
        }
    }

    override fun onBiddingWrapperAdShowError(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        //super.onBiddingWrapperAdShowError(adSource)
        onLog(msg = "onBiddingWrapperAdShowError() --> name = ${maxEcpmObject?.plans?.name}")
        // 重置数据
        if (InterceptUtil.isInterceptAd(getAdType())) {
            InterceptUtil.onAdClose()
        }
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdShowError(maxEcpmObject)
        }
    }

    override fun onBiddingWrapperAdClick(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        //super.onBiddingWrapperAdClick(adSource)
        onLog(
            msg = "onBiddingWrapperAdClick() --> name = ${maxEcpmObject?.plans?.name}",
            writeToFile = false
        )
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdClick(maxEcpmObject)
        }
    }

    override fun onBiddingWrapperAdRewarded(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        //super.onBiddingWrapperAdRewarded(adSource)
        onLog(
            msg = "onBiddingWrapperAdRewarded() --> name = ${maxEcpmObject?.plans?.name}",
            writeToFile = false
        )
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdRewarded(maxEcpmObject)
        }
    }

    override fun onBiddingWrapperAdClose(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        //super.onBiddingWrapperAdClose(adSource)
        onLog(
            msg = "onBiddingWrapperAdClose() --> name = ${maxEcpmObject?.plans?.name}",
            writeToFile = false
        )
        if (InterceptUtil.isInterceptAd(getAdType())) {
            InterceptUtil.onAdClose()
        }
        mUIHandler.post {
            getListener()?.onBiddingWrapperAdClose(maxEcpmObject)
        }
    }


    // ========================================== 加载广告 ===========================================
    // ========================================== 加载广告 ===========================================
    // ========================================== 加载广告 ===========================================


    /**
     * 通用API 加载广告
     */
    suspend fun loadAd(fetchCount: Int = 1, onlyHi: Boolean = false) {
        mFetchCount = fetchCount

        // 只有Native类型才可以一次加载多个广告
        if (fetchCount > 1 && getAdType() != MbAdType.MB_AD_TYPE_NATIVE) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "只有Native类型才可以一次加载多个广告"
                )
            )
        }

        // 请求数量判断
        if (fetchCount < 1) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "fetchCount can not be less than 1"
                )
            )
        }

        // 新用户判断
        if (NewUserShieldStrategy.isNewUser()) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "新用户保护期，不展示广告"
                )
            )
        }

        if (isLoading.get()) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING,
                    "isLoading -- Try again when it's a little thicker"
                )
            )
        }
        // 加载广告
        isLoading.set(true)

        // 测试广告加载失败
        if (TestConfig.isGlobalAdOff()) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "客户端 全局 关闭广告"
                )
            )
        }

        // 如果当前是开屏广告、插屏广告、激励视频广告才需要判断拦截
        if (InterceptUtil.isInterceptAdShowed() && InterceptUtil.isInterceptAd(getAdType())) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "activity ad is showing"
                )
            )
        }

        // 广告触发 统一入口处理
        // 从这个版本开始 场景触发将无法区分广告源
        triggerId = RandomUtils.getTriggerId()
        AdReportProvider.trigger(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adType = getAdType(),
            adSource = MbAdSource.MB_AD_SOURCE_BIDDING, // 特殊标记
            planId = "",
            sceneSubId = getSceneSubId()
        )

        // 场景关闭了直接返回
        val errorMsg = SceneOnOff.isSceneOffV2(getSceneId())
        if (TextUtils.isEmpty(errorMsg).not()) {
            AdReportProvider.reject(
                triggerId = triggerId,
                sceneId = getSceneId(),
                adType = getAdType(),
                adSource = MbAdSource.MB_AD_SOURCE_BIDDING,
                rejectMsg = errorMsg,
                sceneSubId = getSceneSubId()
            )
            return onBiddingError(TAdErrorCode(MbAdSource.MB_AD_SOURCE_BIDDING, errorMsg))
        }

        // 场景展示间隔内，不展示广告
        if (DisplayIntervalTimeManager.canShow(getSceneId()).not()) {
            return onBiddingError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_BIDDING, "场景展示间隔内，不展示广告"
                )
            )
        }

        // 是否需要检查 context
        if (checkContextNonNull()) {
            if (getContext() == null) {
                return onBiddingError(
                    TAdErrorCode(
                        MbAdSource.MB_AD_SOURCE_BIDDING, "context is null"
                    )
                )
            }
        }

        // 耗时任务在子线程中完成
        withContext(Dispatchers.IO) {
            // 数据库获取所有可用计划
            val planList = AdPlansStorageManager.getAdPlan(
                sceneId = getSceneId(), ctxMap = getCtxMap(), logTag = getLogTag()
            )
            // 没有可用计划直接返回失败
            if (planList.isEmpty()) {
                return@withContext onBiddingError(
                    TAdErrorCode(
                        MbAdSource.MB_AD_SOURCE_BIDDING, "there are currently no plans available"
                    )
                )
            }

            // 找出所有符合条件的广告计划
            val pair = getBiddingPlan(planList = planList, fetchCount = fetchCount, onlyHi = onlyHi)
            biddingPlanList = pair.first
            val hasHiAdPlan = pair.second

            // 输出一下获取的广告计划数量
            onLog(
                level = Log.DEBUG,
                msg = "loadAd() --> 开始竞价 --> sceneId = ${getSceneId()} --> ctxMap = ${getCtxMap()} --> planList?.size = ${planList.size} --> hasHiAdPlan = $hasHiAdPlan --> biddingPlanList.size = ${biddingPlanList?.size}",
                writeToFile = false
            )

            // 如果仅有包断没有Hi程序化，那就不需要竞价时间
            if (hasHiAdPlan) {
                mBiddingHandler.postDelayed({
                    onBiddingTime(fetchCount = fetchCount)
                }, getBiddingTime())
            } else {
                onBiddingTime(fetchCount = fetchCount)
            }
        }
    }

    /**
     * 获取竞价时间
     *
     * 目前仅有Banner需要竞价时间
     */
    private fun getBiddingTime(): Long {
        // 只有Banner需要竞价时间，其他的广告类型都是同步获取
        return if (getAdType() == MbAdType.MB_AD_TYPE_BANNER || getAdType() == MbAdType.MB_AD_TYPE_SPLASH) {
            SceneCommonConfig.getBiddingTime(getSceneId()) * 1000L
        } else {
            0
        }
    }

    /**
     * 竞价时间到 需要调用此方法
     */
    private fun onBiddingTime(fetchCount: Int) {
        if (isLoading.get().not()) {
            return
        }

        val biddingPlanList = getBiddingPlanList()

        when (fetchCount) {
            1 -> {
                // 选出ECPM最高的那个广告计划返回
                val maxEcpmObject = getMaxEcpmBiddingPlan(biddingPlanList)
                if (maxEcpmObject == null) {
                    onBiddingError(
                        TAdErrorCode(
                            MbAdSource.MB_AD_SOURCE_BIDDING,
                            "竞价失败 --> maxEcpmObject == null --> biddingPlanList?.size = ${biddingPlanList?.size}"
                        )
                    )
                } else {
                    // 通知场景加载成功
                    onBiddingLoad(maxEcpmObject)
                }
            }

            else -> {
                val maxEcpmBiddingPlanList = getMaxEcpmBiddingPlanList(fetchCount, biddingPlanList)
                if (maxEcpmBiddingPlanList.isEmpty()) {
                    onBiddingError(
                        TAdErrorCode(
                            MbAdSource.MB_AD_SOURCE_BIDDING,
                            "竞价失败 --> maxEcpmBiddingPlanList == null --> biddingPlanList?.size = ${biddingPlanList?.size}"
                        )
                    )
                } else {
                    // 通知场景加载成功
                    onBiddingLoad(maxEcpmBiddingPlanList)
                }
            }
        }
    }

    fun startBidding(){
        mBiddingHandler.removeCallbacksAndMessages(null)
        onBiddingTime(mFetchCount)
    }

    /**
     * 获取最大值
     * 根据比较规则，ecpm 优先级更高，其次是 plans?.sort 的排序值
     *
     * compareBy：优先按照 ecpm 字段进行比较。
     * 如果 ecpm 为 null，默认会将其视为较小的值。
     *
     * thenBy：在 ecpm 相等的情况下，按照 plans?.sort 字段进行比较。
     * 如果 plans 为 null 或 sort 为 null，则默认取 Int.MAX_VALUE。
     *
     * maxWithOrNull：根据自定义的比较器，返回 biddingPlanList 中的最大对象。如果集合为空，返回 null。
     */
    private fun getMaxEcpmBiddingPlan(biddingList: MutableList<BiddingIntermediateMaterialBean>?): BiddingIntermediateMaterialBean? {
        if (biddingList.isNullOrEmpty()) {
            onLog(
                Log.WARN,
                "getMaxEcpmBiddingPlan() --> biddingList.isNullOrEmpty() == true",
                writeToFile = false
            )
            return null
        }

        // 找到最大的对象
        //
        // 如果 ecpm 和 sort 都相等，比较器认为这些对象是“等价”的。
        // 在 biddingList 中，如果存在多个这样的对象，maxWithOrNull() 会返回它们中最后出现的那个对象（也就是在原始列表中靠后的那一个）。
        //
        // index	ecpm	sort
        //  0	    10	    5
        //  1	    10	    5
        //  2	    10	    5
        // 那么 maxWithOrNull() 返回的是 index 为 2 的那个对象。
        return biddingList.maxWithOrNull(compareBy<BiddingIntermediateMaterialBean> { it.ecpm }.thenBy {
            it.plans?.sort ?: Int.MAX_VALUE
        })
    }

    /**
     * 获取 eCPM 最大的前 fetchCount 个广告素材
     *
     * 排序规则：
     * 1. 按 ecpm 从大到小 (null 当作最小值处理)
     * 2. 如果 ecpm 一样，则按 plans.sort 从小到大 (null 当作最大值处理，排后面)
     *
     * @param fetchCount 需要返回的个数
     * @param biddingList 待排序的素材列表
     * @return 返回排序后取前 fetchCount 个元素的可变列表
     */
    private fun getMaxEcpmBiddingPlanList(
        fetchCount: Int,
        biddingList: MutableList<BiddingIntermediateMaterialBean>?,
    ): MutableList<BiddingIntermediateMaterialBean> {
        // 判空处理：列表为空或 fetchCount <= 0 直接返回空列表
        if (biddingList.isNullOrEmpty() || fetchCount <= 0) {
            return mutableListOf()
        }

        return biddingList.sortedWith(
            // 第一排序条件：ecpm 降序，大的排前面；null 视为 Double.MIN_VALUE（无限小）
            compareByDescending<BiddingIntermediateMaterialBean> { it.ecpm ?: Double.MIN_VALUE }
                // 第二排序条件：sort 升序，小的排前面；null 视为 Int.MAX_VALUE（无限大）
                .thenBy { it.plans?.sort ?: Int.MAX_VALUE })
            // 取前 fetchCount 个
            // 即使 fetchCount 比列表元素个数大，也不会报错，只会返回列表中现有的所有元素。
            // 这样是安全的，不需要额外判断。
            .take(fetchCount)
            // 转换成可变列表返回
            .toMutableList()
    }


    // ======================================= 内部逻辑 ==============================================


    /*** 是否使用Hi ecpm*/
    protected fun isUseHiEcpm(adPlans: AdPlans?): Boolean {
        return adPlans?.bidEcpmCent == null || (adPlans.bidEcpmCent ?: 0.0) == 0.0
    }

    /**
     * 获取竞价计划集合
     */
    private suspend fun getBiddingPlan(
        planList: MutableList<AdPlans>,
        fetchCount: Int,
        onlyHi: Boolean,
    ): Pair<MutableList<BiddingIntermediateMaterialBean>, Boolean> {
        val biddingPlan = mutableListOf<BiddingIntermediateMaterialBean>()
        // 获取广告计划对应的素材资源
        var hasHiAdPlan = false

        val filteredPlanList = mutableListOf<AdPlans>()
        // 当前最优的HI虚拟计划，用于比较ecpm
        var currentHiPlan: AdPlans? = null

        planList.forEach { adPlan ->
            if (adPlan.adSource == AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_HI.value) {
                // Hi虚拟计划取bidEcpmCent最高的
                currentHiPlan?.let { curHiPlan ->
                    val curHiEcpm = curHiPlan.bidEcpmCent ?: 0.0
                    val newHiEcpm = adPlan.bidEcpmCent ?: 0.0
                    if (newHiEcpm > curHiEcpm) {
                        filteredPlanList.remove(curHiPlan)
                        filteredPlanList.add(adPlan)
                        currentHiPlan = adPlan
                        onLog(
                            level = Log.DEBUG,
                            msg = "替换高价HI虚拟计划: 旧计划=${curHiPlan.id} - $curHiEcpm, 新计划=${adPlan.id} - $newHiEcpm",
                            writeToFile = false
                        )
                    } else {
                        onLog(
                            level = Log.DEBUG,
                            msg = "跳过低价HI虚拟计划: ${adPlan.id} - $newHiEcpm",
                            writeToFile = false
                        )
                    }
                } ?: run {
                    filteredPlanList.add(adPlan)
                    currentHiPlan = adPlan
                }
            } else if (!onlyHi) {
                filteredPlanList.add(adPlan)
            }
        }

        filteredPlanList.forEach {
            when (it.adSource) {
                AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_HI.value -> {
                    // 需要判断开关是否打开
                    if (SceneOnOff.isSceneHiOff(getSceneId()).not()) {
                        // 任意一个ID不为空，加载HI程序化的广告
                        if (TextUtils.isEmpty(SceneCommonConfig.getHiSavanaPlacementId(getSceneId()))
                                .not() || TextUtils.isEmpty(
                                SceneCommonConfig.getHiSspPlacementId(getSceneId())
                            ).not()
                        ) {
                            // 添加HiSavana 广告源
                            addHiSavanaProvider(biddingPlan, it, fetchCount)
                            hasHiAdPlan = true
                        } else {
                            onLog(
                                level = Log.WARN,
                                msg = "hiSavanaPlacementId is empty",
                                writeToFile = false
                            )
                        }
                    } else {
                        onLog(level = Log.WARN, msg = "程序化广告场景关闭", writeToFile = false)
                    }
                }

                AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS.value -> {
                    if (SceneOnOff.isSceneNonOff(getSceneId()).not()) { // 场景没有关闭
                        // PS Offer 获取成功 添加广告计划
                        if (PsOfferProvider.getPsAdPlans(it)) {
                            biddingPlan.add(
                                BiddingIntermediateMaterialBean(
                                    ecpm = it.bidEcpmCent,
                                    plans = it,
                                    nativeInfo = null,
                                    isExpend = false
                                )
                            )
                        } else {
                            onLog(level = Log.WARN, msg = "PS Offer 获取失败", writeToFile = false)
                        }
                    } else {
                        onLog(level = Log.WARN, msg = "包断广告场景关闭", writeToFile = false)
                    }
                }

                else -> {
                    if (SceneOnOff.isSceneNonOff(getSceneId()).not()) {
                        // 包断广告默认所有的素材都是可用的
                        biddingPlan.add(
                            BiddingIntermediateMaterialBean(
                                ecpm = it.bidEcpmCent,
                                plans = it,
                                nativeInfo = null,
                                isExpend = false
                            )
                        )
                    } else {
                        onLog(level = Log.WARN, msg = "包断广告场景关闭", writeToFile = false)
                    }
                }
            }
        }
        return Pair(biddingPlan, hasHiAdPlan)
    }
}