package com.transsion.ad.monopoly.manager

import android.text.TextUtils
import android.util.Log
import com.blankj.utilcode.util.Utils
import com.transsion.ad.config.TestConfig
import com.transsion.ad.db.MbAdDatabase
import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.MbAdShowLevel
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.plan.AdPlansTransform
import com.transsion.ad.monopoly.intercept.AdPlanIntercept
import com.transsion.ad.monopoly.plan.AdPlanSourceManager

/**
 * @author: zhangxinbing
 * @date : 2024/7/29 14:12
 * @description: 广告计划数据库操作
 */
internal object AdPlansStorageManager {


    private val adPlansDao by lazy {
        MbAdDatabase.getInstance(Utils.getApp()).mbAdPlansDao()
    }


    // =============================================================================================


    /**
     * 获取广告计划
     */
    private suspend fun getAdShowFirstPlans(): List<MbAdDbPlans>? {
        return adPlansDao.getAdShowFirstPlans()
    }

    /**
     * 获取兜底广告计划
     */
    private suspend fun getAdShowFinalPlans(): List<MbAdDbPlans>? {
        return adPlansDao.getAdShowFinalPlans()
    }

    /**
     * 获取所有广告计划
     */
    suspend fun getAllAdPlans(): List<MbAdDbPlans>? {
        return adPlansDao.getAllAdPlans()
    }

    /**
     * 通过广告计划ID获取广告计划
     */
    suspend fun getPlanById(id: String?): MbAdDbPlans? {
        return adPlansDao.getPlanById(id)
    }

    /**
     * 通过广告计划源获取广告计划
     */
    suspend fun getPlansBySource(adSource: String?): List<MbAdDbPlans>? {
        return adPlansDao.getPlansBySource(adSource)
    }

    /**
     * 删除无效信息
     */
    suspend fun deleteExpireVersion(newVersion: String) {
        adPlansDao.deleteExpireVersion(newVersion = newVersion)
    }

    /**
     * 插入一条广告计划
     */
    suspend fun insert(mbAdDbPlans: MbAdDbPlans) {
        adPlansDao.insert(mbAdDbPlans)
    }

    /**
     * 批量插入广告计划
     */
    suspend fun insert(mbAdDbPlans: List<MbAdDbPlans>) {
        adPlansDao.insert(mbAdDbPlans)
    }

    /**
     * 修改数据的有效性
     */
    suspend fun updateAllValid(isValid: Boolean, newVersion: String) {
        adPlansDao.updateAllValid(isValid = isValid, newVersion = newVersion)
    }


    // ============================= 符合条件的广告计划获取 ============================================


    /**
     * 获取满足条件的广告计划
     */
    suspend fun getAdPlans(
        sceneId: String,
        @MbAdShowLevel adShowLevel: String = MbAdShowLevel.MB_AD_SHOW_FIRST,
        ctxMap: Map<String, Any>
    ): AdPlans? {
        // 从数据库获取所有的广告计划
        val allDbPlans = if (TextUtils.equals(adShowLevel, MbAdShowLevel.MB_AD_SHOW_FIRST)) {
            getAdShowFirstPlans()
        } else {
            getAdShowFinalPlans()
        }
        //AdLogger.logSdk("${javaClass.simpleName} --> getAdPlans() --> 从数据库获取所有的广告计划 -- adShowLevel = $adShowLevel --> allAdPlans?.size = ${allDbPlans?.size}")

        // TODO 模拟传递进来的参数
        //val mapOriginal = mutableMapOf<String, String>()
        //mapOriginal["genre"] = "action"

        // 遍历广告计划找到当前最符合条件展示的一个，并进行返回
        allDbPlans?.forEach { dbPlans ->
            // 判断当前广告计划是否命中
            if (AdPlanIntercept.isHit(sceneId, dbPlans, ctxMap).isHit) {
                return AdPlansTransform.transformPlanDb2Bean(dbPlans)
            }
        }
        //AdLogger.logSdkE("${javaClass.simpleName} --> getAdPlans() --> sceneId = $sceneId --> 没有获取到可用广告计划 --> adShowLevel = $adShowLevel --> allAdPlans?.size = ${allDbPlans?.size}")
        return null
    }

    /**
     * 获取符合场景展示的广告计划
     */
    suspend fun getAdPlanBySceneId(
        sceneId: String, ctxMap: Map<String, Any>
    ): MutableList<AdPlans> {
        // 获取所有的广告计划
        val values = mutableListOf<AdPlans>()
        // 将符合条件的广告计划找出来
        getAllAdPlans()?.forEach { dbPlans ->
            // 判断当前广告计划是否命中
            val res = AdPlanIntercept.isHit(sceneId, dbPlans, ctxMap)
            if (res.isHit) {
                val bean = AdPlansTransform.transformPlanDb2Bean(dbPlans)
                if (bean != null) {
                    values.add(bean)
                }
            } else {
                //AdLogger.logSdkE("AdPlansStorageManager --> sceneId = $sceneId --> rejectMsg = ${res.rejectMsg} --> dbPlans.name = ${dbPlans.name}")
            }
        }
        return values
    }

    /**
     * 获取符合场景的包断广告计划
     * 获取指定类型的广告计划
     */
    suspend fun getAdPlan(
        sceneId: String, // 场景ID
        ctxMap: Map<String, Any>, // 上下文
        logTag: String = "", // 广告链路日志 tag
        vararg adPlanEnum: AdPlanSourceManager.AdPlanEnum?
    ): MutableList<AdPlans> {
        // 提取广告计划枚举的值列表
        val adPlanEnumStr =
            if (adPlanEnum.isEmpty()) null else adPlanEnum.mapNotNull { it?.value }.toSet()
        // 获取符合条件的广告计划
        return getAllAdPlans()?.filter { dbPlans ->
            // 筛选符合 adSource 的广告计划
            (adPlanEnumStr == null || adPlanEnumStr.contains(dbPlans.adSource)) // 当 adPlanEnumStr 为空时，跳过 adSource 的筛选
        }?.mapNotNull { dbPlans ->
            // 判断是否命中当前广告计划
            val res = AdPlanIntercept.isHit(sceneId, dbPlans, ctxMap)
            if (res.isHit) {
                AdPlansTransform.transformPlanDb2Bean(dbPlans)
            } else {
                // 打印日志并返回 null，过滤掉未命中的广告计划
                if (TestConfig.isNonHitLog() == true) {
                    AdLogger.logSdk(
                        level = Log.WARN,
                        tag = logTag,
                        msg = "AdPlansStorageManager --> sceneId = $sceneId --> rejectMsg = ${res.rejectMsg} --> dbPlans.name = ${dbPlans.name}"
                    )
                }
                null
            }
        }?.toMutableList() ?: mutableListOf() // 如果为空返回空列表
    }

}