package com.transsion.ad.ps

import android.text.TextUtils
import com.blankj.utilcode.util.Utils
import com.transsion.ad.db.MbAdDatabase
import com.transsion.ad.db.pslink.PsLinkAdPlan
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.monopoly.plan.AdPlanUtil
import com.transsion.ad.util.TimeUtil
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * @author shmizhangxinbing
 * @date : 2025/8/15 17:50
 * @description:
 */
internal object PsDbManager {

    private val psLinkAdPlanDao by lazy {
        MbAdDatabase.getInstance(Utils.getApp()).psLinkAdPlanDao()
    }

    private fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    /**
     * 获取数据库中的Offer数量
     */
    suspend fun getCountByExtAdSlot(extAdSlot: String?): Int {
        return psLinkAdPlanDao.getCountByExtAdSlot(extAdSlot.toString()) // 获取数据库中所有记录的数量
    }

    /**
     * 根据 extAdSlot 查询 updateTimestamp 最小的记录
     */
    suspend fun getPsLinkAdBySlotWithMinTimestamp(extAdSlot: String): PsLinkAdPlan? {
        return psLinkAdPlanDao.getPsLinkAdBySlotWithMinTimestamp(extAdSlot)
    }

    /**
     * 更新整个 PsLinkAdPlan 对象
     */
    suspend fun updatePsLinkAdPlan(psAdPlan: PsLinkAdPlan?) {
        psAdPlan?.let {
            psLinkAdPlanDao.updatePsLinkAdPlan(it)
        }
    }

    /**
     * 插入数据库
     */
    suspend fun insertPsLinkAd(item: PsLinkAdPlan) {
        psLinkAdPlanDao.insertPsLinkAd(item)
    }

    /**
     * 删除数据
     */
    suspend fun deleteNotInIds(ids: List<Int>) {
        psLinkAdPlanDao.deleteNotInIds(ids)
    }

    /**
     * 通过ID获取PsLinkAdPlan
     */
    suspend fun getPsLinkAdById(id: Int): PsLinkAdPlan? {
        return psLinkAdPlanDao.getPsLinkAdById(id)
    }

    /**
     * 删除同一extAdSlot下的所有数据
     */
    suspend fun deleteExpiredPlansBySlot(extAdSlot: String) {
        psLinkAdPlanDao.deleteExpiredPlansBySlot(extAdSlot)
    }

    suspend fun getAllPsLink(): List<PsLinkAdPlan>? {
        return psLinkAdPlanDao.getAllPsLink()
    }


    // =============================================================================================


    /**
     * PS Offer 曝光
     */
    fun onBiddingBuyOutDisplay(plans: AdPlans?) {
        if (AdPlanSourceManager.isPsAdPlan(plans)) {
            val psLinkAdPlan = AdPlanUtil.getAdMaterial(plans)?.psLinkAdPlan ?: return

            CoroutineScope(Dispatchers.IO).launch {
                getPsLinkAdById(psLinkAdPlan.id)?.let { localPsLinkAdPlan ->
                    // 获取今天的日期
                    val currentDate = TimeUtil.getCurrentDate()
                    if (TextUtils.equals(currentDate, localPsLinkAdPlan.showDate)) {
                        localPsLinkAdPlan.showedTimes += 1
                    } else {
                        localPsLinkAdPlan.showedTimes = 1
                        localPsLinkAdPlan.showDate = currentDate
                    }

                    // 刷新数据
                    AdLogger.logPS(
                        "${getClassTag()} --> onBiddingBuyOutDisplay() " +
                                "--> id = ${localPsLinkAdPlan.id} " +
                                "--> showMax = ${localPsLinkAdPlan.showMax} " +
                                "--> showHours = ${localPsLinkAdPlan.showHours} " +
                                "--> showedTimes = ${localPsLinkAdPlan.showedTimes} " +
                                "--> showDate = ${localPsLinkAdPlan.showDate} " +
                                "--> currentDate = $currentDate"
                    )
                    insertPsLinkAd(localPsLinkAdPlan)
                }
            }
        }
    }

    /**
     * PS Offer 点击
     */
    fun onBiddingBuyOutClick(plans: AdPlans?) {
        if (AdPlanSourceManager.isPsAdPlan(plans)) {
            val psLinkAdPlan = AdPlanUtil.getAdMaterial(plans)?.psLinkAdPlan ?: return

            CoroutineScope(Dispatchers.IO).launch {
                getPsLinkAdById(psLinkAdPlan.id)?.let { localPsLinkAdPlan ->
                    // 获取今天的日期
                    val currentDate = TimeUtil.getCurrentDate()
                    if (TextUtils.equals(currentDate, localPsLinkAdPlan.showDate)) {
                        localPsLinkAdPlan.clickedTimes += 1
                    } else {
                        localPsLinkAdPlan.clickedTimes = 1
                        localPsLinkAdPlan.showDate = currentDate
                    }

                    // 刷新数据
                    AdLogger.logPS(
                        "${getClassTag()} --> onBiddingBuyOutClick() " +
                                "--> id = ${localPsLinkAdPlan.id} " +
                                "--> clickMax = ${localPsLinkAdPlan.clickMax} " +
                                "--> clickedTimes = ${localPsLinkAdPlan.clickedTimes} " +
                                "--> showDate = ${localPsLinkAdPlan.showDate} "
                    )
                    insertPsLinkAd(localPsLinkAdPlan)
                }
            }
        }
    }

}