package com.transsion.ad.web.cct

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.text.TextUtils
import androidx.browser.customtabs.CustomTabsIntent
import androidx.core.net.toUri

/**
 * CustomTabsHelper
 *
 * 这个工具类用于在 Android Kotlin 项目中方便地打开 Chrome Custom Tabs，
 * 支持两种模式：
 *  1. PRIORITY_CHROME   优先使用 Chrome（或其 Beta/Dev/Canary 版本），若没有安装则退回到默认浏览器的 Custom Tabs。
 *  2. DEFAULT_BROWSER   使用系统默认浏览器的 Custom Tabs（Google 官方示例逻辑）。
 *
 * 同时提供 fallback 接口，当没有任何支持 Custom Tabs 的浏览器时，可回退到 WebView 打开网页。
 *
 * 使用场景：
 *  - 广告 SDK 里打开落地页
 *  - 应用内浏览网页但不想直接跳出应用
 *
 * 注意事项：
 *  - Custom Tabs 是一个协议，非 Chrome 独有，其他浏览器（如 Firefox、Edge 等）也可能支持。
 *  - 在 Android 11+，需要保证通过 PackageManager 正确查询服务。
 */
class CustomTabsHelper {

    companion object {
        /** Custom Tabs 的服务协议 Action 常量 */
        private const val ACTION_CUSTOM_TABS_CONNECTION =
            "android.support.customtabs.action.CustomTabsService"

        /** 缓存选中的 Custom Tabs 提供者包名，避免重复查询 */
        private var sPackageNameToUse: String? = null

        /**
         * 打开模式枚举：
         *  - PRIORITY_CHROME  优先 Chrome（没有则用默认浏览器）
         *  - DEFAULT_BROWSER  使用默认浏览器逻辑
         */
        enum class Mode {
            PRIORITY_CHROME, DEFAULT_BROWSER
        }

        /**
         * 打开 Custom Tab
         *
         * @param activity         宿主 Activity
         * @param customTabsIntent 已构建好的 CustomTabsIntent
         * @param uri              要打开的 Uri
         * @param fallback         没有可用 Custom Tabs 时的回退接口
         * @param mode             打开模式（优先 Chrome 或默认浏览器）
         */
        fun openCustomTab(
            activity: Activity,
            customTabsIntent: CustomTabsIntent,
            uri: Uri,
            fallback: CustomTabFallback?,
            mode: Mode = Mode.PRIORITY_CHROME,
        ) {
            // 根据模式选择包名
            val packageName = when (mode) {
                Mode.PRIORITY_CHROME -> getPreferredCustomTabPackage(activity)
                Mode.DEFAULT_BROWSER -> getPackageNameToUse(activity)
            }

            if (packageName == null) {
                // 没有可用 Custom Tabs -> WebView fallback
                fallback?.openUri(activity, uri)
            } else {
                // 指定 Custom Tabs 提供者包名
                customTabsIntent.intent.setPackage(packageName)
                customTabsIntent.launchUrl(activity, uri)
            }
        }

        /**
         * 模式 1：优先选择 Chrome（正式版），若无则尝试 Beta/Dev/Canary，最后回退到默认浏览器
         */
        fun getPreferredCustomTabPackage(context: Context): String? {
            val pm = context.packageManager
            val serviceIntent = Intent(ACTION_CUSTOM_TABS_CONNECTION)

            // 优先 Chrome 正式版
            serviceIntent.setPackage("com.android.chrome")
            if (pm.resolveService(serviceIntent, 0) != null) {
                return "com.android.chrome"
            }

            // 再尝试其他 Chrome 系列版本
            val chromePackages = listOf(
                "com.chrome.beta", "com.chrome.dev", "com.chrome.canary"
            )
            for (pkg in chromePackages) {
                serviceIntent.setPackage(pkg)
                if (pm.resolveService(serviceIntent, 0) != null) {
                    return pkg
                }
            }

            // 如果都没有，使用默认浏览器逻辑
            return getPackageNameToUse(context)
        }

        /**
         * 模式 2：Google 官方默认浏览器逻辑
         *
         * 如果默认浏览器支持 Custom Tabs，则使用它；
         * 否则，从支持 Custom Tabs 的列表中选择一个。
         */
        @SuppressLint("QueryPermissionsNeeded")
        fun getPackageNameToUse(context: Context): String? {
            if (sPackageNameToUse != null) return sPackageNameToUse

            val pm = context.packageManager
            val activityIntent = Intent(Intent.ACTION_VIEW, "http://www.example.com".toUri())

            // 获取默认浏览器信息
            val defaultViewHandlerInfo = pm.resolveActivity(activityIntent, 0)
            var defaultViewHandlerPackageName: String? = null
            if (defaultViewHandlerInfo != null) {
                defaultViewHandlerPackageName = defaultViewHandlerInfo.activityInfo.packageName
            }

            // 获取所有支持 VIEW 的应用，并检查是否支持 Custom Tabs Service
            val resolvedActivityList = pm.queryIntentActivities(activityIntent, 0)
            val packagesSupportingCustomTabs: MutableList<String?> = ArrayList()
            for (info in resolvedActivityList) {
                val serviceIntent = Intent()
                serviceIntent.action = ACTION_CUSTOM_TABS_CONNECTION
                serviceIntent.setPackage(info.activityInfo.packageName)
                if (pm.resolveService(serviceIntent, 0) != null) {
                    packagesSupportingCustomTabs.add(info.activityInfo.packageName)
                }
            }

            // 根据官方逻辑选择默认浏览器或第一个支持的浏览器
            sPackageNameToUse = if (packagesSupportingCustomTabs.isEmpty()) {
                null
            } else if (!TextUtils.isEmpty(defaultViewHandlerPackageName) && !hasSpecializedHandlerIntents(
                    context, activityIntent
                ) && packagesSupportingCustomTabs.contains(defaultViewHandlerPackageName)
            ) {
                defaultViewHandlerPackageName
            } else {
                packagesSupportingCustomTabs[0]
            }
            return sPackageNameToUse
        }

        /**
         * 检查是否存在专门处理特定 URL 的 Handler（例如某些深链接 APP）
         */
        @SuppressLint("QueryPermissionsNeeded")
        private fun hasSpecializedHandlerIntents(context: Context, intent: Intent): Boolean {
            try {
                val pm = context.packageManager
                val handlers = pm.queryIntentActivities(intent, PackageManager.GET_RESOLVED_FILTER)
                if (handlers.isEmpty()) return false
                for (resolveInfo in handlers) {
                    val filter = resolveInfo.filter ?: continue
                    if (filter.countDataAuthorities() == 0 || filter.countDataPaths() == 0) continue
                    if (resolveInfo.activityInfo == null) continue
                    return true
                }
            } catch (_: RuntimeException) {
                // 防止 PackageManager 查询异常
            }
            return false
        }
    }
}
