package com.transsion.ad.bidding.nativead

import android.util.Log
import com.cloud.hisavana.sdk.api.adx.TNative
import com.cloud.hisavana.sdk.api.listener.AdListener
import com.cloud.hisavana.sdk.common.bean.TaNativeInfo
import com.cloud.hisavana.sdk.common.constant.TaErrorCode
import com.transsion.ad.hi.HiSavanaAdManager
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.strategy.AdContextManager.getCtxMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.concurrent.ConcurrentLinkedDeque
import java.util.concurrent.ConcurrentSkipListSet
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/9/22 14:06
 * @description: Hi 程序化SSP 广告加载
 */
class BiddingHiSspNativeProvider : AdListener() {

    private fun getClassTag(): String = javaClass.simpleName
    private var isLoading: AtomicBoolean = AtomicBoolean(false) // 当前正在请求中
    private var tNativeAd: TNative? = null // 广告加载对象 负责广告的加载
    private var mPlacementId: String? = null // 场景调用的时候传进来的场景ID
    private val taskList = ConcurrentLinkedDeque<AdListener>() // 任务队列 --> 添加、移除、遍历

    // 定义一个 ConcurrentSkipListSet，使用自定义 Comparator 按 ecpm 降序排序
    // 是一个线程安全的集合，属于 Java 的 java.util.concurrent 包，
    // 底层使用**跳表（SkipList）**作为数据结构来存储元素。
    // 它既保证了线程安全，又能确保元素始终按顺序排列。
    // 每次添加数据时都会自动按照指定的排序规则进行排序。
    private val nativeInfoList = ConcurrentSkipListSet<TaNativeInfo>(Comparator { ad1, ad2 ->
        // 判断是否为重复广告（adCreateId或imgUrl任一相同）
        val isAdCreateIdSame = ad1.adCreateId == ad2.adCreateId
        val isImgUrlSame = ad1.image?.imgUrl == ad2.image?.imgUrl
        if (isAdCreateIdSame || isImgUrlSame) {
            // 重复广告：按bidPrice降序，保留价格高的
            val bidPriceResult = ad2.bidPrice.compareTo(ad1.bidPrice)
            if (bidPriceResult != 0) return@Comparator bidPriceResult
            // 若bidPrice相同，视为完全重复（去重）
            return@Comparator 0
        }

        // 非重复广告：按bidPrice、adCreateId、imgUrl降序排序
        var result = ad2.bidPrice.compareTo(ad1.bidPrice) // 优先按bidPrice降序
        if (result != 0) return@Comparator result

        result = ad2.adCreateId.compareTo(ad1.adCreateId) // 其次adCreateId降序
        if (result != 0) return@Comparator result

        ad2.image.imgUrl.compareTo(ad1.image.imgUrl) // 最后imgUrl降序
    })

    // 缓存数量
    private var cacheLimit = 3


    // =============================================================================================


    companion object {
        /**
         * 当前Map存储所有广告加载缓存池
         */
        private val map by lazy {
            mutableMapOf<String, BiddingHiSspNativeProvider>()
        }

        /**
         * 通过广告位ID -- 获取广告池 -- 在缓存池里面获取广告
         */
        fun getNativeManager(sceneId: String?): BiddingHiSspNativeProvider? {
            val placementId =
                SceneCommonConfig.getHiSspPlacementId(sceneId = sceneId) ?: return null

            if (placementId.isEmpty()) {
                return null
            }

            var manager = map[placementId]
            if (null == manager) {
                val adNativeManager = BiddingHiSspNativeProvider()
                adNativeManager.setSspPlacementId(placementId)
                map[placementId] = adNativeManager
                manager = adNativeManager
            }
            return manager
        }
    }


    // ============================广告SDK回调========================================================


    override fun onAdLoaded(infos: List<TaNativeInfo>?) {
        isLoading.set(false)

        infos?.let {
            nativeInfoList.addAll(infos)
        }

        // 检查容量
        if (isUpperLimit().not()) {
            loadAd()
        }
    }

    override fun onError(p0: TaErrorCode?) {
        isLoading.set(false)
        AdLogger.logSdk(
            tag = AdLogger.TAG_AD_NATIVE,
            msg = "${getClassTag()} --> onError() --> placementId = ${getPlacementId()} --> errorMessage = ${p0?.errorMessage}",
            writeToFile = false,
            level = Log.ERROR
        )
    }

    override fun onNativeAdShow(p0: TaNativeInfo?) {
//        AdLogger.logSdk(
//            tag = AdLogger.TAG_AD_NATIVE,
//            msg = "${getClassTag()} --> onNativeAdShow() --> ${p0?.adCreateId}",
//            writeToFile = false,
//        )
        taskList.forEach {
            it.onNativeAdShow(p0)
        }
    }

    override fun onNativeAdClick(p0: TaNativeInfo?) {
//        AdLogger.logSdk(
//            tag = AdLogger.TAG_AD_NATIVE,
//            msg = "${getClassTag()} --> onNativeAdClick() --> ${p0?.adCreateId}",
//            writeToFile = false
//        )
        taskList.forEach {
            it.onNativeAdClick(p0)
        }
    }


    // =============================================================================================


    private fun getPlacementId(): String = mPlacementId ?: ""

    /**
     * 用于将 SSP广告的曝光和点击事件回调给AbsAdBidding统一处理
     */
    fun addListener(listener: AdListener?) {
        taskList.add(listener)
    }

    fun removeListener(listener: AdListener?) {
        taskList.remove(listener)
    }

    /*** 广告展示的时候需要使用到*/
    fun getNativeAd(): TNative? = tNativeAd

    fun setSspPlacementId(placementId: String?) {
        mPlacementId = placementId
    }

    /**
     * 将没有使用过的广告重新添加到缓存池中
     */
    fun addUnusedAdToPool(nativeInfo: TaNativeInfo?) {
        nativeInfo?.let {
            if (tNativeAd?.isReady(it) == true) {
                nativeInfoList.add(it)
            }
        }
    }


    // =============================================================================================


    /**
     * 加载广告 --> 现在仅支持串行请求
     */
    fun loadAd() {
        // 如果没有初始化，那就不进行任何操作
        if (HiSavanaAdManager.isInitialized().not()) {
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "${getClassTag()} --> loadAd() --> 广告SDK没有初始化",
                writeToFile = false
            )
            return
        }

        // 缓存池有足够的广告
        if (isUpperLimit()) {
            return
        }

        // 如果正在请求中那就return 这个isLoading 仅在SDK回调的onLoad和onError修改
        if (isLoading.get()) {
            return
        }
        isLoading.set(true)

        // 加载广告
        if (null == tNativeAd) {
            tNativeAd = TNative(getPlacementId())
            tNativeAd?.setListener(this)
            tNativeAd?.setAdCount(cacheLimit)
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "============= ${getClassTag()} --> loadAd() --> " + "mPlacementId = ${getPlacementId()} --> create TNativeAd() ============= ",
                writeToFile = false
            )
        }
        tNativeAd?.setAdCount(cacheLimit) // cacheLimit 没次加载前都设置为最新的值，防止上次设置的缓存数失效
        tNativeAd?.loadAd()
    }

    /**
     * 预加载广告
     */
    fun preLoadAd(sceneId: String, reqCount: Int) {
        CoroutineScope(Dispatchers.IO).launch {
            // 数据库获取广告计划
            val planList = AdPlansStorageManager.getAdPlan(
                sceneId = sceneId, ctxMap = getCtxMap(genre = null)
            )

            // 无广告计划不预取
            if (planList.isEmpty()) {
                return@launch
            }

            // 无Hi虚拟计划不预取
            val hasHiAdPlan = planList.any { AdPlanSourceManager.isHiAdPlan(it) }
            if (!hasHiAdPlan) {
                return@launch
            }

            if (reqCount > cacheLimit) {
                cacheLimit = reqCount
            }
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "${getClassTag()} --> preLoadAd(${reqCount}) --> ${getPlacementId()}",
                writeToFile = false
            )
            loadAd()
        }
    }

    // =============================================================================================


    /**
     * 同步获取多条广告
     */
    fun getAdsSync(reqCount: Int, inFull: Boolean = false): List<TaNativeInfo>? {
        if (reqCount > cacheLimit) {
            cacheLimit = reqCount
        }

        return kotlin.runCatching {
            // 移除过期广告
            nativeInfoList.iterator().apply {
                while (hasNext()) {
                    val info = next()
                    if (tNativeAd?.isReady(info) == false) {
                        info.destroy()
                        remove()
                    }
                }
            }

            if (inFull && nativeInfoList.size < reqCount) {
                // 不足数不填充
                return@runCatching null
            }

            val nativeInfos = mutableListOf<TaNativeInfo>()
            repeat(minOf(reqCount, nativeInfoList.size)) {
                nativeInfoList.pollFirst()?.let { nativeInfos.add(it) }
            }
            nativeInfos
        }.getOrElse {
            null
        }.also {
            // 无论获取成功还是失败都触发一次加载广告，保证缓存池有足够的广告
            loadAd()
        }
    }

    /**
     * 缓存池是否达到阈值,大于等于阈值了说明广告有富余没有消费完
     */
    private fun isUpperLimit(): Boolean {
        val upper = nativeInfoList.size >= cacheLimit
        if (upper) {
            AdLogger.logSdk(
                tag = AdLogger.TAG_AD_NATIVE,
                msg = "${getClassTag()} --> isUpperLimit() --- 达到阈值了 " + "--> nativeInfoList.size = ${nativeInfoList.size} " + "-- cacheUpperLimit = $cacheLimit",
                writeToFile = false
            )
        }
        return upper
    }
}