package com.transsion.ad.monopoly.manager

import android.text.TextUtils
import android.util.Log
import com.cloud.hisavana.sdk.common.util.MD5Utils
import com.tn.lib.net.manager.NetServiceGenerator
import com.transsion.ad.MbAdContents
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.AdMaterialList
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.plan.AdPlansTransform
import com.transsion.ad.monopoly.model.MbAdPlansDto
import com.transsion.ad.monopoly.model.MbAdConfigApi
import com.transsion.ad.monopoly.model.MbAdImage
import com.transsion.ad.ps.PSAdPlanRequestManager
import com.transsion.ad.scene.SceneStorage
import com.transsion.ad.strategy.AdContextManager
import com.transsion.ad.strategy.AdMmkv
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.report.AdReportProvider
import com.transsion.ad.strategy.AdResDownloadManager
import com.transsion.ad.strategy.NewUserShieldStrategy
import com.transsion.ad.util.FileUtil
import com.transsion.ad.util.TimeUtil
import java.io.File
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author: zhangxinbing
 * @date : 2024/7/31 10:41
 * @description: 广告计划的请求
 */
object AdPlansRequestManager {

    private val mbAdConfigApi by lazy {
        NetServiceGenerator.instance.getService(MbAdConfigApi::class.java)
    }

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 是否正在请求
     *
     * 处理多次触发，仅保持一个完整的请求。
     */
    private var isAdConfigLoading: AtomicBoolean = AtomicBoolean(false)


    // =========================== 👇是网络请求 ======================================================


    /**
     * 网络请求非标广告计划列表
     *
     * 1. 请求非标
     * 2. 判断并保存非标计划-- 数据库保存
     * 3. 下载最新计划的非标资源
     * 4. 全部资源下载完成后 删除过期资源
     */
    suspend fun requestNonAdPlanList(
        adConfigUrl: String, callback: ((result: Boolean) -> Unit?)? = null
    ) {

        if (isAdConfigLoading.get()) {
            AdLogger.logAdInit(
                "${getClassTag()} --> requestNonAdPlanList() --> 正在请求中.... 稍后再试",
                level = Log.WARN
            )
            return
        }
        isAdConfigLoading.set(true)

        kotlin.runCatching {
            // 本地保存的广告计划版本号
            val lastVersion = AdMmkv.mmkvWithID.getString(MbAdContents.NON_AD_PLAN_VERSION, "")
            // 本地保存的场景信息版本号
            val lastSceneVersion =
                AdMmkv.mmkvWithID.getString(MbAdContents.NON_AD_SCENE_VERSION, "")
            AdLogger.logAdInit("${getClassTag()} --> requestNonAdPlanList() --> 开始请求广告配置接口 --> lastVersion = $lastVersion --> lastSceneVersion = $lastSceneVersion --> url = $adConfigUrl")

            // 发起请求
            val mbAdConfig = mbAdConfigApi.getAdConfig(
                url = adConfigUrl, version = lastVersion ?: "", versionAdScene = lastSceneVersion
            )

            // 数据解析
            val result = handleResponse(mbAdConfig, lastVersion)
            callback?.invoke(result)
            // 无论包断广告结果如何，都需要更新PS数据
            PSAdPlanRequestManager.updatePsLink()
            isAdConfigLoading.set(false)
        }.getOrElse {
            isAdConfigLoading.set(false)
            callback?.invoke(false)
            // 广告计划请求失败了不做任何处理
            AdLogger.logAdInit(
                "${getClassTag()} --> requestNonAdPlanList() --> fail fail fail --> 请求非标广告计划列表失败 it = $it",
                level = Log.ERROR
            )
        }
    }


    // ================================== 数据解析 ===================================================


    /**
     * 解析响应数据
     *
     * 版本号不变 --> 数据没有变化 --> 客户端仅处理检查资源的下载
     * 版版本号改变 --> 增量更新 --> 客户端仅处理检查资源的下载/删除过期资源
     */
    private suspend fun handleResponse(mbAdPlansDto: MbAdPlansDto?, lastVersion: String?): Boolean {
        if (mbAdPlansDto == null) {
            AdLogger.logAdInit(
                "${getClassTag()} --> handleResponse() --> mbAdConfig == null --> lastVersion = $lastVersion",
                level = Log.ERROR
            )
            return false
        }

        // 最新版本
        val newVersion = mbAdPlansDto.data?.version.toString()
        val adPlanSize = mbAdPlansDto.data?.adPlans?.size

        if (mbAdPlansDto.code != "0") {
            AdLogger.logAdInit(
                "${getClassTag()} --> handleResponse() --> mbAdConfig.code != 0 --> lastVersion = $lastVersion --> newVersion = $newVersion",
                level = Log.ERROR
            )
            return false
        }

        // 新用户屏蔽
        NewUserShieldStrategy.saveNewUserShieldData(mbAdPlansDto.data)

        // genre 这个字段和版本号一样，一定会返回的
        AdContextManager.saveGenre(mbAdPlansDto.data)

        // 场景信息保存，如果版本号不变服务端将不下发 adSceneConfigData 数据
        SceneStorage.setSceneConfig(mbAdPlansDto)

        // 版本号一致，服务端不会下发list 省流量
        if (TextUtils.equals(lastVersion, newVersion)) {
            AdLogger.logAdInit("${getClassTag()} --> handleResponse() --> 版本号一致，服务端不会下发list 省流量 --> lastVersion = $lastVersion --> newVersion = $newVersion")
            return true
        }

        // 到这里意味着客户端已经知道服务端有新的广告计划了,服务端去重
        AdReportProvider.onNewAdPlanReceive(lastVersion = lastVersion, newVersion = newVersion)

        // 增量更新
        AdLogger.logAdInit("${getClassTag()} --> handleResponse() --> lastVersion = $lastVersion --> newVersion = $newVersion --> size = $adPlanSize --> 开始增量更新")

        // 删除过期广告计划数据
        // deleteExpireAdPlan(mbAdPlansDto.data?.version.toString(), lastVersion)
        // 保证计划关闭的及时性，将不是最新版本的计划标记为不可用
        AdLogger.logAdInit("${getClassTag()} --> handleResponse() --> 保证计划关闭的及时性，将不是最新版本的计划标记为不可用 --> isValid = false")
        AdPlansStorageManager.updateAllValid(isValid = false, newVersion = newVersion)

        // 将请求到的数据保存到数据库
        saveAndDownload(mbAdPlansDto)

        // 检查结果
        checkResult(mbAdPlansDto, lastVersion)
        return true
    }


    // ================================ 素材下载及数据库保存 ===========================================


    /**
     * 将非标广告计划列表保存到数据库 下载资源
     */
    private suspend fun saveAndDownload(mbAdPlansDto: MbAdPlansDto?) {

        // 最新版本版本号
        val newVersion = mbAdPlansDto?.data?.version.toString()

        // 遍历所有的计划 --> 下载素材、更新数据
        mbAdPlansDto?.data?.adPlans?.forEach { adPlan ->
            // 查询数据库是否已经存在该广告计划
            val adPlanDb = AdPlansStorageManager.getPlanById(adPlan.id)

            // DataBean 赋值
            adPlan.version = newVersion // 版本号赋值
            adPlan.isValid = true // 可用，前置条件已经将所有计划设置为不可用了
            adPlan.ctxAttributeConfig = mbAdPlansDto.data.ctxAttributeConfig // 客户端处理数据转移

            // 广告计划存在
            if (adPlanDb != null) {
                // 将数据库对象转换成 --> DateBean
                val db2Bean = AdPlansTransform.transformPlanDb2Bean(adPlanDb)
                val materialList = db2Bean?.adMaterialList?.last() // 取出素材
                val materialSuccess = materialList?.downloadMaterialSuccess ?: false // 素材是否已经下载完成

                // DataBean --> 覆盖数据库数据
                db2Bean?.ctxAttributeConfig = adPlan.ctxAttributeConfig // 更新配置
                adPlanDb.version = adPlan.version // 更新版本号
                adPlanDb.isValid = adPlan.isValid // 更新可用状态

                // 处理跨天判断
                var showedTimes: Int? = null
                if (TextUtils.equals(adPlanDb.showDate, TimeUtil.getCurrentDate())) {
                    showedTimes = adPlanDb.showedTimes
                }
                // 上次展示日期展示次数
                adPlan.showDate = adPlanDb.showDate
                adPlan.showedTimes = showedTimes

                // 素材不存在或者计划更新时间戳有变化，都需要下载素材
                if (!materialSuccess || TextUtils.equals(
                        adPlanDb.adPlanUpdateTime, adPlan.adPlanUpdateTime
                    ).not()
                ) {
                    AdLogger.logAdInit(
                        writeToFile = false,
                        msg = "${getClassTag()} --> saveAndDownload() --> 广告计划已存在数据库 --> 内容 有有有 变化 --> 下载素材、更新数据库 --> adPlan.name = ${adPlan.name} --> adPlan.id = ${adPlan.id}"
                    )
                    downloadMaterialAndSaveAdPlans(adPlan) // 更新数据
                } else {
                    // 重新赋值 给DataBean ，用于无效资源删除
                    adPlan.adMaterialList = db2Bean?.adMaterialList // 素材
                    adPlan.advertiserAvatarPath = db2Bean?.advertiserAvatarPath // 广告主头像

                    AdLogger.logAdInit(
                        writeToFile = false,
                        msg = "${getClassTag()} --> saveAndDownload() --> 广告计划已存在数据库 --> 内容 无无无 变化 --> 将DB数据赋值给DataBean --> adPlan.name = ${adPlan.name} --> adPlan.id = ${adPlan.id}"
                    )
                    AdPlansStorageManager.insert(adPlanDb) // 更新数据库数据
                }
            } else {
                // 广告计划不存在
                AdLogger.logAdInit("${getClassTag()} --> saveAndDownload() --> 广告计划不存在数据库 --> 下载素材、保存数据库 --> adPlan.name = ${adPlan.name} --> adPlan.id = ${adPlan.id}")
                downloadMaterialAndSaveAdPlans(adPlan) // 更新数据
            }
        }
    }

    /**
     * 下载素材成功之后保存数据库
     *
     * 走到这里就是需要下载素材了
     */
    private suspend fun downloadMaterialAndSaveAdPlans(adPlans: AdPlans) {
        // PS虚拟计划和 Hi虚拟计划不需要下载素材的
        if (AdPlanSourceManager.isPsAdPlan(adPlans)
            || AdPlanSourceManager.isHiAdPlan(adPlans)
            || AdPlanSourceManager.isIconAdPlan(adPlans)) {
            AdLogger.logAdInit("${getClassTag()} --> downloadMaterialAndSaveAdPlans() --> PS虚拟计划和、Hi虚拟计划 --> 不需要下载素材的 --> ${adPlans.adSource} --> adPlan.name = ${adPlans.name} --> adPlan.id = ${adPlans.id}")
            //是ps广告计划 将资源直接设置为成功 然后保存数据库。为了能让广告大版本更新
            val adMaterial = AdMaterialList()
            adMaterial.downloadMaterialSuccess = true
            adPlans.adMaterialList = listOf(adMaterial)
            updateAdPlanToDatabase(adPlans)
        } else {
            // 广告计划素材下载完成之后保存到数据库
            //AdBiddingLogger.logAdInit("${getClassTag()} --> downloadMaterialAndSaveAdPlans() --> 发起 非标广告素材下载及数据库保存 --> adPlans.name = ${adPlans.name}")
            if (downloadAdPlan(adPlans)) {
                updateAdPlanToDatabase(adPlans)
            }
        }
    }

    /**
     * 下载非标广告计划素材
     */
    private fun downloadAdPlan(it: AdPlans?): Boolean {
        //AdBiddingLogger.logAdInit("${getClassTag()} --> downloadAdPlan() --> 下载非标广告计划素材 start --> adPlans.id = ${it?.id} ......")

        // 下载广告主头像
        // 广告主头像本地文件地址
        val advertiserAvatarPath = downloadAdvertiserAvatar(it?.advertiserAvatar)
        it?.advertiserAvatarPath = advertiserAvatarPath
        //AdBiddingLogger.logAdInit("${getClassTag()} --> --> downloadAdPlan() --> 广告主头像下载完成 --> advertiserAvatarPath = $advertiserAvatarPath")

        // 扩展图片
        downloadExtImage(it?.extImage)

        // 开始下载广告计划下面的广告素材列表 -- 目前一个计划对应一个素材
        it?.adMaterialList?.forEach { adMaterial ->
            // 获取下载资源的地址
            val url = if (adMaterial.type == AdMaterialList.NON_AD_TYPE_TEXT) {
                adMaterial.image?.url
            } else {
                adMaterial.video?.url
            }
            // 隐藏后缀 .mp4这样的
            val mineType = MbAdContents.MINE_TYPE
            val fileName = MD5Utils.toMd5(url)
            val destination =
                MbAdContents.NON_AD_VIDEO_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"
            val file = File(destination)
            //下载视频类型的
            if (adMaterial.type == AdMaterialList.NON_AD_TYPE_VIDEO) {
                val remoteFileSize = adMaterial.video?.size ?: 1
                if (file.isFile && file.exists() && file.length() > 0) {
                    //不需要重新下载 copy一次 都处理完后可删除old文件
                    adMaterial.video?.path = destination
                    adMaterial.downloadMaterialSuccess = true
                    return true
                } else {
                    val downloadFileSuccess = AdResDownloadManager.downloadFile(url, destination)
                    // size check
                    val localFileSize = file.length()
                    if (downloadFileSuccess && (localFileSize >= remoteFileSize)) {
                        //AdBiddingLogger.logAdInit("${getClassTag()} --> downloadAdPlan() --> 视频素材下载成功 --> adPlans.id = ${it.id} -- adMaterial.id = ${adMaterial.id} -- destination = $destination")
                        adMaterial.video?.path = destination
                        adMaterial.downloadMaterialSuccess = true
                        return true
                    } else {
                        // 下载失败不处理
                        AdLogger.logAdInit(
                            "${getClassTag()} --> downloadAdPlan() --> 视频素材下载失败 -- adPlans.id = ${it.id} downloadFileSuccess = $downloadFileSuccess -- localFileSize = $localFileSize -- remoteFileSize = $remoteFileSize destination = $destination",
                            level = Log.ERROR
                        )
                        return false
                    }
                }
            } else {
                //下载图类的  可能被压缩 判断条件不一致
                if (file.isFile && file.exists() && file.length() > 0) {
                    //不需要重新下载 copy一次 都处理完后可删除old文件
                    adMaterial.image?.path = destination
                    adMaterial.downloadMaterialSuccess = true
                    return true
                } else {
                    val downloadFileSuccess = AdResDownloadManager.downloadFile(url, destination)
                    // size check
                    val localFileSize = file.length()
                    if (downloadFileSuccess && localFileSize > 0) {
                        //AdBiddingLogger.logAdInit("${getClassTag()} --> downloadAdPlan() --> 图片素材下载成功 --> adPlans.id = ${it.id} -- adMaterial.id = ${adMaterial.id} -- destination = $destination")
                        adMaterial.image?.path = destination
                        adMaterial.downloadMaterialSuccess = true
                        return true
                    } else {
                        // 下载失败不处理
                        AdLogger.logAdInit(
                            "${getClassTag()} --> downloadAdPlan() --> 图片素材下载失败 -- adPlans.id = ${it.id} downloadFileSuccess = $downloadFileSuccess -- localFileSize = $localFileSize -- destination = $destination",
                            level = Log.ERROR
                        )
                        return false
                    }
                }
            }
        }
        return false
    }

    /**
     * 保存到本地数据库
     */
    private suspend fun updateAdPlanToDatabase(adPlans: AdPlans): Boolean {
        // 数据类型转换
        val transformAdPlans = AdPlansTransform.transformAdPlans2NonAdPlans(adPlans)
        AdPlansStorageManager.insert(transformAdPlans)
        // 2. 将非标广告计划列表保存到数据库
        //AdLogger.logAdInit("${getClassTag()} --> updateAdPlanToDatabase() --> 保存非标广告计划列表到数据库成功 --> adPlan.name = ${transformAdPlans.name} --> adPlan.id = ${transformAdPlans.id}")
        return true
    }

    /**
     * 下载广告主图标
     */
    private fun downloadAdvertiserAvatar(advertiserAvatar: String?): String {
        // 文件本地命名
        //val fileName = System.currentTimeMillis().toString()
        val fileName = MD5Utils.toMd5(advertiserAvatar)
        val mineType = MbAdContents.MINE_TYPE

        // 文件本地存储路径
        val destination =
            MbAdContents.NON_AD_VIDEO_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"

        // 判断一下文件是否已经存在
        if (FileUtil.isFileExists(destination)) {
            return destination
        }

        // 下载文件
        val downloadFile = AdResDownloadManager.downloadFile(advertiserAvatar, destination)
        return if (downloadFile) {
            destination
        } else {
            ""
        }
    }

    /**
     * 扩展图片下载
     */
    private fun downloadExtImage(extImage: MbAdImage?) {
        // 空判断
        if (extImage == null) {
            return
        }
        if (TextUtils.isEmpty(extImage.url)) {
            return
        }

        // 获取下载地址
        val url = extImage.url
        val mineType = MbAdContents.MINE_TYPE // 隐藏后缀 .mp4这样的
        val fileName = MD5Utils.toMd5(url)
        val destination =
            MbAdContents.NON_AD_VIDEO_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"

        // 判断一下文件是否已经存在
        if (FileUtil.isFileExists(destination)) {
            extImage.path = destination
            return
        }

        // 下载文件
        val downloadFile = AdResDownloadManager.downloadFile(url, destination)
        if (downloadFile) {
            // 下载成功，给path赋值
            extImage.path = destination
            //AdBiddingLogger.logAdInit("${getClassTag()} --> downloadExtImage() --> extImage下载成功 --> path = ${extImage.path}")
        }
    }


    // =============================================================================================


    /**
     * 价差结果
     */
    private suspend fun checkResult(mbAdPlansDto: MbAdPlansDto?, lastVersion: String?) {
        //更新完计划且下载完素材 新计划可用
        if (mbAdPlansDto?.data?.adPlans != null && checkDownloadMaterial(mbAdPlansDto.data.adPlans)) {
            // 新版本广告计划更新完成
            val newVersion = mbAdPlansDto.data.version.toString()
            AdReportProvider.onNewAdPlanUpdateSuccess(
                lastVersion = lastVersion, newVersion = newVersion
            )
            // 保存最新版本号
            AdMmkv.mmkvWithID.putString(MbAdContents.NON_AD_PLAN_VERSION, newVersion)

            // 删除过期计划 因为前面已经调用了，所以这里没有必要再次调用了
            // 重新调用
            deleteExpireAdPlan(newVersion = newVersion, lastVersion = lastVersion)

            // 删除无效资源
            deleteExpireMaterial(mbAdPlansDto.data.adPlans)
            // 获取所有可用计划
            AdLogger.logAdInit("${getClassTag()} --> checkResult() --> 更新完计划且下载完素材,新计划可用 --> lastVersion = $lastVersion --> newVersion = $newVersion --> size = ${AdPlansStorageManager.getAllAdPlans()?.size}")
        }
    }


    // ================================= 无效资源删除 ================================================


    /**
     * 删除过期计划
     */
    private suspend fun deleteExpireAdPlan(newVersion: String, lastVersion: String?) {
        if (TextUtils.isEmpty(newVersion)) {
            return
        }
        AdPlansStorageManager.deleteExpireVersion(newVersion = newVersion)
        AdLogger.logAdInit("${getClassTag()} --> deleteExpireAdPlan() --> newVersion = $newVersion -- lastVersion = $lastVersion --> 删除过期计划 -- success")
    }

    /**
     * 检查请求到的广告计划中素材是否下载完成
     */
    private suspend fun checkDownloadMaterial(adPlans: MutableList<AdPlans>): Boolean {
        // 默认所有素材下载成功，只要发现一个没有下载，那当前判断不通过
        var res = true

        // 遍历所有计划进行判断
        adPlans.forEach { plan ->
            // 重新从数据库查询
            AdPlansStorageManager.getPlanById(plan.id)?.let { nonAdPlans ->
                // 结构转换
                val db2Bean = AdPlansTransform.transformPlanDb2Bean(nonAdPlans)

                // 获取资源数据
                if (db2Bean?.adMaterialList?.isNotEmpty() == true) {
                    db2Bean.adMaterialList?.forEachIndexed { _, adMaterial ->
                        //素材中有未下载完成的为false
                        if (!adMaterial.downloadMaterialSuccess) {
                            res = false
                            return@forEach
                        }
                    }
                } else {
                    //空素材直接设置为空
                    res = false
                    return@forEach
                }

                // 如果是自有类型才需要判断
                if (AdPlanSourceManager.isDefaultAdPlan(plan)) {
                    // 扩展图片也是需要下载成功的
                    if (db2Bean.extImage != null && TextUtils.isEmpty(db2Bean.extImage?.url)
                            .not()
                    ) {
                        val empty = TextUtils.isEmpty(db2Bean.extImage?.path)
                        if (empty) {
                            res = false
                            return@forEach
                        }
                    }
                }

            } ?: run {
                // 如果数据库没有对应的数据直接返回 false
                res = false
                return@forEach
            }
        }
        return res
    }

    /**
     * 删除过期资源
     */
    private fun deleteExpireMaterial(adPlans: MutableList<AdPlans>) {
        val pathSet = hashSetOf<String?>()
        adPlans.forEach {
            it.adMaterialList?.forEach { adMaterialList ->
                if (AdMaterialList.NON_AD_TYPE_TEXT == adMaterialList.type) {
                    pathSet.add(adMaterialList.image?.path)
                } else {
                    pathSet.add(adMaterialList.video?.path)
                }
            }
            // 广告主头像也不能删除了
            if (TextUtils.isEmpty(it.advertiserAvatarPath).not()) {
                pathSet.add(it.advertiserAvatarPath)
            }
        }
        val destinationDir = MbAdContents.NON_AD_VIDEO_DOWNLOAD_FILE_PATH + File.separatorChar
        // 删除过期资源
        FileUtil.deleteFile(File(destinationDir), set = pathSet)
        AdLogger.logAdInit("${getClassTag()} --> deleteExpireMaterial() --> 删除过期资源 -- success")
    }

}