package com.transsion.ad.ps

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Build
import android.text.TextUtils
import android.util.Log
import android.view.View
import com.blankj.utilcode.util.AppUtils
import com.blankj.utilcode.util.Utils
import com.cloud.hisavana.sdk.common.bean.ImgListDTO
import com.cloud.hisavana.sdk.common.bean.PslinkInfo
import com.transsion.ad.log.AdLogger
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.strategy.AdClickManager
import org.json.JSONObject
import java.util.Locale

/**
 * @author xinbing.zhang
 * @date :2023/10/17 19:14
 * @description:
 */
object PsLinkUtils {

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * Palm Store 包名
     */
    private const val PS_PACKAGE_NAME = "com.transsnet.store"

    // A B C D E F(A插屏样式  B顶部悬浮  C底部弹窗， DEF与ABC对应样式一样但功能有区别，DEF样式在半屏点击install按钮会自动关闭半屏，同时当任务加入下载队列后下次跳半屏，不会打开半屏页，会Toast提示)
    const val PS_LINK_DIALOG_STYLE = "B"

    /**
     * 当前是否安装了PS
     */
    fun isPsInstalled(): Boolean {
        return AppUtils.isAppInstalled(PS_PACKAGE_NAME)
    }

    fun checkPsInstalled(): Boolean {
        if (AppUtils.isAppInstalled(PS_PACKAGE_NAME)) {
            return true
        }
        return checkAppInstalled(PS_PACKAGE_NAME)
    }


    /**
     * 广告点击跳转处理
     */
    fun adClick(itemInfo: RecommendInfo?, isAutoDownload: Boolean = false, scene: String) {
        val gpLink = itemInfo?.gpLink
//        val gpLink = "https://play.google.com/store/apps/details?id=com.game.rescueagent&hl=zh"
        //不支持ps半屏 + 有gp链接 直接跳转gp
        if (!isSupportPsLink() && gpLink?.contains("play.google.com") == true) {
            AdClickManager.jumpDeeplink(gpLink)
        } else {
            if (isSupportPsLink()) {
                startPsActivity(
                    itemInfo, isAutoDownload = isAutoDownload, scene
                )
            } else {
                // 打开PS全屏
                startPsApp(itemInfo)
            }
        }
    }

    /**
     * 通过包名检查是否安装过了
     */
    fun checkAppInstalled(pkgName: String?): Boolean {
        if (TextUtils.isEmpty(pkgName)) {
            return false
        }
        var packageInfo: PackageInfo? = null
        kotlin.runCatching {
            packageInfo = Utils.getApp().packageManager.getPackageInfo(pkgName ?: "", 0)
        }
        return packageInfo != null
    }

    /**
     * 判断PS是否支持下载状态回调(aidl连接之前可用)
     *
     * @return  true: 支持，   false: 不支持
     */
    fun isSupportPSDistrob(): Boolean {
        val psService = "com.transsnet.store.action.CommonDistribution"
        val packageName = PS_PACKAGE_NAME
        var isSupport = false
        try {
            val service = Intent(psService)
            service.setPackage(packageName)
            val ps: PackageManager = Utils.getApp().packageManager
            val target = ps.queryIntentServices(service, 0)
            if (target.isNotEmpty()) {
                isSupport = true
            }
        } catch (e: Exception) {
            AdLogger.logPS(
                "PsLinkUtils --> isSupportPSDistrob() --> it = ${
                    Log.getStackTraceString(
                        e
                    )
                }", level = Log.ERROR
            )
        }
        return isSupport
    }

    /**
     * 打开 PS 半屏
     */
    fun startPsActivity(
        itemInfo: RecommendInfo?, isAutoDownload: Boolean = false, scene: String
    ): Boolean {
        return kotlin.runCatching {
            val intent = Intent()
            intent.action = Intent.ACTION_VIEW

            // 输出一下 deeplink
            val deeplink = getDeeplink(itemInfo, scene = scene)
            AdLogger.logPS("${getClassTag()} --> startPsActivity() --> isAutoDownload = $isAutoDownload --> deeplink = $deeplink")
            intent.data = Uri.parse(deeplink)

            intent.putExtra(
                "PslinkInfo", getPsLinkInfo(itemInfo, isAutoDownload)
            ) //半屏必填  PslinkInfo结构的Parcel

            intent.putExtra("versionCode", itemInfo?.versionCode) //应用版本号；int
            intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TASK)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            intent.putExtra("psExtendFields", getExtendFields(itemInfo, scene))
            Utils.getApp().startActivity(intent)
            true
        }.getOrElse {
            AdLogger.logPS(
                "${getClassTag()} --> startPsActivity --> it = ${Log.getStackTraceString(it)}",
                level = Log.ERROR
            )
            false
        }
    }

    private fun getPsLinkInfo(itemInfo: RecommendInfo?, isAutoDownload: Boolean): PslinkInfo {
        //应用基础信息
        val info = PslinkInfo()
        info.clickType = if (isAutoDownload) 2 else 1 //2:可配置自动下载， 1：非自动下载
        info.itemID = itemInfo?.itemID // 应用的itemId
        info.packageName = itemInfo?.packageName //下载应用的包名
        info.appPackageName = Utils.getApp().packageName //媒体包名
        info.name = itemInfo?.name ?: itemInfo?.detail?.name // 应用名
        info.star = itemInfo?.star //评星
        info.downloadCount = itemInfo?.downloadCount?.toInt() ?: 0 //下载次数
        info.sourceSize = itemInfo?.size?.toString() //包体大小
        //info.iconUri = uri //应用ICON，缓存到本地的uri
        info.iconUrl = itemInfo?.iconUrl // 应用ICON url地址
        val list: MutableList<ImgListDTO> = ArrayList()

        if (!TextUtils.isEmpty(itemInfo?.detail?.img0)) {
            val imgInfo = ImgListDTO()
            //imgInfo.uri = uri //应用截图，缓存到本地的uri，可空；
            imgInfo.url = itemInfo?.detail?.img0 //应用截图 url地址
            list.add(imgInfo)
            imgInfo.hv = getScreenshotMode(
                itemInfo?.detail?.screenshotMode, list.size - 1
            ) //截图类型： 0竖图  1横图  3视频
        }

        if (!TextUtils.isEmpty(itemInfo?.detail?.img1)) {
            val imgInfo = ImgListDTO()
            //imgInfo.uri = uri //应用截图，缓存到本地的uri，可空；
            imgInfo.url = itemInfo?.detail?.img1 //应用截图 url地址
            list.add(imgInfo)
            imgInfo.hv = getScreenshotMode(
                itemInfo?.detail?.screenshotMode, list.size - 1
            ) //截图类型： 0竖图  1横图  3视频
        }

        if (!TextUtils.isEmpty(itemInfo?.detail?.img2)) {
            val imgInfo = ImgListDTO()
            //imgInfo.uri = uri //应用截图，缓存到本地的uri，可空；
            imgInfo.url = itemInfo?.detail?.img2 //应用截图 url地址
            list.add(imgInfo)
            imgInfo.hv = getScreenshotMode(
                itemInfo?.detail?.screenshotMode, list.size - 1
            ) //截图类型： 0竖图  1横图  3视频
        }

        if (!TextUtils.isEmpty(itemInfo?.detail?.img3)) {
            val imgInfo = ImgListDTO()
            //imgInfo.uri = uri //应用截图，缓存到本地的uri，可空；
            imgInfo.url = itemInfo?.detail?.img3 //应用截图 url地址
            list.add(imgInfo)
            imgInfo.hv = getScreenshotMode(
                itemInfo?.detail?.screenshotMode, list.size - 1
            ) //截图类型： 0竖图  1横图  3视频
        }

        if (!TextUtils.isEmpty(itemInfo?.detail?.img4)) {
            val imgInfo = ImgListDTO()
            //imgInfo.uri = uri //应用截图，缓存到本地的uri，可空；
            imgInfo.url = itemInfo?.detail?.img4 //应用截图 url地址
            list.add(imgInfo)
            imgInfo.hv = getScreenshotMode(
                itemInfo?.detail?.screenshotMode, list.size - 1
            ) //截图类型： 0竖图  1横图  3视频
        }
        info.imgList = list
        info.simpleDescription = itemInfo?.simpleDescription //应用一句话描述信息
        return info
    }

    /**
     * 获取图片的类型
     */
    private fun getScreenshotMode(list: List<Int?>?, pos: Int): Int {
        if (list.isNullOrEmpty() || pos < 0) {
            return 0
        }
        return list[pos % list.size] ?: 0
    }

    private fun getDeeplink(itemInfo: RecommendInfo?, scene: String? = ""): String {
        /**
         * HalfScreenType：必填，支持展示样式：A B C D E F（A插屏半屏， B顶部悬浮弹框， C底部弹出的半屏），
         * DEF分别与ABC展示样式一致，区别是加入队列后不可重复打开；
         *
         * HalfScreenType: A B C D E F(A插屏样式 B顶部悬浮 C底部弹窗, DEF与ABC对应)
         * isHalfScreen: 1  代表半屏详情页，不带为全屏详情页
         */
        return "palmplay://thirdlauncher.com/" + // 跳转PS
                "?entryType=AppDetail" + // AppDetail    代表详情页
                "&packageName=${itemInfo?.packageName}" + // 应用包名，需要下载的三方应用包名，非媒体包名
                "&_source=Moviebox" + // 来源，PM传入"PM"，Moviebox传入："Moviebox"

                // source，pid都是固定值，siteid代表appid，sub_siteid代表代码位ID，
                // trackType是2.5.6.0_S新增的归因标识（注意pid，siteid，sub_siteid,trackType为EW特有参数，PM,Moviebox，aha无需传入）
//                "&pid=eagllwin_int" + //
//                "&siteid=2301069UF9YjuO" + //
//                "&sub_siteid=230223UEiWd8Ps" + //
                "&HalfScreenType=$PS_LINK_DIALOG_STYLE" + // A B C D E F(A插屏样式  B顶部悬浮  C底部弹窗， DEF与ABC对应样式一样但功能有区别，DEF样式在半屏点击install按钮会自动关闭半屏，同时当任务加入下载队列后下次跳半屏，不会打开半屏页，会Toast提示)
                "&isHalfScreen=1" + // 此字段在V8.4.3版本上有用，在V8.4.5以上版本看HalfScreenType字段
                "&sceneCode=$scene" //场景码 v9.3.2版本新增参数，场景可配 mb3.0.00版本引入
    }

    /**
     * 没有ps半屏的版本展示全屏
     */
    private fun getDeeplinkFullScreen(itemInfo: RecommendInfo?): String {
        return "palmplay://thirdlauncher.com/" + // 跳转PS
                "?entryType=AppDetail" + // AppDetail    代表详情页
                "&packageName=${itemInfo?.packageName}" + // 应用包名，需要下载的三方应用包名，非媒体包名
                "&_source=Moviebox"  // 来源，PM传入"PM"，Moviebox传入："Moviebox"

    }

    /**
     * 通过包名或类名启动另一个APP
     */
    fun startApplicationWithPackageName(packName: String?): Boolean {
        return try {
            // 通过包名获取此APP详细信息，包括Activities、services、versioncode、name等等
            val packageInfo =
                Utils.getApp().packageManager.getPackageInfo(packName ?: "", 0) ?: return false

            // 创建一个类别为CATEGORY_LAUNCHER的该包名的Intent
            val resolveIntent = Intent(Intent.ACTION_MAIN, null)
            resolveIntent.addCategory(Intent.CATEGORY_LAUNCHER)
            resolveIntent.setPackage(packageInfo.packageName)

            // 通过getPackageManager()的queryIntentActivities方法遍历
            val resolveInfoList =
                Utils.getApp().packageManager.queryIntentActivities(resolveIntent, 0)
            if (resolveInfoList.isEmpty()) {
                return false
            }

            val resolveInfo = resolveInfoList.iterator().next()
            if (resolveInfo?.activityInfo != null) {
                // packageName = 参数packName
                val packageName = resolveInfo.activityInfo.packageName
                // 这个就是我们要找的该APP的LAUNCHER的Activity[组织形式：packageName.mainActivityName]
                val className = resolveInfo.activityInfo.name
                if (TextUtils.isEmpty(packageName) || TextUtils.isEmpty(className)) {
                    return false
                }
                // LAUNCHER Intent
                val intent = Intent(Intent.ACTION_MAIN)
                intent.addCategory(Intent.CATEGORY_LAUNCHER)
                intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
                // 设置ComponentName参数1:packageName参数2:MainActivity路径
                val cn = ComponentName(packageName, className)
                intent.component = cn
                Utils.getApp().startActivity(intent)
                // 包名打开也需要上报埋点
                return true
            }
            false
        } catch (e: java.lang.Exception) {
            AdLogger.logPS(
                "${getClassTag()} --> it = ${Log.getStackTraceString(e)}", level = Log.ERROR
            )
            false
        }
    }

    private fun getExtendFields(recommendInfo: RecommendInfo?, scene: String): String {
        val json = JSONObject().apply {
            put("id", recommendInfo?.id)
            put("isOffer", recommendInfo?.isOffer)
            put("showType", recommendInfo?.showType)
            put("planName", recommendInfo?.planName)
            put("media", "MovieBox")
            put("scene", scene)
            put("planId", recommendInfo?.id)
            put("c_source", "PS")
            put("adSource", recommendInfo?.adSource)
            put("mbVc", AppUtils.getAppVersionCode().toString())
        }
        return json.toString()
    }


    /**
     * 当前PS版本是否支持半屏
     * PS支持半屏版本号为V8.4.3； V8.4.3以下版本不支持半屏;  支持半屏的版本号：8403100
     * 843及以上是支持ew调半屏，912及以上是支持moviebox调半屏；
     */
    fun isSupportPsLink(): Boolean {
        return getVersionCode(Utils.getApp(), PS_PACKAGE_NAME) >= 8403100
    }

    /**
     * 获取指定包名的版本号
     *
     * @param context 本应用程序上下文
     * @param packageName 你想知道版本信息的应用程序的包名
     * @return 目标版本号
     */
    private fun getVersionCode(context: Context, packageName: String): Long {
        var versionCode = 0L
        kotlin.runCatching {
            val packageManager = context.packageManager
            val packInfo = packageManager.getPackageInfo(packageName, 0)
            versionCode = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                packInfo.longVersionCode
            } else {
                packInfo.versionCode.toLong()
            }
        }
        return versionCode
    }


    fun getPSVersionCode(): Long {
        return getVersionCode(Utils.getApp(), PS_PACKAGE_NAME)
    }


    /**
     * 是否是 RTL
     */
    fun isRTLLayout(): Boolean {
        return TextUtils.getLayoutDirectionFromLocale(Locale.getDefault()) == View.LAYOUT_DIRECTION_RTL
    }

    fun startPsApp(itemInfo: RecommendInfo?): Boolean {
        return kotlin.runCatching {
            val intent = Intent()
            intent.action = Intent.ACTION_VIEW
            intent.data = Uri.parse(getDeeplinkFullScreen(itemInfo))
            intent.putExtra("versionCode", AppUtils.getAppVersionCode()) //应用版本号；int
            intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TASK)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            Utils.getApp().startActivity(intent)
            true
        }.getOrElse {
            AdLogger.logPS(
                "${getClassTag()} --> startPsActivity --> it = ${
                    Log.getStackTraceString(
                        it
                    )
                }", level = Log.ERROR
            )
            false
        }
    }
}