package com.transsion.ad.scene

import android.text.TextUtils
import com.blankj.utilcode.util.GsonUtils
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import com.transsion.ad.MbAdContents
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.model.MbAdPlansDto
import com.transsion.ad.strategy.AdMmkv

/**
 * @author: zhangxinbing
 * @date : 2025/2/17 15:04
 * @description: 场景存储
 */
object SceneStorage {

    /**
     * 缓存 --> 优化数据库保存
     */
    private val cacheMap = HashMap<String, JsonObject>()

    /**
     *         {
     *                 "id": "GlobConfigScene",
     *                 "newUserMaskTime": 3,
     *                 "nativeCacheUpperLimit": 2,
     *                 "isOpenAd": true,
     *                 "adLinkOff": false,
     *                 "lowMemoryValue": 1,
     *                 "videoCd": 15,
     *                 "interstitialCd": 5,
     *                 "showAdTag":"false"
     *         },
     */
    private const val SCENE_CONFIG_KEY_ID = "id"

    private fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    /**
     * 是否已经有数据了
     */
    fun hasData(): Boolean {
        // 没有缓存那就从本地读取
        val value = AdMmkv.mmkvWithID.getString(MbAdContents.MMKV_KEY_AD_SCENE_CONFIG_DATA, "")
        return TextUtils.isEmpty(value).not()
    }

    fun setScene4Assets(assetsString: String) {
        AdMmkv.mmkvWithID.putString(MbAdContents.MMKV_KEY_AD_SCENE_CONFIG_DATA, assetsString)// 保存数据
        AdLogger.logAdInit("${getClassTag()} --> setScene4Assets() --> 场景信息保存到MMKV --> success")
    }

    /**
     * 保存数据
     */
    fun setSceneConfig(mbAdPlansDto: MbAdPlansDto) {
        val localAdSceneVersion = AdMmkv.mmkvWithID.getString(MbAdContents.NON_AD_SCENE_VERSION, "")
        val newVersion = mbAdPlansDto.data?.versionAdScene
        if (TextUtils.equals(localAdSceneVersion, newVersion).not()) {
            AdMmkv.mmkvWithID.putString(MbAdContents.NON_AD_SCENE_VERSION, newVersion) // 保存最新版本号
            AdMmkv.mmkvWithID.putString(
                MbAdContents.MMKV_KEY_AD_SCENE_CONFIG_DATA, mbAdPlansDto.data?.adSceneConfigData
            )// 保存数据
            AdLogger.logAdInit("${getClassTag()} --> setSceneConfig() --> 场景信息保存到MMKV --> success --> newVersion = $newVersion")
        } else {
            AdLogger.logAdInit("${getClassTag()} --> setSceneConfig() --> 版本号一致，服务端不会下发list 省流量 --> localAdSceneVersion = $localAdSceneVersion --> newVersion = $newVersion")
        }
    }

    /**
     * 获取所有配置
     */
    fun getSceneConfig(): String? {
        return kotlin.runCatching {
            AdMmkv.mmkvWithID.getString(MbAdContents.MMKV_KEY_AD_SCENE_CONFIG_DATA, "")
        }.getOrElse {
            null
        }
    }

    /**
     * 获取当前的版本号
     */
    fun getSceneVersion(): String? =
        AdMmkv.mmkvWithID.getString(MbAdContents.NON_AD_SCENE_VERSION, "")

    /**
     * 通过场景ID 获取场景的配置信息
     */
    fun getSceneConfig(sceneId: String?): JsonObject? {
        kotlin.runCatching {
            // 返回缓存
            if (cacheMap.containsKey(sceneId)) {
                return cacheMap[sceneId]
            }

            // 没有缓存那就从本地读取
            val value = AdMmkv.mmkvWithID.getString(MbAdContents.MMKV_KEY_AD_SCENE_CONFIG_DATA, "")
            val array = GsonUtils.fromJson(value, JsonArray::class.java)
            array?.forEach {
                it?.asJsonObject?.let { json ->
                    val scene = json.get(SCENE_CONFIG_KEY_ID)?.asString ?: ""
                    cacheMap[scene] = json
                }
            }

            // 返回缓存
            if (cacheMap.containsKey(sceneId)) {
                return cacheMap[sceneId]
            }
        }
        return null
    }

}