package com.transsion.ad.strategy

import android.graphics.Rect
import android.os.Handler
import android.os.HandlerThread
import android.os.Looper
import android.view.View
import java.util.concurrent.CopyOnWriteArrayList

/**
 * @author: zhangxinbing
 * @date : 2023/12/15 16:56
 * @description: 单独开启一个线程，用于检测View是否在Window中可见
 */
object MeasureManager {

    /*** View可见性检测回调*/
    interface ViewVisibilityListener {

        fun onVisibilityChanged(isVisible: Boolean)

        fun getVisibilityView(): View?

        /**
         * 检测阈值
         */
        fun getVisibilityThreshold(): Double = 10.0
    }

    /*** 检测目标View是否在Window中完全可见*/
    private val mClipRect = Rect()

    private fun checkVisibility(view: View?, callBack: ((visibility: Double) -> Unit)? = null) {
        view?.let {
            if (it.windowVisibility != View.VISIBLE) {
                callBack?.invoke(0.0)
                return
            }
            // 在window中可见一个像素 true 否则 false
            val globalVisibleRect = it.getGlobalVisibleRect(mClipRect)
            if (!globalVisibleRect || it.isShown.not()) {
                callBack?.invoke(0.0)
                return
            }
            // 可见区域面积
            val visibleViewArea = mClipRect.height().toDouble() * mClipRect.width()
            // 目标View总面积
            val totalViewArea: Double = it.height.toDouble() * it.width
            if (totalViewArea <= 0) {
                callBack?.invoke(0.0)
                return
            }
            val area = visibleViewArea * 100 / totalViewArea
            callBack?.invoke(area)
        }
    }

    /*** 判断View是否在屏幕中可见*/
    fun isVisibility(view: View?): Boolean {
        view?.let {
            if (it.windowVisibility != View.VISIBLE) {
                return false
            }

            // 在window中可见一个像素 true 否则 false
            val globalVisibleRect = it.getGlobalVisibleRect(mClipRect)
            if (!globalVisibleRect || it.isShown.not()) {
                return false
            }

            // 可见区域面积
            val visibleViewArea = mClipRect.height().toDouble() * mClipRect.width()
            // 目标View总面积
            val totalViewArea: Double = it.height.toDouble() * it.width
            if (totalViewArea <= 0) {
                return false
            }

            val area = visibleViewArea * 100 / totalViewArea
            return area > 0
        }
        return false
    }

    // =============================================================================================


    private fun getClassTag(): String = javaClass.simpleName

    /*** 任务列表 --> listener*/
    private val sessionList = CopyOnWriteArrayList<ViewVisibilityListener>()

    /*** 单独开启一个线程处理View的曝光检测*/
    private val myWorkerThreadHandler by lazy(mode = LazyThreadSafetyMode.NONE) {
        HandlerThread("MyHandlerThread").apply {
            start()
        }
    }

    /*** 子线程Handler*/
    private val myWorkerHandler by lazy {
        Handler(myWorkerThreadHandler.looper)
    }

    /*** 检测自己是否可见*/
    val mHandler by lazy {
        Handler(Looper.getMainLooper())
    }


    // =============================================================================================


    /*** 添加任务*/
    fun addSession(listener: ViewVisibilityListener) {
        // 所有的增加删除都在一个线程完成
        myWorkerHandler.post {
            if (!sessionList.contains(listener)) {
                sessionList.add(listener)
            }

            // 如果只有一个那就开启监测
            if (sessionList.size == 1) {
                start()
            }
        }
    }

    /*** 移除任务*/
    fun removeSession(listener: ViewVisibilityListener) {
        // 所有的增加删除都在一个线程完成
        myWorkerHandler.post {
            sessionList.remove(listener)
        }
    }

    /*** 开始*/
    fun start() {
        myWorkerHandler.post {
            measure()
        }
    }

    /*** 停止*/
    fun stop() {
        myWorkerHandler.post {
            myWorkerHandler.removeCallbacks(measureRunnable)
        }
        measure()
    }


    // =============================================================================================


    /*** 定时检测任务*/
    private val measureRunnable = Runnable {
        measure()
    }

    /*** 测量 --> 遍历所有数据进行测量*/
    private fun measure() {
        // AdLogger.log("${getClassTag()} --> measure() -- ThreadName = ${Thread.currentThread().name} -- size = ${sessionList.size}")

        // 移除所有消息
        myWorkerHandler.removeCallbacks(measureRunnable)

        // 遍历检查
        sessionList.forEach {
            checkVisibility(it.getVisibilityView()) { visibility: Double ->
                if (visibility >= it.getVisibilityThreshold()) {
                    it.onVisibilityChanged(true)
                } else {
                    it.onVisibilityChanged(false)
                }
            }
        }

        // 判断是否还有任务了
        if (sessionList.isEmpty()) {
            // 如果没有了 那就不处理
//            AdLogger.logCommon(
//                msg = "${getClassTag()} --> measure() --> true = sessionList.isEmpty()",
//                writeToFile = false
//            )
        } else {
            // 循转执行
            myWorkerHandler.postDelayed(measureRunnable, 500)
        }
    }

}