package com.transsion.ad.util

import android.content.Context
import android.content.Intent
import android.content.pm.ApplicationInfo
import android.content.pm.PackageManager
import android.net.Uri

/**
 * @author shmizhangxinbing
 * @date : 2025/7/11 14:23
 * @description:
 */
object AppUtil {

    /**
     * 根据包名判断应用是否已安装
     *
     * @param context 上下文
     * @param packageName 应用包名
     * @return 是否已安装
     *
     * 如果仅仅是使用 PackageManager.getPackageInfo() 方法通过包名判断应用是否已安装，这种操作不需要任何权限。
     */
    fun isAppInstalled(context: Context?, packageName: String?): Boolean {
        if (packageName.isNullOrEmpty() || context == null) {
            return false
        }
        val packageManager = context.packageManager ?: return false

        return try {
            // 根据不同的系统版本调用兼容的方法
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.TIRAMISU) {
                // Android 13 (API 33) 及更高版本
                packageManager.getPackageInfo(packageName, PackageManager.PackageInfoFlags.of(0))
            } else {
                // Android 12 (API 31) 及以下版本
                packageManager.getPackageInfo(packageName, 0)
            }
            true
        } catch (e: PackageManager.NameNotFoundException) {
            false
        }
    }

    /**
     * 判断应用是否是系统应用
     *
     * @param context 上下文
     * @param packageName 应用包名
     * @return 是否是系统应用
     */
    fun isSystemApp(context: Context, packageName: String?): Boolean {
        if (packageName.isNullOrEmpty()) {
            return false
        }

        val packageManager = context.packageManager
        return try {
            val applicationInfo = packageManager.getApplicationInfo(packageName, 0)
            (applicationInfo.flags and ApplicationInfo.FLAG_SYSTEM) != 0
        } catch (e: PackageManager.NameNotFoundException) {
            false
        }
    }

    fun openApp(context: Context?, packageName: String?) {
        val launchIntent = context?.packageManager?.getLaunchIntentForPackage(packageName ?: "")
        if (launchIntent != null) {
            context.startActivity(launchIntent)
        } else {
            // 应用未找到，提示错误
            //Toast.makeText(context, "无法启动应用", Toast.LENGTH_SHORT).show()
        }
    }

    fun openAppInStore(context: Context?, packageName: String?) {
        try {
            val intent = Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id=$packageName"))
            context?.startActivity(intent)
        } catch (e: Exception) {
            kotlin.runCatching {
                // 如果设备上没有应用商店，打开浏览器访问应用下载页面
                val intent = Intent(
                    Intent.ACTION_VIEW,
                    Uri.parse("https://play.google.com/store/apps/details?id=$packageName")
                )
                context?.startActivity(intent)
            }
        }
    }

    fun openThirdPartyApp(context: Context?, packageName: String?) {
        if (isAppInstalled(context, packageName)) {
            openApp(context, packageName)
        } else {
            openAppInStore(context, packageName)
        }
    }

}
