package com.transsion.ad.view

import android.content.Context
import android.graphics.Canvas
import android.graphics.Path
import android.graphics.RectF
import android.util.AttributeSet
import android.widget.FrameLayout
import androidx.core.content.withStyledAttributes
import com.transsion.ad.R

/**
 * @author shmizhangxinbing
 * @date : 2025/8/28 10:47
 * @description:
 *
 * 如果你希望每个圆角都能独立设置，可以用 Path 和 Canvas 实现裁剪，这样就可以分别设置四个角的圆角半径了。
 * 这种方式支持 API 16+。
 * 性能很好，实际项目中广泛使用。
 * 你可以灵活设置每个角的圆角半径。
 * 若需要动态更改圆角，只需重新设置属性并调用 invalidate() 即可。
 */
class CustomRoundCornerLayout @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    private var allCornerRadius: Float = 0f
        set(value) {
            field = value
            updateRadii()
            invalidate()
        }

    var topLeftRadius: Float = 0f
        set(value) {
            field = value
            updateRadii()
            invalidate()
        }
    var topRightRadius: Float = 0f
        set(value) {
            field = value
            updateRadii()
            invalidate()
        }
    var bottomLeftRadius: Float = 0f
        set(value) {
            field = value
            updateRadii()
            invalidate()
        }
    var bottomRightRadius: Float = 0f
        set(value) {
            field = value
            updateRadii()
            invalidate()
        }

    private val path = Path()
    private val rect = RectF()

    // 用于实际绘制的圆角数组
    private val radii = FloatArray(8)

    init {
        context.withStyledAttributes(attrs, R.styleable.CustomRoundCornerLayout) {
            allCornerRadius = getDimension(R.styleable.CustomRoundCornerLayout_allCornerRadius, 0f)
            topLeftRadius =
                getDimension(R.styleable.CustomRoundCornerLayout_topLeftRadius, allCornerRadius)
            topRightRadius =
                getDimension(R.styleable.CustomRoundCornerLayout_topRightRadius, allCornerRadius)
            bottomLeftRadius =
                getDimension(R.styleable.CustomRoundCornerLayout_bottomLeftRadius, allCornerRadius)
            bottomRightRadius =
                getDimension(R.styleable.CustomRoundCornerLayout_bottomRightRadius, allCornerRadius)
        }
        updateRadii()
        setWillNotDraw(false)
    }

    private fun updateRadii() {
        radii[0] = if (topLeftRadius > 0f) topLeftRadius else allCornerRadius
        radii[1] = radii[0]
        radii[2] = if (topRightRadius > 0f) topRightRadius else allCornerRadius
        radii[3] = radii[2]
        radii[4] = if (bottomRightRadius > 0f) bottomRightRadius else allCornerRadius
        radii[5] = radii[4]
        radii[6] = if (bottomLeftRadius > 0f) bottomLeftRadius else allCornerRadius
        radii[7] = radii[6]
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        rect.set(0f, 0f, w.toFloat(), h.toFloat())
        path.reset()
        path.addRoundRect(rect, radii, Path.Direction.CW)
        path.close()
    }

    override fun dispatchDraw(canvas: Canvas) {
        val save = canvas.save()
        canvas.clipPath(path)
        super.dispatchDraw(canvas)
        canvas.restoreToCount(save)
    }
}
