package com.transsion.ad.view.blur;

/**
 * Copyright (C) 2018 Wasabeef
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;

import androidx.annotation.NonNull;

import com.blankj.utilcode.util.ImageUtils;
import com.bumptech.glide.load.engine.bitmap_recycle.BitmapPool;
import com.bumptech.glide.load.resource.bitmap.BitmapTransformation;

import java.security.MessageDigest;


public class BlurTransformation extends BitmapTransformation {

    private static final int VERSION = 1;
    private static final String ID = "BlurTransformation." + VERSION;

    private static final int MAX_RADIUS = 25;
    private static final int DEFAULT_DOWN_SAMPLING = 1;

    private final int radius;

    private final int sampling;
    private boolean isUseCustomBlur = false;

    public BlurTransformation() {
        this(MAX_RADIUS, DEFAULT_DOWN_SAMPLING);
    }

    @Override
    protected Bitmap transform(@NonNull BitmapPool pool, @NonNull Bitmap toTransform, int outWidth, int outHeight) {

        int width = toTransform.getWidth();
        int height = toTransform.getHeight();
        int scaledWidth = width / sampling;
        int scaledHeight = height / sampling;

        Bitmap bitmap = pool.get(scaledWidth, scaledHeight, Bitmap.Config.ARGB_8888);

        Canvas canvas = new Canvas(bitmap);
        canvas.scale(1 / (float) sampling, 1 / (float) sampling);
        Paint paint = new Paint();
        paint.setFlags(Paint.FILTER_BITMAP_FLAG);
        canvas.drawBitmap(toTransform, 0, 0, paint);

        if (isUseCustomBlur) {
            bitmap = FastBlur.blur(bitmap, radius, true);
        } else {
            bitmap = ImageUtils.fastBlur(bitmap, 1, radius);
        }

        return bitmap;
    }

    public BlurTransformation(int radius) {
        this(radius, DEFAULT_DOWN_SAMPLING);
    }

    /**
     * 1. 模糊半径（Blur Radius）
     * 定义：模糊半径是应用模糊效果时，指定的模糊程度。它通常表示在模糊过程中每个像素影响的邻近像素的范围。
     * 影响：模糊半径越大，模糊效果越明显，图像看起来会越模糊。相反，半径越小，模糊效果越轻微，图像更清晰。
     * 示例：如果你设置模糊半径为 10，意味着每个像素会影响周围 10 像素范围内的像素，使整个图像呈现出较强的模糊效果。
     * 2. 采样率（Sampling Rate）
     * 定义：采样率是指在应用模糊效果时，图像被缩小的程度。它通常通过将原始图像的宽度和高度分别除以一个整数值来实现。
     * 影响：采样率越高（即数字越大），图像在模糊处理前被缩小得越多，从而处理后的图像尺寸也越小。这可以在处理大量图像时减少内存使用和提升性能，因为较小的图像处理起来更快。
     * 示例：如果你设置采样率为 2，意味着原始图像会被缩小到原始尺寸的一半，在此基础上应用模糊效果。处理完成后，图像会被放大回原来的尺寸。这样可能会出现更平滑的模糊效果，但在细节上可能会有所损失。
     * 3.结论
     * 将模糊半径和采样率结合使用，可以灵活地控制图像的模糊效果和处理性能。选择合适的值可以根据你的实际需求，比如性能、图像质量等进行调整。
     *
     * @param radius   模糊半径（Blur Radius）
     * @param sampling 采样率（Sampling Rate）
     */
    public BlurTransformation(int radius, int sampling) {
        this.radius = radius;
        this.sampling = sampling;
    }

    public BlurTransformation(int radius, int sampling, boolean isUseCustomBlur) {
        this.radius = radius;
        this.sampling = sampling;
        this.isUseCustomBlur = isUseCustomBlur;
    }

    @Override
    public String toString() {
        return "BlurTransformation(radius=" + radius + ", sampling=" + sampling + ")";
    }

    @Override
    public boolean equals(Object o) {
        return o instanceof BlurTransformation && ((BlurTransformation) o).radius == radius && ((BlurTransformation) o).sampling == sampling;
    }

    @Override
    public int hashCode() {
        return ID.hashCode() + radius * 1000 + sampling * 10;
    }

    @Override
    public void updateDiskCacheKey(@NonNull MessageDigest messageDigest) {
        messageDigest.update((ID + radius + sampling).getBytes(CHARSET));
    }
}
