package com.transsion.ad.bidding.icon

import android.os.Handler
import android.os.Looper
import android.text.TextUtils
import android.util.Log
import com.hisavana.common.bean.AdditionalInfo
import com.hisavana.common.bean.TAdErrorCode
import com.hisavana.common.bean.TAdNativeInfo
import com.hisavana.mediation.ad.TNativeAd
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.model.MbAdSource
import com.transsion.ad.monopoly.model.MbAdType
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.ps.PsLinkUtils
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.report.AdReportProvider
import com.transsion.ad.report.BiddingStateEnum
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.scene.SceneOnOff
import com.transsion.ad.scene.SceneStorage
import com.transsion.ad.strategy.AdClickManager
import com.transsion.ad.strategy.AdContextManager.getCtxMap
import com.transsion.ad.strategy.NewUserShieldStrategy
import com.transsion.ad.util.RandomUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/8/20 20:27
 * @description: Icon 广告加载,当前仅处理数据的加载和曝光处理
 *
 * 1. 并发请求 --> Hi程序化+PS兜底
 * 2. 竞价时间到 --> 组合返回数据
 * 3. 回调结果 --> 将结果回调
 */
class BiddingIconAdManager : BiddingTAdditionalListener() {

    /**
     * 两个接口暂存数据
     */
    private var temporaryStorageHi: MutableList<TAdNativeInfo> = mutableListOf()    // Hi程序化结果集合
    private var temporaryStoragePs: MutableList<RecommendInfo> = mutableListOf()    // PS兜底接口结果集合
    private var valueList: MutableList<BiddingWrapperIconBean>? = mutableListOf()   // 返回结果结合

    private fun getClassTag(): String = javaClass.simpleName

    private var mSceneId: String? = null                            // 广告场景ID
    private var mCallback: BiddingTAdditionalListener? = null       // 回调
    private var targetNum: Int = 0                                  // 默认返回一条数据
    private var isLoading: AtomicBoolean = AtomicBoolean(false)     // 标记是否正在加载中

    private var hiAdPlan: AdPlans? = null
    private var psAdPlan: AdPlans? = null

    /*** 用于竞价计时*/
    private val mHandler by lazy {
        Handler(Looper.getMainLooper())
    }

    /*** 延时活动*/
    private val delayRunnable = Runnable {
        assemblyData("竞价时间到")
    }

    /**
     * Icon 广告加载对象
     */
    private var hiSavanaIconAdManager: BiddingHiSavanaIconAdManager? = null
    private var pSDistribution: BiddingPSDistributionProvider? = null

    /**
     * 展示Icon的View
     */
    private val mBiddingWrapperIconView = mutableSetOf<BiddingWrapperIconView>()

    /**
     * 埋点使用 链路关系
     */
    private var triggerId: String = ""
    private fun getTriggerId(): String = triggerId


    // ================================ 广告数据结果回调 ==============================================


    /**
     * PS 兜底
     */
    override fun onPSDistributionReady(data: MutableList<RecommendInfo>?) {
        super.onPSDistributionReady(data)
        AdLogger.logIcon(
            "${getClassTag()} --> onPSDistributionReady() --> mSceneId = ${getSceneId()} --> data.size = ${data?.size}",
            writeToFile = false
        )
        temporaryStoragePs.clear()
        data?.let {
            temporaryStoragePs.addAll(data)
        }
        // 其他数据是否回来了
        if (temporaryStorageHi.isNotEmpty()) {
            mHandler.post {
                assemblyData("数据都回来了 --> onPSDistributionReady")
            }
        }
    }

    override fun onMbIconShow(info: RecommendInfo?) {
        super.onMbIconShow(info)
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = psAdPlan?.id?: "",
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            //psId = info?.id,
            //psPackageName = info?.packageName,
            psRecommendInfo = info,
            sceneSubId = ""
        )

        // 保存已经展示标识
        valueList?.forEach {
            if (it.recommendInfo?.id == info?.id) {
                it.isDisplay = true
            }
        }

        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onBiddingWrapperAdDisplay() --> name = ${psAdPlan?.name}",
            writeToFile = false
        )
    }

    override fun onMbIconShowError(info: RecommendInfo?) {
        super.onMbIconShowError(info)

        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onIconShowError() --> name = ${psAdPlan?.name}",
            writeToFile = false
        )
    }

    override fun onMbIconClick(info: RecommendInfo?) {
        super.onMbIconClick(info)
        // 统一上报点击埋点
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = psAdPlan?.id?:"",
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            //psId = info?.id,
            //psPackageName = info?.packageName,
            psRecommendInfo = info,
            sceneSubId = ""
        )

        // ICON 点击事件需要特殊处理
        // 广告点击回调
        AdClickManager.onClick(null)

        // 本地检测没有安装 -- 打开PS半屏
        // 需要检测已安装的ps版本是否支持半屏，不支持的走全屏
        PsLinkUtils.adClick(info, isAutoDownload = true, scene = getSceneId() ?: "")

        // 回调给场景
        mCallback?.onMbIconClick(info)

        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onBiddingWrapperAdClick() --> name = ${psAdPlan?.name}",
            writeToFile = false
        )
    }

    override fun onBiddingBuyOutError(p0: TAdErrorCode?, maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingBuyOutError(p0, maxEcpmObject)
        AdLogger.logIcon("${getClassTag()} --> sceneId = ${getSceneId()} --> onBiddingBuyOutError() --> errorMessage = ${p0?.errorMessage}")
    }

    /**
     * 程序化
     */
    override fun onHiIconAdReady(tAdNativeInfos: MutableList<TAdNativeInfo>) {
        super.onHiIconAdReady(tAdNativeInfos)
        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onIconAdReady() --> tAdNativeInfos.size = ${tAdNativeInfos.size}",
            writeToFile = false
        )

        // 保存数据
        tAdNativeInfos.let {
            temporaryStorageHi.clear()
            temporaryStorageHi.addAll(it)
        }
        // 其他数据是否回来了
        if (temporaryStoragePs.isNotEmpty()) {
            mHandler.post {
                assemblyData("数据都回来了 --> onHiIconAdReady")
            }
        }
    }

    override fun onShow(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onShow(p0, p1)
        // 统一上报展示埋点
        AdReportProvider.display(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = hiAdPlan?.id?:"",
            adSource = MbAdSource.MB_AD_SOURCE_HISAVANA,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            //psId = null,
            //sceneSubId = ""
            psRecommendInfo = null
        )

        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onBiddingWrapperAdDisplay() --> name = ${hiAdPlan?.name}",
            writeToFile = false
        )
    }

    override fun onShowError(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onShowError(p0, p1)
        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onIconShowError() --> name = ${hiAdPlan?.name}, errorMessage = ${p0?.errorMessage}",
            writeToFile = false
        )
    }

    override fun onClick(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onClick(p0, p1)
        // 统一上报点击埋点
        AdReportProvider.adClick(
            triggerId = getTriggerId(),
            sceneId = getSceneId(),
            adPlanId = "",
            adSource = p0?.adSource,
            adId = "",
            adType = MbAdType.MB_AD_TYPE_ICON,
            //psId = null,
            psRecommendInfo = null,
            sceneSubId = ""
        )
        // 回调给场景
        mCallback?.onClick(p0, p1)

        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onBiddingWrapperAdClick() --> name = ${hiAdPlan?.name}",
            writeToFile = false
        )
    }

    override fun onLoadFailure(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onLoadFailure(p0, p1)
        mCallback?.onLoadFailure(p0, p1)
        AdLogger.logIcon("${getClassTag()} --> sceneId = ${getSceneId()} --> onLoadFailure() " +
                "--> placementId = ${p1.placementId} --> errorMessage = ${p0?.errorMessage}")
    }

    // =============================================================================================

    fun addBiddingWrapperIconView(iconView: BiddingWrapperIconView) {
        mBiddingWrapperIconView.add(iconView)
    }

    /**
     * 资源回收
     */
    fun destroy() {
        mHandler.removeCallbacksAndMessages(null)
        mCallback = null
        temporaryStorageHi.clear()
        temporaryStoragePs.clear()
        pSDistribution?.destroy()
        mBiddingWrapperIconView.forEach {
            it.destroy()
        }
        valueList?.clear()
        hiSavanaIconAdManager?.destroy()

        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> destroy() --> 资源回收",
            writeToFile = false
        )
    }

    fun getSceneId(): String? = mSceneId

    fun getListener(): BiddingTAdditionalListener = this

    fun getNativeAd(): TNativeAd? = hiSavanaIconAdManager?.getNativeAd()

    /**
     * 加载广告
     */
    suspend fun loadIconAd(
        sceneId: String,
        isFilteringInstalled: Boolean,
        targetNum: Int,
        listener: BiddingTAdditionalListener?
    ) {

        // 新用户判断
        if (NewUserShieldStrategy.isNewUser()) {
            return onFailCallback(listener = listener, errorMsg = "新用户保护期，不展示广告")
        }

        // 场景开关
        val errorMsg = SceneOnOff.isSceneOffV2(sceneId)
        if (TextUtils.isEmpty(errorMsg).not()) {
            onFailCallback(listener = listener, errorMsg = errorMsg)
            return
        }

        if (isLoading.get()) {
            return
        }
        isLoading.set(true)

        // 上报埋点
        triggerId = RandomUtils.getTriggerId()
        AdReportProvider.trigger(
            triggerId = getTriggerId(),
            sceneId = sceneId,
            adType = MbAdType.MB_AD_TYPE_ICON,
            adSource = MbAdSource.MB_AD_SOURCE_BUY_OUT,
            planId = "",
            sceneSubId = ""
        )

        mCallback = listener
        mSceneId = sceneId
        this.targetNum = targetNum

        withContext(Dispatchers.IO) {
            // 数据库获取场景计划
            val planList = AdPlansStorageManager.getAdPlan(
                sceneId = sceneId, ctxMap = getCtxMap(genre = null)
            )
            // 没有计划直接返回失败
            if (planList.isEmpty()) {
                return@withContext onFailCallback(
                    listener = listener,
                    errorMsg = "${getClassTag()} --> sceneId = $sceneId --> there are currently no plans available"
                )
            }

            // 加载广告
            innerLoadAd(sceneId, isFilteringInstalled, planList)
        }
    }

    // ===================================== 数据加载 ===============================================


    /**
     * 失败回调
     */
    private fun onFailCallback(listener: BiddingTAdditionalListener?, errorMsg: String) {
        AdLogger.logIcon(msg = errorMsg, level = Log.ERROR)
        listener?.onBiddingError(TAdErrorCode(MbAdSource.MB_AD_SOURCE_WRAPPER_AD, errorMsg))
    }

    /**
     * 获取优先级
     * ps hi
     */
    private fun isHiPriority(): Boolean {
        return TextUtils.equals(
            SceneStorage.getSceneConfig(getSceneId())?.get("priority")?.asString ?: "hi", "hi"
        )
    }

    /**
     * 加载广告
     */
    private suspend fun innerLoadAd(
        sceneId: String,
        isFilteringInstalled: Boolean,
        planList: MutableList<AdPlans>
    ) {
        val error = SceneOnOff.isSceneOffV2(sceneId)
        if (TextUtils.isEmpty(error).not()) {
            onFailCallback(listener = mCallback, errorMsg = error)
            return
        }

        val biddingTime = SceneCommonConfig.getBiddingTime(sceneId, 2)
        AdLogger.logIcon(
            msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> loadAd() --> 开始加载广告 --> biddingTime = $biddingTime",
            writeToFile = false
        )

        // 是否存在Hi程序化虚拟计划
        hiAdPlan = kotlin.runCatching {
            planList.first {
                it.adSource == AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_HI.value
            }
        }.getOrNull()

        // 场景开关判断
        if (hiAdPlan != null && SceneOnOff.isSceneHiOff(sceneId).not()) {
            // PS 接口和EW同时请求
            // 1. 加载HiSavana广告
            if (hiSavanaIconAdManager == null) {
                hiSavanaIconAdManager = BiddingHiSavanaIconAdManager()
                hiSavanaIconAdManager?.setSceneId(sceneId)
                hiSavanaIconAdManager?.setListener(this)
            }
            hiSavanaIconAdManager?.loadAd()
        } else {
            AdLogger.logIcon(
                msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> innerLoadAd() --> 程序化广告无计划或场景关闭",
                level = Log.WARN,
                writeToFile = false
            )
        }

        // 是否存在Icon广告计划
        psAdPlan = kotlin.runCatching {
            planList.first {
                it.adSource == AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_ICON.value
            }
        }.getOrNull()

        // 场景开关判断
        if (psAdPlan != null && SceneOnOff.isSceneNonOff(sceneId).not()) {
            // 2. 加载PS商单
            if (pSDistribution == null) {
                pSDistribution = BiddingPSDistributionProvider()
                pSDistribution?.setListener(this)
                pSDistribution?.setFilteringInstalled(isFilteringInstalled)
            }
            pSDistribution?.loadAd(scene = sceneId)
        } else {
            AdLogger.logIcon(
                msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> innerLoadAd() --> 包断广告无计划或场景关闭",
                level = Log.WARN,
                writeToFile = false
            )
        }

        // 3. 竞价时间是4秒
        mHandler.postDelayed(delayRunnable, biddingTime * 1000L)
    }

    // ================================== 处理数据 ===================================================

    /**
     * 结果回来之后进行展示
     *
     * 因为是并行发起两个请求，最多等待4秒之后开始展示数据。
     */
    private fun assemblyData(msg: String = "") {
        // 重置延时消息
        mHandler.removeCallbacksAndMessages(null)

        if (isLoading.get().not()) {
            return
        }
        isLoading.set(false)

        runCatching {
            // 组装数据
            val excessiveList = combineAndRetrieveData(isHiFirst = isHiPriority())

            val maxObject = excessiveList.firstOrNull()
            val maxAdPlan = when (maxObject?.type) {
                BiddingWrapperIconBean.WRAPPER_ICON_ITEM_HI -> hiAdPlan
                BiddingWrapperIconBean.WRAPPER_ICON_ITEM_PS -> psAdPlan
                else -> null
            }
            val ecpmList = excessiveList.map { it.ecpm }
            val message = if (excessiveList.isEmpty())
                "竞价失败 --> 无广告数据"
            else
                "竞价成功 --> ecpmList = $ecpmList --> max item: ecpm = ${maxObject?.ecpm}, plans.id = ${maxAdPlan?.id}, plans.name = ${maxAdPlan?.name}"
            AdReportProvider.biddingReport(
                triggerId = getTriggerId(),
                sceneId = getSceneId() ?: "",
                adType = MbAdType.MB_AD_TYPE_ICON,
                msg = message,
                result = if (excessiveList.isEmpty())
                    BiddingStateEnum.BIDDING_REPORT_BIDDING_FAIL
                else
                    BiddingStateEnum.BIDDING_REPORT_BIDDING_SUCCESS,
                ecpmList = ecpmList.toString(),
                ecpm = "${maxObject?.ecpm}",
                planId = "${maxAdPlan?.id}",
                planName = "${maxAdPlan?.name}",
                sceneSubId = ""
            )

            AdLogger.logIcon(
                msg = "${getClassTag()} --> sceneId = ${getSceneId()} --> onBiddingLoad() --> $message --> targetNum = $targetNum --> data.size = ${excessiveList.size}",
                writeToFile = false
            )

            val value = if (targetNum > 0) {
                excessiveList.take(targetNum).toMutableList()
            } else {
                excessiveList
            }

            valueList?.clear()
            valueList?.addAll(value)

            mCallback?.onWrapperIconReady(value)
        }
    }

    private fun getHi(hiItem: TAdNativeInfo): BiddingWrapperIconBean {
        return BiddingWrapperIconBean(
            type = BiddingWrapperIconBean.WRAPPER_ICON_ITEM_HI,
            recommendInfo = null,
            nativeInfo = hiItem,
            ecpm = hiItem.ecpmPrice
        )
    }

    private fun getPs(psItem: RecommendInfo): BiddingWrapperIconBean {
        return BiddingWrapperIconBean(
            type = BiddingWrapperIconBean.WRAPPER_ICON_ITEM_PS,
            recommendInfo = psItem,
            nativeInfo = null,
            ecpm = psAdPlan?.bidEcpmCent ?: 0.0
        )
    }

    private fun combineAndRetrieveData(isHiFirst: Boolean = false): MutableList<BiddingWrapperIconBean> {
        val result = mutableListOf<BiddingWrapperIconBean>()

        // 添加Hi icon
        temporaryStorageHi.forEach { hiItem ->
            result.add(getHi(hiItem))
        }

        // 添加PS icon
        temporaryStoragePs.forEach { psItem ->
            result.add(getPs(psItem))
        }

        result.sortWith(compareByDescending<BiddingWrapperIconBean> { it.ecpm }
                .thenBy { if (isHiFirst) it.type else -it.type }) // 价格相同时，Hi优先(isHiFirst=true)或PS优先

        temporaryStorageHi.clear()
        temporaryStoragePs.clear()

        return result
    }
}