package com.transsion.ad.ps

import android.os.SystemClock
import android.text.TextUtils
import android.util.Log
import com.blankj.utilcode.util.GsonUtils
import com.cloud.hisavana.sdk.common.util.MD5Utils
import com.tn.lib.net.manager.NetServiceGenerator
import com.tn.lib.net.utils.ParamHelper
import com.transsion.ad.MbAdContents
import com.transsion.ad.config.TestConfig
import com.transsion.ad.db.pslink.PsLinkAdPlan
import com.transsion.ad.log.AdLogger
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.model.MbAdImage
import com.transsion.ad.monopoly.plan.AdPlanMaterialManager
import com.transsion.ad.ps.model.PsLinkAdInfo
import com.transsion.ad.ps.model.PsLinkDto
import com.transsion.ad.scene.SceneGlobalConfig
import com.transsion.ad.strategy.AdMmkv
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.strategy.AdResDownloadManager
import com.transsion.ad.util.FileUtil
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONObject
import java.io.File

/**
 * @author: zhangxinbing
 * @date : 2025/3/19 17:50
 * @description: PS 接口请求
 */
internal object PSAdPlanRequestManager {

    private const val PAGE_SIZE = 10 // 每页数量
    private const val KEY_PS_LINK_REQUEST_GAP = "key_ps_link_request_gap" // PS 接口请求时间间隔

    private fun getClassTag(): String = javaClass.simpleName

    private val service by lazy { NetServiceGenerator.instance.getService(PsLinkApi::class.java) }


    // ====================================== 更新数据 ===============================================


    /**
     * 更新PS数据
     *
     * 1. 只要出发了 ad/config 就会请求PS分发接口
     * 2. PS分发接口 有请求间隔
     */
    suspend fun updatePsLink() {
        // 获取数据库的 AdSourcePS 计划 发起请求
        AdPlansStorageManager.getPlansBySource(AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS.value)
            ?.forEach { plan ->
                // PS 广告位ID
                var psScene = -1
                if (!TextUtils.isEmpty(plan.extAdSlot)) {
                    psScene = plan.extAdSlot?.toInt() ?: -1
                }
                if (psScene > 0) {
                    // 资源是否已经下载完成了
                    // 如果有资源 就需要判断请求间隔，没有资源直接请求。
                    if (AdPlanMaterialManager.hasPsOffer(plan)) {
                        val lastTimeStamp =
                            AdMmkv.getAdMMKV().getLong(KEY_PS_LINK_REQUEST_GAP + plan.id, 0)
                        val currentTime = SystemClock.elapsedRealtime()
                        val psLinkInterval =
                            SceneGlobalConfig.getPsLinkInterval() * 1000L // 间隔时间 单位秒
                        // 判断时间隔
                        if (currentTime - lastTimeStamp > psLinkInterval) {
                            // 触发PS的下载
                            getPsLinkListBySlot(0, 10, psScene = psScene, nonId = plan.id)
                            // 保存请求时间
                            AdMmkv.getAdMMKV()
                                .putLong(KEY_PS_LINK_REQUEST_GAP + plan.id, currentTime)
                        } else {
                            AdLogger.logPS(
                                "${getClassTag()} --> updatePsLink() --> ID = ${plan.id} --> name = ${plan.name} --> psScene = $psScene --> 当前有资源，且在请求间隔内 --> 不做处理",
                                level = Log.WARN
                            )
                        }
                    } else {
                        // 触发PS的下载
                        getPsLinkListBySlot(0, 10, psScene = psScene, nonId = plan.id)
                        // 保存请求时间
                        AdMmkv.getAdMMKV().putLong(
                            KEY_PS_LINK_REQUEST_GAP + plan.id, SystemClock.elapsedRealtime()
                        )
                    }
                }
            }

        // 每个类型的处理完成之后，删除无效资源
        PsStorageManager.deleteInvalidFile()
        AdLogger.logPS("${getClassTag()} --> analyzeData() --> 删除无效素材文件")
    }


    // ===================================== PS 接口请求 ============================================


    /**
     * 请求参数
     */
    private fun getParams(pageIndex: Int, pageSize: Int, scene: Int?): JSONObject {
        // 请求参数组装
        val jsonObject = JSONObject()
        jsonObject.put("bu", "mb")
        jsonObject.put("pageIndex", pageIndex) // 页码，从0开始
        jsonObject.put("pageSize", pageSize) // 每页数量，默认10
        jsonObject.put("media", 1)// 固定值
        jsonObject.put("scene", scene)// 下图 场景代码

        // 通用数据
        val commonParams = PsLinkApi.commonParams(System.currentTimeMillis())
        commonParams.forEach {
            jsonObject.put(it.key, it.value)
        }

        // 返回
        return jsonObject
    }

    /**
     * 发起请求
     */
    suspend fun getPsLinkListBySlot(
        pageIndex: Int = 0,
        pageSize: Int = PAGE_SIZE,
        psScene: Int?,
        nonId: String,
    ) {

        // 埋点
        PSReportUtil.reportPsLinkRequest(
            psState = PSReportUtil.PSRequestState.REQUEST_TRIGGER,
            count = null,
            errorMsg = null,
            psScene = psScene.toString(),
            scene = PSReportUtil.PsDistributeSceneEnum.PS_DISTRIBUTE_SCENE_AD_PLAN.value
        )

        withContext(Dispatchers.IO) {
            kotlin.runCatching {
                // 获取请求参数
                val jsonObject = getParams(pageIndex, pageSize, psScene)
                val url =
                    if (TestConfig.isReleasePsApi()) "https://feature-api.palmplaystore.com" else "https://test-feature-api.palmplaystore.com"
                AdLogger.logPS("${getClassTag()} --> getPsLinkListBySlot() --> psScene = $psScene --> url = $url --> jsonObject = $jsonObject")

                // 请求
                val postPsRecommendList =
                    service.postPsCandidateList(ParamHelper.createBody(jsonObject.toString()), url)

                // PS 商单结果 埋点
                PSReportUtil.reportPsLinkRequest(
                    psState = PSReportUtil.PSRequestState.REQUEST_SUCCESS,
                    count = postPsRecommendList?.data?.size,
                    errorMsg = "",
                    psScene = psScene.toString(),
                    scene = PSReportUtil.PsDistributeSceneEnum.PS_DISTRIBUTE_SCENE_AD_PLAN.value
                )

                // 解析数据
                analyzeData(psScene.toString(), nonId, postPsRecommendList)

            }.getOrElse {
                AdLogger.logPS(
                    "${getClassTag()} --> getPsLinkList() --> psScene = $psScene --> it = $it",
                    level = Log.ERROR
                )

                // Ps 商单请求失败
                PSReportUtil.reportPsLinkRequest(
                    psState = PSReportUtil.PSRequestState.REQUEST_FAIL,
                    count = null,
                    errorMsg = "${it.message}",
                    scene = psScene.toString(),
                    psScene = PSReportUtil.PsDistributeSceneEnum.PS_DISTRIBUTE_SCENE_AD_PLAN.value
                )
            }
        }
    }


    // ================================ 数据解析 =====================================================


    /**
     * 新版本数据处理
     */
    private suspend fun analyzeData(
        sceneStr: String, adPlanId: String, postPsRecommendList: PsLinkDto?
    ) {

        // 输出请求结果
        AdLogger.logPS("${getClassTag()} --> analyzeData() --> PS接口返回 --> psScene = $sceneStr --> adPlanId = $adPlanId --> data = ${postPsRecommendList?.data?.size} --> 开始增量更新")

        // 通过ID 获取数据库中是否存在相同ID的数据
        val ids = mutableListOf<Int>()
        postPsRecommendList?.data?.forEach { recommendInfo ->
            //如果不支持pslink 且没有gplink链接 过滤掉商单
            if (!PsLinkUtils.isSupportPsLink() && TextUtils.isEmpty(recommendInfo.gpLink)) {
                return@forEach
            }

            // 保存数据
            val id = (sceneStr + recommendInfo.id).hashCode()
            ids.add(id)

            // 下载素材+创建数据库对象
            val psLinkAdPlan = getPsLinkAdPlan(sceneStr, adPlanId, recommendInfo)

            // 更新数据库
            psLinkAdPlan?.let {
                PsDbManager.insertPsLinkAd(it)
            }
        }

        // 删除数据库无效数据
        PsDbManager.deleteNotInIds(ids)
        AdLogger.logPS("${getClassTag()} --> analyzeData() --> 删除数据库无效Offer")
    }

    /**
     * 下载素材 组装数据库对象
     */
    private suspend fun getPsLinkAdPlan(
        sceneStr: String, adPlanId: String, recommendInfo: RecommendInfo
    ): PsLinkAdPlan? {

        // 触发埋点
        PSReportUtil.reportMaterialState(
            psState = PSReportUtil.PSRequestState.REQUEST_TRIGGER,
            recommendInfo = recommendInfo,
            sceneStr = sceneStr,
            adPlanId
        )

        val itemDetail = recommendInfo.detail
        val adImage = MbAdImage() // 图片素材对象
        adImage.url = recommendInfo.showContent ?: itemDetail?.img0 // 图片素材地址

        // 下载素材
        val result = downloadImage(adImage)
        return if (result) {
            val psLinkAdInfo = PsLinkAdInfo(
                advertiserName = recommendInfo.name,
                advertiserAvatar = recommendInfo.iconUrl,
                advertiserAvatarPath = downloadAdvertiserAvatar(recommendInfo.iconUrl),
                title = itemDetail?.name,
                desc = itemDetail?.simpleDescription,
                buttonText = recommendInfo.buttonText, // 广告按钮文案
                url = adImage.url,
                path = adImage.path,
            )
            val id = (sceneStr + recommendInfo.id).hashCode()
            val adPlan = PsLinkAdPlan(
                id = id,
                nonId = adPlanId,
                adSource = AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS.value,
                extAdSlot = sceneStr,
                rank = 0,
                psPlanId = recommendInfo.id.toString(),
                psLinkAdInfoStr = GsonUtils.toJson(psLinkAdInfo),
                psInfoJson = GsonUtils.toJson(recommendInfo),
                updateTimestamp = System.currentTimeMillis(), // 优先展示列表第一个数据

                // 新增字段
                showMax = recommendInfo.showMax,
                clickMax = recommendInfo.clickMax,
                showHours = recommendInfo.showHours,
            )

            // 获取本地数据库数据,用本地已经产生的数据覆盖接口返回数据
            val localPsLinkAdPlan = PsDbManager.getPsLinkAdById(id)
            if (localPsLinkAdPlan == null) {
                AdLogger.logPS("${getClassTag()} --> getPsLinkAdPlan() --> name = ${recommendInfo.detail?.name} --> 数据库 不存在 --> 直接保存")
            } else {
                AdLogger.logPS("${getClassTag()} --> getPsLinkAdPlan() --> name = ${recommendInfo.detail?.name} --> 数据库 存在、替换数据再保存 --> showedTimes = ${localPsLinkAdPlan.showedTimes} --> clickedTimes = ${localPsLinkAdPlan.clickedTimes} --> showDate = ${localPsLinkAdPlan.showDate}")
                adPlan.showedTimes = localPsLinkAdPlan.showedTimes
                adPlan.clickedTimes = localPsLinkAdPlan.clickedTimes
                adPlan.showDate = localPsLinkAdPlan.showDate
            }

            // 下载成功埋点
            PSReportUtil.reportMaterialState(
                psState = PSReportUtil.PSRequestState.REQUEST_SUCCESS,
                recommendInfo = recommendInfo,
                sceneStr = sceneStr,
                adPlanId
            )

            // 结果对象
            adPlan
        } else {
            // 下载失败埋点
            PSReportUtil.reportMaterialState(
                psState = PSReportUtil.PSRequestState.REQUEST_FAIL,
                recommendInfo = recommendInfo,
                sceneStr = sceneStr,
                adPlanId = adPlanId
            )
            null
        }
    }


    // ================================ 下载素材 =====================================================


    /**
     * 下载图片素材
     */
    private fun downloadImage(mbAdImage: MbAdImage): Boolean {
        // 隐藏后缀 .mp4这样的
        val mineType = MbAdContents.MINE_TYPE
        val url = mbAdImage.url
        val fileName = MD5Utils.toMd5(url)
        val destination =
            MbAdContents.NON_AD_PS_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"
        val file = File(destination)
        //下载图类的  可能被压缩 判断条件不一致
        if (file.isFile && file.exists() && file.length() > 0) {
            //不需要重新下载 copy一次 都处理完后可删除old文件
            mbAdImage.path = destination
            return true
        } else {
            val downloadFileSuccess = AdResDownloadManager.downloadFile(url, destination)
            // size check
            val localFileSize = file.length()
            if (downloadFileSuccess && localFileSize > 0) {
                mbAdImage.path = destination
                return true
            } else {
                // 下载失败不处理
                AdLogger.logPS(
                    "${getClassTag()} --> downloadAdPlan() --> 图片素材下载失败 --  downloadFileSuccess = $downloadFileSuccess -- localFileSize = $localFileSize -- destination = $destination",
                    level = Log.ERROR
                )
                return false
            }
        }
    }

    /**
     * 下载广告主图标
     */
    private fun downloadAdvertiserAvatar(advertiserAvatar: String?): String {
        // 文件本地命名
        val fileName = MD5Utils.toMd5(advertiserAvatar)
        val mineType = MbAdContents.MINE_TYPE

        // 文件本地存储路径
        val destination =
            MbAdContents.NON_AD_PS_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"

        // 判断一下文件是否已经存在
        if (FileUtil.isFileExists(destination)) {
            return destination
        }

        // 下载文件
        val downloadFile = AdResDownloadManager.downloadFile(advertiserAvatar, destination)
        return if (downloadFile) {
            destination
        } else {
            ""
        }
    }
}