package com.transsin.networkmonitor

import android.os.SystemClock
import com.transsin.networkmonitor.Consts.FAILURE
import com.transsin.networkmonitor.Consts.SUCCESS
import com.transsin.networkmonitor.ErrorCode.CODE_CANCEL
import com.transsin.networkmonitor.ErrorCode.CODE_FAKE_NETWORK
import com.transsin.networkmonitor.ErrorCode.CODE_NO_NETWORK
import com.transsin.networkmonitor.ErrorCode.CODE_UNKNOWN
import com.transsin.networkmonitor.ErrorCode.MSG_FAKE_NETWORK
import com.transsin.networkmonitor.ErrorCode.MSG_NETWORK_DISCONNECT
import com.transsin.networkmonitor.ErrorCode.MSG_UNKNOWN
import okhttp3.*
import okhttp3.EventListener
import java.io.*
import java.lang.Long.max
import java.net.*
import java.util.*
import javax.net.ssl.SSLHandshakeException

/**
 * @Author:huangxiushuo
 * @Date: 2022-07-22 17:01
 * @Description:
 * @param notReportData 表示数据是不需要上报的，一般外部是测试环境的时候会传true
 * @param cronetFilter 表示如果用拦截器的方式实现了Cronet请求需要过滤Cronet的请求产生的脏数据
 */
internal class MonitorEventListener(
    private val listener: EventListener?,
    private val usage: Int,
    private val notReportData: Boolean,
    channel: String? = "All",
    cronetFilter: Boolean,
    private val abTag: String? = null,
    private val notReportCancel: Boolean,
    private val notReportNoNetwork: Boolean
) : EventListener(), NetWorkListener {

    private var mCallStartTime = 0L
    private var mDnsStartTime = 0L
    private var mSecureConnectStartTime = 0L
    private var mTcpStartTime = 0L
    private var mReqHeadersStartTime = 0L
    private var mReqHeadersEndTime = 0L
    private var mReqBodyEndTime = 0L
    private var mRespHeadersStartTime = 0L
    private val mMonitorData = MonitorData.create(usage, channel)
    /**
     * 标识实际不上报的数据
     * notReportData || cronetFilter表示如果标识notReportData或者开启了Croent请求过滤都不上报数据
     * @see onDowngrade 表示这次请求是Cronet降级请求,不参与Cronet过滤，需要改变此值
     */
    private var realNotReport = notReportData || cronetFilter
    private var mRespHeadersEndTime = 0L
    private var httpCode = CODE_UNKNOWN
    private var errorCode = CODE_UNKNOWN
    private var errorMsg = MSG_UNKNOWN
    private val randomNumber = (1..10).random()

    override fun callStart(call: Call) {
        super.callStart(call)
        "[callStart]".printEvent()
        try {
            //后续统一扩展字段
            val isDownload = call.request().headers().get("isDownload")
            isDownload?.let {
                mMonitorData.usage = if (it == "true" || it == "T") 1 else 0
            }
            val offlineAd = call.request().headers().get("offlineAd")
            offlineAd?.let { mMonitorData.offlineAd = it.toInt() }
            val host = call.request().url().host()
            val path = call.request().url().encodedPath()
            mMonitorData.path = path ?: ""
            mMonitorData.host = host ?: ""
            val url = (call.request().url().toString())
            mMonitorData.completeApi = if (randomNumber <= 2) {
                url
            } else {
                ""
            }
            mMonitorData.serverApi = if (usage == 0 && url.contains("?")) {
                url.substring(0, url.indexOf("?"))
            } else url
            val tag = call.request().headers().get("infoeyes-tag")
            mMonitorData.compareTag = tag ?: abTag.orEmpty()
            mMonitorData.step = 0
        } catch (e: Exception) {
            "[get header exception] ${e::class.java.simpleName} : ${e.message}".printErrEvent()
        }
        mCallStartTime = realtime()
        listener?.callStart(call)
    }


    override fun dnsStart(call: Call, domainName: String) {
        super.dnsStart(call, domainName)
        "[dnsStart] domainName: $domainName".printEvent()
        mDnsStartTime = realtime()
        mMonitorData.step = 1
        listener?.dnsStart(call, domainName)
    }

    override fun dnsEnd(call: Call, domainName: String, inetAddressList: List<InetAddress>) {
        super.dnsEnd(call, domainName, inetAddressList)
        val addresses = inetAddressList.joinToString(
            separator = ",",
            prefix = "[",
            postfix = "]"
        )
        "[dnsEnd] inetAddressList $addresses".printEvent()
        mMonitorData.dnsTime = realtimeDiff(mDnsStartTime)
        listener?.dnsEnd(call, domainName, inetAddressList)
    }

    override fun connectStart(call: Call, inetSocketAddress: InetSocketAddress, proxy: Proxy) {
        super.connectStart(call, inetSocketAddress, proxy)
        mTcpStartTime = realtime()
        mMonitorData.ipAddr = inetSocketAddress.address.hostAddress.orEmpty()
        "[connectStart] inetSocketAddress : $inetSocketAddress, proxy : $proxy".printEvent()
        mMonitorData.step = 2
        listener?.connectStart(call, inetSocketAddress, proxy)
    }

    override fun secureConnectStart(call: Call) {
        super.secureConnectStart(call)
        "[secureConnectStart]".printEvent()
        mMonitorData.step = 3
        mSecureConnectStartTime = realtime()
        listener?.secureConnectStart(call)
    }

    override fun secureConnectEnd(call: Call, handshake: Handshake?) {
        super.secureConnectEnd(call, handshake)
        "[secureConnectEnd]".printEvent()
        mMonitorData.tlsVersion = handshake?.tlsVersion()?.name ?: ""
        mMonitorData.sslTime = realtimeDiff(mSecureConnectStartTime)
        listener?.secureConnectEnd(call, handshake)
    }

    override fun connectEnd(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?
    ) {
        super.connectEnd(call, inetSocketAddress, proxy, protocol)
        "[connectEnd] inetSocketAddress $inetSocketAddress".printEvent()

        mMonitorData.tcpTime = realtimeDiff(mTcpStartTime)
        mMonitorData.protocol = protocol?.name ?: ""
        listener?.connectEnd(call, inetSocketAddress, proxy, protocol)
    }

    override fun connectFailed(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?,
        ioe: IOException
    ) {
        super.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
        "[connectFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        mMonitorData.protocol = protocol?.name ?: ""
        listener?.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
    }

    override fun connectionReleased(call: Call, connection: Connection) {
        super.connectionReleased(call, connection)
        "[connectionReleased]".printEvent()
        listener?.connectionReleased(call, connection)
    }

    override fun connectionAcquired(call: Call, connection: Connection) {
        super.connectionAcquired(call, connection)
        "[connectionAcquired] connection: ${connection}".printEvent()
        mMonitorData.tlsVersion = connection.handshake()?.tlsVersion()?.name ?: ""
        mMonitorData.protocol = connection.protocol().name
        listener?.connectionAcquired(call, connection)
    }

    override fun requestHeadersStart(call: Call) {
        super.requestHeadersStart(call)
        "[requestHeadersStart]".printEvent()
        mReqHeadersStartTime = realtime()
        mMonitorData.step = 4
        listener?.requestHeadersStart(call)
    }

    override fun requestHeadersEnd(call: Call, request: Request) {
        super.requestHeadersEnd(call, request)
        "[requestHeadersEnd] request :${request}".printEvent()
        mReqHeadersEndTime = realtime()
        mMonitorData.sendTime = mReqHeadersEndTime - mReqHeadersStartTime
        listener?.requestHeadersEnd(call, request)
    }

    //may not call
    override fun requestBodyStart(call: Call) {
        super.requestBodyStart(call)
        "[requestBodyStart]".printEvent()
        mMonitorData.step = 5
        listener?.requestBodyStart(call)
    }

    //may not call
    override fun requestBodyEnd(call: Call, byteCount: Long) {
        super.requestBodyEnd(call, byteCount)
        "[requestBodyEnd] byteCount: ${byteCount}".printEvent()
        mReqBodyEndTime = realtime()
        mMonitorData.run {
            reqBodySize = byteCount
            sendTime = mReqBodyEndTime - mReqHeadersStartTime
            step = 6
        }

        listener?.requestBodyEnd(call, byteCount)
    }

    override fun requestFailed(call: Call, ioe: IOException) {
        super.requestFailed(call, ioe)
        "[requestFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        listener?.requestFailed(call, ioe)
    }

    override fun responseHeadersStart(call: Call) {
        super.responseHeadersStart(call)
        "[responseHeadersStart]".printEvent()
        mRespHeadersStartTime = realtime()
        mMonitorData.waitTime = mRespHeadersStartTime - max(mReqBodyEndTime, mReqHeadersEndTime)
        mMonitorData.step = 7
        listener?.responseHeadersStart(call)
    }

    override fun responseHeadersEnd(call: Call, response: Response) {
        super.responseHeadersEnd(call, response)
        "[responseHeadersEnd] code : ${response.code()}".printEvent()
        val provider = response.headers().get("x-response-cdn") ?: ""
        val serverTime = (response.headers().get("server-time") ?: "0").toLong()
        provider.printEvent()
        mMonitorData.cdnProvider = provider
        mRespHeadersEndTime = realtime()
        mMonitorData.recTime = realtimeDiff(mRespHeadersStartTime)
        mMonitorData.cdnReqTime = realtimeDiff(mCallStartTime)
        mMonitorData.serverTime = serverTime
        mMonitorData.contentEncoding = response.headers().get("content-encoding") ?: ""
        httpCode = response.code()
        listener?.responseHeadersEnd(call, response)
    }

    //may not call
    override fun responseBodyStart(call: Call) {
        super.responseBodyStart(call)
        "[responseBodyStart]".printEvent()
        mMonitorData.step = 8
        listener?.responseBodyStart(call)
    }

    //may not call
    override fun responseBodyEnd(call: Call, byteCount: Long) {
        super.responseBodyEnd(call, byteCount)
        "[responseBodyEnd] byteCount: ${byteCount}".printEvent()
        mMonitorData.run {
            resBodySize = byteCount
            recTime = realtimeDiff(mRespHeadersStartTime)
            cdnDownTime = realtimeDiff(mRespHeadersEndTime)
            cdnSpeed = "${(byteCount / 1024f) / (cdnDownTime / 1000f)}"
        }
        mMonitorData.step = 9
        listener?.responseBodyEnd(call, byteCount)
    }

    override fun responseFailed(call: Call, ioe: IOException) {
        super.responseFailed(call, ioe)
        "[responseFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        listener?.responseFailed(call, ioe)
    }

    override fun callEnd(call: Call) {
        super.callEnd(call)
        "[callEnd]".printEvent()
        mMonitorData.run {
            callResultCode = SUCCESS
            reqTime = realtimeDiff(mCallStartTime)
            errorCode = httpCode
            step = 10
            //请求成功  1. errorCode = -1，命中缓存 2. 304命中缓存
            if (errorCode != 304 && !realNotReport) {
                track()
            }
        }
        listener?.callEnd(call)
    }

    /**
     * [IOException]
     * -[UnknownHostException]
     * -[SSLException]
     * --[SSLHandshakeException]
     * -[SocketTimeoutException]
     */
    override fun callFailed(call: Call, ioe: IOException) {
        super.callFailed(call, ioe)
        "[callFailed]  ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        mMonitorData.callResultCode = FAILURE
        if (isNetDone()) {
            // 无网
            if (!NetworkMonitor.isNetworkConnected) {
                errorCode = CODE_NO_NETWORK
                errorMsg = MSG_NETWORK_DISCONNECT
            }
            // 假网
            else if (NetworkMonitor.isFakeNetwork) {
                errorCode = CODE_FAKE_NETWORK
                errorMsg = MSG_FAKE_NETWORK
            } else {
                errorCode = CodeUtil.fetchCode(httpCode, ioe)
            }
        } else {
            errorCode = CodeUtil.fetchCode(httpCode, ioe)
        }


        mMonitorData.errorCode = errorCode
        mMonitorData.errorMsg = if (errorMsg == MSG_UNKNOWN) ioe.shortMsg() else errorMsg
        if (!realNotReport) {
            var isTrack = true
            // notReportCancel开启时，errorCode是CODE_CANCEL时不上报
            if (notReportCancel && errorCode == CODE_CANCEL) {
                isTrack = false
            }
            // notReportNoNetwork开启时，errorCode是MSG_NETWORK_DISCONNECT时不上报
            if (notReportNoNetwork && errorCode == CODE_NO_NETWORK){
                isTrack = false
            }
            if (isTrack) {
                mMonitorData.track()
            }
        }
        listener?.callFailed(call, ioe)
    }

    private fun isNetDone(): Boolean {
        if (NetworkMonitor.isNetInitDone || SystemClock.uptimeMillis() - NetworkMonitor.initStartTime > 3000) {
            return true
        }
        return false
    }

    override fun onRequestUpdate(request: Request) {
        "[onRequestUpdate]".printEvent()
        runCatching {
            val host = request.url().host()
            val path = request.url().encodedPath()
            mMonitorData.path = path ?: ""
            mMonitorData.host = host ?: ""
            val url = (request.url().toString())
            mMonitorData.completeApi = if (randomNumber <= 2) {
                url
            } else {
                ""
            }
            mMonitorData.serverApi = if (usage == 0 && url.contains("?")) {
                url.substring(0, url.indexOf("?"))
            } else url
        }
    }

    override fun onDowngrade(request: Request, errorCode: Int) {
        "[onDowngrade]".printEvent()
        /**
         * 如果notReportData是false(不是非上报数据),则设置realNotReport为false(上报数据)
         */
        if (!notReportData) {
            realNotReport = false
        }
        mMonitorData.requestDowngrade = errorCode
    }
}