package com.transsin.networkmonitor

import android.os.SystemClock
import com.transsin.networkmonitor.ErrorCode.CODE_CANCEL
import com.transsin.networkmonitor.ErrorCode.CODE_NO_NETWORK
import com.transsin.networkmonitor.ErrorCode.CODE_UNKNOWN
import okhttp3.Headers
import org.chromium.net.NetworkException
import org.chromium.net.RequestFinishedInfo
import java.net.URL

/**
 * @Author lihao
 * @Time 2024/3/8 12:07
 * @Company Transsion
 * desc
 */
object CronetMonitorHelper {

    /**
     * cronet日志上报
     * requestInfo cronet返回的RequestFinishedInfo
     * usage 使用场景 0：API请求， 1：CDN
     * channel 业务方渠道标识
     * tag  也务ab tag
     * isTest 是否测试环境
     * notReportCancel 是否不上报取消的请求
     * notReportNoNetwork 是否不上报无网的请求
     */
    fun logMatrix(requestInfo: RequestFinishedInfo, usage: Int, channel: String, tag: String,
                  isTest: Boolean, notReportCancel: Boolean = false, notReportNoNetwork: Boolean = false) {
        //notReportCancel开启时，errorCode是CODE_CANCEL时不上报
        if (notReportCancel && requestInfo.finishedReason == RequestFinishedInfo.CANCELED) {
            return
        }

        val protocol = requestInfo.responseInfo?.negotiatedProtocol ?: ""
        val monitorData = MonitorData.create(usage, channel)
        monitorData.step = 0
        monitorData.httpType = 2
        monitorData.protocol = protocol
        var errorCode = CODE_UNKNOWN
        val httpCode = requestInfo.responseInfo?.httpStatusCode ?: CODE_UNKNOWN

        val matrix = requestInfo.metrics
        val randomNumber = (1..10).random()
        val url = requestInfo.url.toString()
        monitorData.completeApi = if (randomNumber <= 2) {
            url
        } else ""
        try {
            val header = (requestInfo.annotations.find { it is Headers } as? Headers)?.apply {
                get("isDownload")?.let {
                    monitorData.usage = if (it == "true" || it == "T") 1 else 0
                }
                monitorData.offlineAd = if (get("offlineAd") == "1") 1 else 0
            }
            monitorData.compareTag = header?.get("infoeyes-tag") ?: tag
            monitorData.serverApi = if (monitorData.usage == 0 && url.contains("?")) {
                url.substring(0, url.indexOf("?"))
            } else url
            val host = URL(url).host
            val path = URL(url).path
            monitorData.host = host
            monitorData.path = path
        } catch (e: Exception) {
            e.message?.printEvent()
        }

        val success = requestInfo.finishedReason == RequestFinishedInfo.SUCCEEDED
        if (success) {
            monitorData.callResultCode = Consts.SUCCESS
            // dns time
            val dnsStart = matrix.dnsStart?.time ?: 0
            val dnsEnd = matrix.dnsEnd?.time ?: 0
            if (dnsStart > 0) {
                monitorData.step = 1
            }
            monitorData.dnsTime = dnsEnd - dnsStart
            // connect time
            val connectStart = matrix.connectStart?.time ?: 0
            val connectEnd = matrix.connectEnd?.time ?: 0
            monitorData.tcpTime = connectEnd - connectStart
            if (connectStart > 0) {
                monitorData.step = 2
            }
            // ssl time
            val sslStart = matrix.sslStart?.time ?: 0
            val sslEnd = matrix.sslEnd?.time ?: 0
            monitorData.sslTime = sslEnd - sslStart
            if (sslStart > 0) {
                monitorData.step = 3
            }
            // send time
            val sendStart = matrix.sendingStart?.time ?: 0
            val sendEnd = matrix.sendingEnd?.time ?: 0
            monitorData.sendTime = sendEnd - sendStart
            if (sendStart > 0) {
                monitorData.step = 5
            }
            if (sendEnd > 0) {
                monitorData.step = 6
            }
            // rec time
            val responseStart = matrix.responseStart?.time ?: 0
            val responseEnd = matrix.requestEnd?.time ?: 0
            if (responseStart > 0) {
                monitorData.step = 7
            }
            if (responseEnd > 0) {
                monitorData.step = 10
            }
            monitorData.recTime = responseEnd - responseStart
            // wait time
            monitorData.waitTime = responseStart - sendEnd
            // reqBodySize  resBodySize
            monitorData.resBodySize = matrix.receivedByteCount ?: 0
            monitorData.reqBodySize = matrix.sentByteCount ?: 0

            val allHeaders = requestInfo.responseInfo?.allHeaders

            // 获取"Content-Encoding"头部
            val contentEncodings = allHeaders?.get("Content-Encoding")
            if (contentEncodings != null) {
                for (contentEncoding in contentEncodings) {
                    monitorData.contentEncoding = contentEncoding
                    break
                }
            }
            monitorData.serverTime = (allHeaders?.get("server-time")?.get(0) ?: "0").toLong()
            monitorData.reqTime = matrix?.totalTimeMs ?: 0

            if (monitorData.usage == 1) {
                val provider = allHeaders?.get("x-response-cdn")?.get(0) ?: ""
                monitorData.cdnProvider = provider
                monitorData.cdnReqTime = responseStart - (matrix?.requestStart?.time ?: 0)
                monitorData.cdnDownTime = monitorData.recTime
                monitorData.cdnSpeed = "${(monitorData.resBodySize / 1024f) / (monitorData.recTime / 1000f)}"
            }

            errorCode = httpCode
        } else {
            monitorData.callResultCode = Consts.FAILURE
            val isCancel = requestInfo.finishedReason == RequestFinishedInfo.CANCELED
            var errorMsg = requestInfo.exception?.message.orEmpty()
            if (requestInfo.exception is NetworkException) {
                val ioe: NetworkException = requestInfo.exception as NetworkException
                if (isNetDone()) {
                    // 无网
                    if (!NetworkMonitor.isNetworkConnected) {
                        errorCode = CODE_NO_NETWORK
                        errorMsg = ErrorCode.MSG_NETWORK_DISCONNECT
                    }
                    // 假网
                    else if (NetworkMonitor.isFakeNetwork) {
                        errorCode = ErrorCode.CODE_FAKE_NETWORK
                        errorMsg = ErrorCode.MSG_FAKE_NETWORK
                    } else {
                        errorCode = CodeUtil.fetchCronetException(httpCode, ioe, isCancel)
                    }
                } else {
                    errorCode = CodeUtil.fetchCronetException(httpCode, ioe, isCancel)
                }
            } else {
                errorCode = if (isCancel) {
                    CODE_CANCEL
                } else {
                    CODE_UNKNOWN
                }
            }
            monitorData.errorMsg = errorMsg

        }
        monitorData.errorCode = errorCode
        // 上报athena
        if(!isTest){
            if (errorCode == 304){
                return
            }
            //notReportNoNetwork开启时，无网时不上报
            if (notReportNoNetwork && errorCode == CODE_NO_NETWORK) {
                return
            }
            monitorData.track()
        }
    }

    private fun isNetDone(): Boolean {
        return NetworkMonitor.isNetInitDone || SystemClock.uptimeMillis() - NetworkMonitor.initStartTime > 3000
    }
}