package com.transsin.networkmonitor

import android.Manifest
import android.annotation.SuppressLint
import android.app.Application
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.ConnectivityManager.NetworkCallback
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkInfo
import android.net.NetworkRequest
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.os.SystemClock
import android.telephony.TelephonyManager
import android.util.Log
import androidx.core.app.ActivityCompat
import androidx.core.os.bundleOf
import com.transsin.networkmonitor.Consts.APP_ID
import com.transsin.networkmonitor.Consts.REPORT_ID
import com.transsin.networkmonitor.config.NetworkConfigManager
import com.transsin.networkmonitor.stat.DataMonitorHelper
import com.transsin.util.GlobalUtils
import com.transsin.util.io
import java.util.concurrent.atomic.AtomicBoolean


/**
 * @Author:huangxiushuo
 * @Date: 2023-05-17 10:46
 * @Description:
 */
object NetworkMonitor {


    const val TAG = "NetworkMonitor"
    internal var isTest: Boolean = false
    internal var isNetworkConnected: Boolean = false
    internal var isFakeNetwork: Boolean = false
    internal var isNetInitDone: Boolean = false
    internal var initStartTime: Long = 0
    internal var isInit = AtomicBoolean(false)

    /**
     * NetworkMonitor初始化方法
     * @param application
     * @param hosts 需要监控的域名
     * @param isTest 是否测试
     */
    @JvmStatic
    fun init(application: Application, hosts: Array<String> = arrayOf(), isTest: Boolean = false) {
        try {
            if (isInit.compareAndSet(false, true)) {
                this.isTest = isTest
                NetworkConfigManager.init(application, isTest)
                DataMonitorHelper.init(application, REPORT_ID.toString())
                HostMatcher.setHosts(hosts)
                registerNetworkCallback(application)
                initStartTime = SystemClock.uptimeMillis()
                trackNetworkState()
                registerNetworkReceiver(application)
            }
        } catch (t: Throwable) {
            t.printStackTrace()
        }
    }

    @JvmStatic
    fun init(application: Application, isTest: Boolean = false) {
        this.init(application, arrayOf(), isTest)
    }

    private val mReceiver: BroadcastReceiver = object : BroadcastReceiver() {
        override fun onReceive(c: Context, intent: Intent) {
            try {
                io().execute {
                    GlobalUtils.currentNet = parseNetworkType(c)
                }
            } catch (t: Throwable) {
                t.printStackTrace()
            }
        }
    }

    private fun registerNetworkCallback(context: Context) {
        try {
            val connectivityManager =
                context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager ?: return
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                val request = NetworkRequest.Builder().build()
                connectivityManager.registerNetworkCallback(request, object : NetworkCallback() {
                    override fun onCapabilitiesChanged(
                        network: Network,
                        networkCapabilities: NetworkCapabilities
                    ) {
                        super.onCapabilitiesChanged(network, networkCapabilities)
                        val isNetworkValid =
                            networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
                        isNetworkConnected =
                            networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
                        isFakeNetwork = isNetworkConnected && !isNetworkValid
                        isNetInitDone = true
                        if (!isNetworkConnected) {
                            GlobalUtils.currentNet = 99
                        }
                        Log.d(
                            Utils.GLOBAL_TAG,
                            "isNetworkConnected : ${isNetworkConnected},  isFakeNetwork : $isFakeNetwork"
                        )
                    }

                    override fun onAvailable(network: Network) {
                        super.onAvailable(network)
                        Log.d(Utils.GLOBAL_TAG, "onAvailable : ${network}")
                    }

                    override fun onUnavailable() {
                        super.onUnavailable()
                        Log.d(Utils.GLOBAL_TAG, "onUnavailable")
                    }

                    override fun onBlockedStatusChanged(network: Network, blocked: Boolean) {
                        super.onBlockedStatusChanged(network, blocked)
                        Log.d(Utils.GLOBAL_TAG, "onBlockedStatusChanged")
                    }

                    override fun onLost(network: Network) {
                        super.onLost(network)
                        Log.d(Utils.GLOBAL_TAG, "onLost")
                        isNetworkConnected = false
                        GlobalUtils.currentNet = 99
                    }

                    override fun onLosing(network: Network, maxMsToLive: Int) {
                        super.onLosing(network, maxMsToLive)
                        Log.d(Utils.GLOBAL_TAG, "onLosing")
                    }
                })
                GlobalUtils.currentNet = parseNetworkType(context)
            }
        } catch (t: Throwable) {
            t.printStackTrace()
        }

    }

    /**
     * 初始化sdk时上报网络状态
     */
    private fun trackNetworkState() {
        try{
            Handler(Looper.getMainLooper()).postDelayed( {
                var netStatus = Consts.NetStatus.NO_NETWORK
                if (isNetworkConnected) {
                    netStatus = Consts.NetStatus.CONNECT_NETWORK
                }
                if (isFakeNetwork) {
                    netStatus = Consts.NetStatus.FAKE_NETWORK
                }

                DataMonitorHelper.commitBootStat(Consts.EVENT_MONITOR_INIT, bundleOf(
                    "net_status" to netStatus))
            }, 5000)
        }catch(e:Exception){
            Log.e(Utils.GLOBAL_TAG,  Log.getStackTraceString(e))
        }
    }

    private fun registerNetworkReceiver(context: Context) {
        val filter = IntentFilter()
        filter.addAction("android.net.conn.CONNECTIVITY_CHANGE")
        try {
            context.registerReceiver(mReceiver, filter)
        } catch (t: Throwable) {
            Log.e(TAG, t.message ?: "")
        }
    }

    fun parseNetworkType(context: Context): Int {
        try {
            val connectivityManager =
                context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
                    ?: return 0

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                // For Android Marshmallow (API 23) and above
                val network = connectivityManager.activeNetwork ?: return 99

                val capabilities = connectivityManager.getNetworkCapabilities(network)
                    ?: return 99

                if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
                    return 1
                } else if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
                    return getMobileNetworkClass(context)
                } else {
                    return 99
                }
            } else {
                // For older Android versions
                val activeNetworkInfo = connectivityManager.activeNetworkInfo
                if (activeNetworkInfo != null) {
                    if (activeNetworkInfo.type == ConnectivityManager.TYPE_WIFI) {
                        return 1
                    } else if (activeNetworkInfo.type == ConnectivityManager.TYPE_MOBILE) {
                        return getMobileNetworkClass(context)
                    } else {
                        return 99
                    }
                }
            }
        } catch (t: Throwable) {
            Log.e(TAG, t.message ?: "")
        }

        return 0
    }

    @SuppressLint("MissingPermission")
    private fun getMobileNetworkClass(context: Context): Int {
        try {
            val telephonyManager =
                context.getSystemService(Context.TELEPHONY_SERVICE) as? TelephonyManager
                    ?: return  0
            var networkType = 0
            var subtypeName: String? = ""
            if (android.os.Build.VERSION.SDK_INT < Build.VERSION_CODES.R) {
                networkType = telephonyManager.getNetworkType()
            } else {
                if (ActivityCompat.checkSelfPermission(context, Manifest.permission.READ_PHONE_STATE) == PackageManager.PERMISSION_GRANTED
                    || telephonyManager.hasCarrierPrivileges()) {
                    networkType = telephonyManager.getDataNetworkType()
                }
            }

            if (networkType == TelephonyManager.NETWORK_TYPE_UNKNOWN) {
                val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
                    ?: return  0
                val networkInfo = connectivityManager.getActiveNetworkInfo();
                if (networkInfo != null) {
                    networkType = networkInfo.getSubtype();
                    subtypeName = networkInfo.getSubtypeName();
                }
            }

            when (networkType) {
                //2G
                TelephonyManager.NETWORK_TYPE_GSM, TelephonyManager.NETWORK_TYPE_GPRS, TelephonyManager.NETWORK_TYPE_CDMA,
                TelephonyManager.NETWORK_TYPE_EDGE, TelephonyManager.NETWORK_TYPE_1xRTT, TelephonyManager.NETWORK_TYPE_IDEN ->
                    return 2
                //3G
                TelephonyManager.NETWORK_TYPE_EVDO_0, TelephonyManager.NETWORK_TYPE_EVDO_A, TelephonyManager.NETWORK_TYPE_EVDO_B, TelephonyManager.NETWORK_TYPE_EHRPD,
                TelephonyManager.NETWORK_TYPE_HSUPA, TelephonyManager.NETWORK_TYPE_HSDPA, TelephonyManager.NETWORK_TYPE_HSPA, TelephonyManager.NETWORK_TYPE_HSPAP, TelephonyManager.NETWORK_TYPE_UMTS, TelephonyManager.NETWORK_TYPE_TD_SCDMA ->
                    return 3
                //4G
                TelephonyManager.NETWORK_TYPE_LTE, 19 //TelephonyManager.NETWORK_TYPE_LTE_CA
                    , TelephonyManager.NETWORK_TYPE_IWLAN ->
                    return 4
                //5G
                TelephonyManager.NETWORK_TYPE_NR ->
                    return 5

                else ->
                    if (!subtypeName.isNullOrEmpty()) {
                        if (subtypeName.equals("TD-SCDMA", true)
                            || subtypeName.equals("WCDMA", true)
                            || subtypeName.equals("CDMA2000", true)
                        ) {
                            return 3
                        } else {
                            return 0
                        }
                    }
            }
        } catch (t: Throwable) {
            t.printStackTrace()
        }
        return 0
    }
}