package com.transsin.networkmonitor.config

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build
import android.os.Bundle
import android.os.Handler
import android.os.HandlerThread
import android.os.Message
import com.google.gson.Gson
import com.shalltry.aisearch.core.utils.LogUtil
import com.shalltry.aisearch.core.utils.ProcessUtil
import com.shalltry.aisearch.core.utils.SPUtils
import com.transsin.networkmonitor.Utils
import com.transsin.networkmonitor.config.broadcast.NetworkUpdateCDNBroadcastReceiver
import com.transsin.networkmonitor.config.broadcast.NetworkUpdateCDNBroadcastReceiver.Companion.UPDATE_CDN_ACTION
import com.transsin.util.GlobalUtils
import org.json.JSONObject
import java.util.concurrent.atomic.AtomicBoolean

object NetworkConfigManager {

    const val TAG = "netInspect:Config"

    private var STORE_NAME: String? = null
    private var KEY_CONFIG_DATA = "key_netinspector_config"
    private var KEY_CONFIG_UPDATE_TIME = "key_netinspector_update_time"
    private val isInit = AtomicBoolean(false)
    private var mHandlerThread: HandlerThread? = null
    private var mHandler: Handler? = null
    var mIsTest = false
    fun init(context: Context?, isTest: Boolean? = false) {
        try {
            mIsTest = isTest ?: false
            if (context == null) {
                LogUtil.e(TAG, "ConfigManager init fail: context is null!")
                return
            }
            if (isInit.compareAndSet(false, true)) {
                LogUtil.d(TAG, "ConfigManager [init] start")
                registerBroadcast(context)
                initHandlerThread(context)
                loadLocalConfigTask()
            }
        } catch (t: Throwable) {
            LogUtil.e(TAG, "ConfigManager init fail:$t")
        }

    }

    private fun initHandlerThread(context: Context) {
        LogUtil.d(TAG, "ConfigManager [initHandlerThread] start")
        mHandlerThread = HandlerThread("aisearch-data_HandlerThread")
        mHandlerThread?.start()
        mHandlerThread?.getLooper()?.apply {
            mHandler = object : Handler(this) {
                override fun handleMessage(msg: Message) {
                    super.handleMessage(msg)
                    try {
                        when (msg.what) {
                            1 -> {
                                //加载本地config
                                LogUtil.d(TAG, "ConfigManager [loadLocalConfigTask] run")
                                val storeName = getStoreName(context)
                                val configStr = SPUtils.getStringValue(context, storeName, KEY_CONFIG_DATA)
                                LogUtil.d(TAG, "ConfigManager [loadLocalConfigTask] configStr:$configStr")
                                parseStrToModel(configStr)
                                val configUpdateTime = SPUtils.getLongValue(context, storeName, KEY_CONFIG_UPDATE_TIME)
                                val currentTime = System.currentTimeMillis()
                                val diffTime = currentTime - configUpdateTime
                                //如果没有持久化数据，则使用业务设置的默认数据
                                if (GlobalUtils.configData.networkRate == -999) {
                                    GlobalUtils.configData.networkRate = GlobalUtils.defaultRate
                                    LogUtil.e(TAG, "network use default rate = ${GlobalUtils.defaultRate}")
                                }
                                if (GlobalUtils.configData.bootRate == -999) {
                                    GlobalUtils.configData.bootRate = GlobalUtils.defaultRate
                                    LogUtil.e(TAG, "boot use default rate = ${GlobalUtils.defaultRate}")
                                }
                                if (GlobalUtils.configData.customRate == -999) {
                                    GlobalUtils.configData.customRate = GlobalUtils.defaultRate
                                    LogUtil.e(TAG, "custom use default rate = ${GlobalUtils.defaultRate}")
                                }
                                LogUtil.d(TAG, "ConfigManager [loadLocalConfigTask] currentTime:$currentTime, configUpdateTime:$configUpdateTime, diffTime:$diffTime")
                                if (Math.abs(diffTime) > 86400000 ) {//一天间隔
                                    loadRemoteConfigTask()
                                }
                            }
                            2 -> {
                                //发起config remote请求
                                LogUtil.d(TAG, "ConfigManager [loadRemoteConfigTask] run")
                                val result = NetworkConfigLoader.loadConfig()
                                if (result.isNotEmpty()) {
                                    sendUpdateConfigBroadcast(context, result)
                                }
                            }
                            3 -> {
                                //更新config
                                val configStr = msg.data.getString("config")
                                LogUtil.d(TAG, "ConfigManager [updateConfig] run configStr:$configStr")
                                if (configStr?.isNotEmpty() == true) {
                                    val storeName = getStoreName(context)
                                    SPUtils.setStringValue(context, storeName, KEY_CONFIG_DATA, configStr)
                                    SPUtils.setLongValue(context, storeName, KEY_CONFIG_UPDATE_TIME, System.currentTimeMillis())
                                    parseStrToModel(configStr)
                                }
                            }
                            else -> {}
                        }
                    } catch (t: Throwable) {
                        LogUtil.e(TAG, "handleMessage fail:$t")
                    }
                }
            }
        }
    }

    private fun loadLocalConfigTask() {
        LogUtil.d(TAG, "ConfigManager [loadLocalConfigTask]")
        mHandler?.obtainMessage(1)?.apply {
            mHandler?.sendMessage(this)
        }
    }

    private fun loadRemoteConfigTask() {
        LogUtil.d(TAG, "ConfigManager [loadRemoteConfigTask]")
        mHandler?.obtainMessage(2)?.apply {
            mHandler?.sendMessage(this)
        }
    }
    private fun sendUpdateConfigBroadcast(context: Context, configStr: String) {
        LogUtil.d(TAG, "ConfigManager [sendUpdateConfigBroadcast], ${context.packageName}")
        val intent = Intent(UPDATE_CDN_ACTION)
        intent.putExtra("config", configStr)
        intent.setPackage(context.packageName)
        context.sendBroadcast(intent)
    }
    fun updateConfig(configStr: String) {
        LogUtil.d(TAG, "ConfigManager [updateConfig] configStr:$configStr")
        mHandler?.obtainMessage(3)?.apply {
            this.data = Bundle().apply {
                putString("config", configStr)
            }
            mHandler?.sendMessage(this)
        }
    }

    private fun parseStrToModel(str: String) {
        try {
            val parseStr = str.ifEmpty {
                "{}"
            }
            LogUtil.d(TAG, "ConfigManager [parseStrToModel] parseStr:$parseStr")

            //
            val gson = Gson()

            val config:NetworkConfigModel?  = gson.fromJson(str, NetworkConfigModel::class.java)
            parseRateInfo(config?.network)?.let {
                GlobalUtils.configData.networkRate = it
            }
            config?.network?.channel?.let {
                GlobalUtils.configData.networkChannel = it
            }
            parseRateInfo(config?.boot)?.let {
                GlobalUtils.configData.bootRate = it
            }
            config?.boot?.channel?.let {
                GlobalUtils.configData.bootChannel = it
            }
            parseRateInfo(config?.custom)?.let {
                GlobalUtils.configData.customRate = it
            }
            config?.custom?.channel?.let {
                GlobalUtils.configData.customChannel = it
            }

        } catch (t: Throwable) {
            LogUtil.e(TAG, "parseStrToModel fail:$t")
        }
    }

    private fun parseRateInfo(configModel: ConfigModel?): Int? {
        try {
            if (configModel == null) return null
            var rateInfo = configModel.rate?.find {//先匹配version & country
                it.appVersion == Utils.getVersionName() && it.country == Utils.getMcc()
            }
            if (rateInfo == null) {//再匹配app version
                rateInfo = configModel.rate?.find {
                    it.appVersion == Utils.getVersionName() && it.country == "*"
                }
            }

            if (rateInfo == null) {//再匹配country
                rateInfo = configModel.rate?.find {
                    it.appVersion == "*" && it.country == Utils.getMcc()
                }
            }

            if (rateInfo == null) {//再匹配默认规则
                rateInfo = configModel.rate?.find {
                    it.appVersion == "*" && it.country == "*"
                }
            }
            return rateInfo?.rate
        } catch (t: Throwable) {
            LogUtil.e(TAG, t.message ?: "")
        }
        return null
    }

    private fun getLongItem(jsonObject: JSONObject, key: String, defaultValue: Long): Long {
        var ret = defaultValue
        try {
            ret = jsonObject.getLong(key)
            LogUtil.d(TAG, "[getLongItem] key:$key, value:$ret")
        } catch (t: Throwable) {
            LogUtil.e(TAG, "[getLongItem] key:$key, value[defaultValue]:$ret, fail:$t")
        }
        return ret
    }

    private fun getDoubleItem(jsonObject: JSONObject, key: String, defaultValue: Double): Double {
        var ret = defaultValue
        try {
            ret = jsonObject.getDouble(key)
            LogUtil.d(TAG, "[getLongItem] key:$key, value:$ret")
        } catch (t: Throwable) {
            LogUtil.e(TAG, "[getLongItem] key:$key, value[defaultValue]:$ret, fail:$t")
        }
        return ret
    }

    private fun getIntItem(jsonObject: JSONObject, key: String, defaultValue: Int): Int {
        var ret = defaultValue
        try {
            ret = jsonObject.getInt(key)
            LogUtil.d(TAG, "[getIntItem] key:$key, value:$ret")
        } catch (t: Throwable) {
            LogUtil.e(TAG, "[getIntItem] key:$key, value[defaultValue]:$ret, fail:$t")
        }
        return ret
    }

    private fun getBooleanItem(
        jsonObject: JSONObject,
        key: String,
        defaultValue: Boolean
    ): Boolean {
        var ret = defaultValue
        try {
            ret = jsonObject.getBoolean(key)
            LogUtil.d(TAG, "[getBooleanItem] key:$key, value:$ret")
        } catch (t: Throwable) {
            LogUtil.e(TAG, "[getBooleanItem] key:$key, value[defaultValue]:$ret, fail:$t")
        }
        return ret
    }

    @SuppressLint("UnspecifiedRegisterReceiverFlag")
    private fun registerBroadcast(context: Context) {
        try {
            LogUtil.d(TAG, "ConfigManager [registerBroadcast] start")
            val filter = IntentFilter(UPDATE_CDN_ACTION)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                context.registerReceiver(
                    NetworkUpdateCDNBroadcastReceiver(),
                    filter,
                    Context.RECEIVER_NOT_EXPORTED
                )
            } else {
                context.registerReceiver(
                    NetworkUpdateCDNBroadcastReceiver(),
                    filter
                )
            }
        } catch (t: Throwable) {
            t.printStackTrace()
        }
    }

    private fun getStoreName(context: Context): String {
        if (STORE_NAME.isNullOrEmpty()) {
            STORE_NAME = "network_config_file_${ProcessUtil.getCurrentProcessName(context).replace(":", "_")}"
        }
        LogUtil.d(TAG, "[getStoreName] STORE_NAME:$STORE_NAME")
        return STORE_NAME!!
    }
}