package tech.palm.lib;

import android.util.Log;

import java.lang.reflect.Method;
import java.util.function.Supplier;


public class SingleLiceFactory<T, S extends T> {
    private static final String TAG = "SingleLiceFactory";
    private static boolean DEBUG = true;

    private final Class<T> mInterfaceClass;
    private final Supplier<T> mFuncInstanceDefault;
    private final Supplier<S> mFuncInstance;

    private S mServiceLice;
    private T mDefaultServiceLice;

    public SingleLiceFactory(Class<T> interfaceClass, Supplier<T> funcInstanceDefault, Supplier<S> funcInstance) {
        mInterfaceClass = interfaceClass;
        mFuncInstanceDefault = funcInstanceDefault;
        mFuncInstance = funcInstance;
    }

    public S instance() {
        if (mServiceLice == null) {
            mServiceLice = mFuncInstance.get();
        }
        return mServiceLice;
    }

    public T instance(String className) {
        T res = null;
        final Class<?> callerClass = mInterfaceClass;
        try {
            final Class<?> classFactory = Class.forName(className, true, callerClass.getClassLoader());
            Method methodInstance = classFactory.getDeclaredMethod("instance");
            Object obj = methodInstance.invoke(classFactory);
            if (mInterfaceClass.isInstance(obj)) {
                res = (T) obj;
                if (DEBUG) {
                    Log.d(TAG, "instance successfully. "
                            + obj + " from " + callerClass.getName());
                }
            } else if (obj != null) {
                Log.w(TAG, "instance failed. "
                        + obj + " " + " from " + callerClass.getName());
            }
        } catch (Throwable e) {
            if (DEBUG) {
                Log.w(TAG, "instance failed. " + className + " " + " from " + callerClass.getName(), e);
            } else {
                Log.w(TAG, "instance failed. " + className + " " + " from " + callerClass.getName() + " " + e);
            }
        }
        if (res == null) {
            if (mDefaultServiceLice == null) {
                mDefaultServiceLice = mFuncInstanceDefault.get();
            }
            res = mDefaultServiceLice;
            Log.d(TAG, "using " + mDefaultServiceLice + " instead of "
                    + className + " " + " from " + callerClass.getName());
        }

        return res;
    }
}
