package com.transsion.xuanniao.account.comm.widget;


import android.content.Context;
import android.graphics.Canvas;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.ProgressBar;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.content.res.AppCompatResources;
import androidx.customview.widget.ViewDragHelper;

import com.transsion.xuanniao.account.R;
import com.transsion.xuanniao.account.comm.utils.LogUtils;

public class BehaviorSlideView extends FrameLayout {

    private BehaviorSlideView mSlideView;
    /**
     * 滑动滑块
     */
    private ImageView mSlideBlock;
    private ProgressBar mProgressBar;
    /**
     * 拽托帮助类
     */
    private ViewDragHelper mViewDragHelper;

    private SlideListener mSlideListener;
    /**
     * 失败状态
     */
    private boolean isError = false;
    /**
     * 滑块状态
     */
    private int mSlideState = ViewDragHelper.STATE_IDLE;
    /**
     * 是否允许滑动
     */
    private boolean isDragEnable = true;

    public interface SlideListener {
        void onViewSlide(float distance);
        void onViewReleased(float value);
    }

    public BehaviorSlideView(@NonNull Context context) {
        this(context, null);
    }

    public BehaviorSlideView(@NonNull Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public BehaviorSlideView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs, defStyleAttr);
    }

    private void init(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        mSlideView = this;
        mViewDragHelper = ViewDragHelper.create(this, 0.2f, new ViewDragHelper.Callback() {

            @Override
            public boolean tryCaptureView(@NonNull View child, int pointerId) {
                //判断能拽托的View，这里会遍历内部子控件来决定是否可以拽托，我们只需要滑块可以滑动
                return child == mSlideBlock && !isError && isDragEnable;
            }

            @Override
            public int clampViewPositionHorizontal(@NonNull View child, int left, int dx) {
//                LogUtils.d("clampViewPositionHorizontal left = " + left);
                //拽托子View横向滑动时回调，回调的left，则是可以滑动的左上角x坐标
                int blockWidth = mSlideBlock.getWidth();
                //限制左右临界点
                int fullWidth = mSlideView.getWidth();
                //最少的左边
                int leftMinDistance = getPaddingStart();
                //最多的右边
                int leftMaxDistance = fullWidth - getPaddingEnd() - blockWidth;
                //修复两端的临界值
                if (left < leftMinDistance) {
                    left =  leftMinDistance;
                } else if (left > leftMaxDistance) {
                    left = leftMaxDistance;
                }
                if (mProgressBar != null) {
                    mProgressBar.setProgress((left + blockWidth / 2) * 100 / fullWidth);
                }
                if (mSlideListener != null) {
                    mSlideListener.onViewSlide(left);  // 距离
                }
                return left;
            }

            @Override
            public int clampViewPositionVertical(@NonNull View child, int top, int dy) {
                //拽托子View纵向滑动时回调，锁定顶部padding距离即可，不能不复写，否则少了顶部的padding，位置就偏去上面了
                return getPaddingTop();
            }

            @Override
            public void onViewCaptured(@NonNull View capturedChild, int activePointerId) {
                super.onViewCaptured(capturedChild, activePointerId);
            }

            @Override
            public void onViewReleased(@NonNull View releasedChild, float xvel, float yvel) {
                super.onViewReleased(releasedChild, xvel, yvel);
                //获取滑块当前的位置
                int blockLeft = releasedChild.getLeft();
                //获取滑道宽度
                int fullWidth = mSlideView.getWidth();
                if (mSlideListener != null) {
                    mSlideListener.onViewReleased(blockLeft * 1.0f / fullWidth);  // 0 ~ 1
                } else {
                    reset();
                }
            }

            @Override
            public void onViewDragStateChanged(int state) {
                super.onViewDragStateChanged(state);
                LogUtils.d("onViewDragStateChanged state = " + state);
                mSlideState = state;
                if (state == ViewDragHelper.STATE_IDLE) {
                    if (isError) {
                        resetState();
                    }
                }
            }
        });
    }

    /**
     * 恢复默认UI
     */
    private void resetState() {
        isError = false;
        if (mProgressBar != null) {
            mProgressBar.setProgressDrawable(AppCompatResources.getDrawable(getContext(),
                    R.drawable.xn_behavior_progress_normal));
        }
        if (mSlideBlock != null) {
            mSlideBlock.setImageResource(R.drawable.xn_slide_block_normal);
        }
    }

    public void setSlideListener(SlideListener slideListener) {
        mSlideListener = slideListener;
    }

    /**
     * 滑块复位
     */
    public void reset() {
        if (mViewDragHelper != null) {
            mViewDragHelper.smoothSlideViewTo(mSlideBlock, getPaddingStart(), 0);  // 复位
            invalidate();
        }
    }

    /**
     * 校验失败，红色复位
     */
    public void resetWithError() {
        isError = true;
        if (mProgressBar != null) {
            mProgressBar.setProgressDrawable(AppCompatResources.getDrawable(getContext(),
                    R.drawable.xn_behavior_progress_error));
        }
        if (mSlideBlock != null) {
            mSlideBlock.setImageResource(R.drawable.xn_slide_block_error);

            if (mViewDragHelper != null) {
                boolean continueSliding =
                        mViewDragHelper.smoothSlideViewTo(mSlideBlock, getPaddingStart(), 0);  // 复位
                LogUtils.d("resetWithError continueSliding = " + continueSliding);
                if (continueSliding) {
                    invalidate();
                } else {
                    postDelayed(new Runnable() {  // 在起点位置
                        @Override
                        public void run() {
                            if (isError) {
                                resetState();
                            }
                        }
                    }, 1000L);
                }
            }
        }
    }

    /**
     * 是否空闲
     */
    public boolean isIdle() {
        return mSlideState == ViewDragHelper.STATE_IDLE;
    }

    /**
     * 设置状态
     */
    public void setDragEnable(boolean enable) {
        isDragEnable = enable;
    }

    @Override
    public void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        int left = mSlideBlock.getLeft();
        int blockWidth = mSlideBlock.getWidth();
        int fullWidth = getWidth();
        if (mProgressBar != null) {
            mProgressBar.setProgress((left + blockWidth / 2) * 100 / fullWidth);
        }
        if (mSlideListener != null) {
            mSlideListener.onViewSlide(left);  // 距离
        }
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        mSlideBlock = findViewById(R.id.slideBlock);
        mProgressBar = findViewById(R.id.progressBar);
    }

    @Override
    public boolean onInterceptTouchEvent(MotionEvent ev) {
        //将onInterceptTouchEvent委托给ViewDragHelper
        return mViewDragHelper.shouldInterceptTouchEvent(ev);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        //将onTouchEvent委托给ViewDragHelper
        mViewDragHelper.processTouchEvent(event);
        return true;
    }

    @Override
    public void computeScroll() {
        super.computeScroll();
        //判断是否移动到头了，未到头则继续
        if (mViewDragHelper != null) {
            if (mViewDragHelper.continueSettling(true)) {
                invalidate();
            }
        }
    }
}