package com.transsion.palmsdk;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.transsion.palmsdk.util.PalmUtil;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

/**
 * 授权配置参数
 */
public class PalmAuthParam implements Parcelable {

    private String mAppid;
    private String mScopes;
    private String mRedirectUri;
    private String mState;  //随机字串，防CSRF

    private boolean mReqToken;  //是否返回token，默认返回auth code
    private boolean mIgnorePalmApp; // 无视PalmID App
    private boolean mShowTPLogin;  // 是否支持google/facebook登录
    private int mServerMode;   //  服务器环境
    private String mTudcSpName;   // TUDC登录数据文件
    private boolean mOnlyPhoneNum;   // 支持移动号码注册/登录
    private boolean mShowPrevLogin;   // 显示预登录弹框
    private String mIntentActivity;   // 通过Activity回调登录结果
    private String mAppProcess;     // 主进程进程名
    private boolean mBindTransID;    // 关联系统账号登录状态

    private String mPkgName;
    private String mAppSign;

    private PalmAuthParam() {}

    private PalmAuthParam(Builder builder) {
        if (builder == null) {
            return;
        }
        mAppid = builder.appid;
        if (builder.scopes == null
                || builder.scopes.length == 0) {
            mScopes = Scope.INFO_BASIC;
        } else {
            StringBuilder sBuilder = new StringBuilder();
            for (String scope : builder.scopes) {
                if (sBuilder.length() == 0) {
                    sBuilder.append(scope);
                } else {
                    sBuilder.append(" ").append(scope);
                }
            }
            if (sBuilder.indexOf(Scope.INFO_BASIC) < 0) {  // 不包含info_base
                sBuilder.insert(0, Scope.INFO_BASIC + " ");
            }
            mScopes = sBuilder.toString();
        }
        if (TextUtils.isEmpty(builder.redirectUri)) {
            mRedirectUri = PalmConstants.PALM_REDIRECT_URI;
        } else {
            mRedirectUri = builder.redirectUri;
        }
        mState = PalmUtil.randomString(8);
        mReqToken = builder.reqToken;
        mIgnorePalmApp = builder.ignorePalmApp;
        mShowTPLogin = builder.showTPLogin;
        mServerMode = builder.serverMode;
        mTudcSpName = builder.tudcSpName;
        mOnlyPhoneNum = builder.onlyPhoneNum;
        mShowPrevLogin = builder.showPrevLogin;
        mIntentActivity = builder.intentActivity;
        mAppProcess = builder.appProcess;
        mBindTransID = builder.bindTransID;
    }

    /**
     * 拷贝请求信息+获取token完成FullSDK登录
     */
    protected PalmAuthParam copy() {
        PalmAuthParam authParam = new PalmAuthParam();
        authParam.mAppid = mAppid;
        authParam.mScopes = mScopes;
        authParam.mState = mState;
        authParam.mAppSign = mAppSign;
        authParam.mPkgName = mPkgName;

        authParam.mReqToken = true;
        authParam.mShowTPLogin = mShowTPLogin;  // 显示三方登录
        authParam.mRedirectUri = PalmConstants.PALM_REDIRECT_URI;
        authParam.mOnlyPhoneNum = mOnlyPhoneNum;  // 指定登录方式
        authParam.mShowPrevLogin = mShowPrevLogin;  // 预登录弹框
        authParam.mIntentActivity = mIntentActivity;  // 指定回调方式
        authParam.mBindTransID = mBindTransID;
        return authParam;
    }

    protected PalmAuthParam(Parcel in) {
        mAppid = in.readString();
        mScopes = in.readString();
        mRedirectUri = in.readString();
        mState = in.readString();
        mReqToken = in.readInt() == 1;
        mShowTPLogin = in.readInt() == 1;
        mPkgName = in.readString();
        mAppSign = in.readString();
        try {
            String option = in.readString();
            PalmUtil.LOG.d("option = " + option);
            JSONObject optionJs = new JSONObject(option);
            mOnlyPhoneNum = optionJs.optBoolean("onlyPhoneNum", false);
            mShowPrevLogin = optionJs.optBoolean("showPrevLogin", true);
            mIntentActivity = optionJs.optString("intentActivity", "");
        } catch (Exception e) {
//            e.printStackTrace();
            mShowPrevLogin = true;
        }
    }

    public static final Creator<PalmAuthParam> CREATOR = new Creator<PalmAuthParam>() {
        @Override
        public PalmAuthParam createFromParcel(Parcel in) {
            return new PalmAuthParam(in);
        }

        @Override
        public PalmAuthParam[] newArray(int size) {
            return new PalmAuthParam[size];
        }
    };

    public String getAppid() {
        return mAppid;
    }

    public String getScopes() {
        return mScopes;
    }

    public String getState() {
        return mState;
    }

    public String getRedirectUri() {
        return mRedirectUri;
    }

    public boolean isReqToken() {
        return mReqToken
                && TextUtils.equals(mRedirectUri, PalmConstants.PALM_REDIRECT_URI);
    }

    public boolean ignorePalmApp() {
        return mIgnorePalmApp;
    }

    /**
     * 钱包需求-动态选择是否关联PalmID
     */
    public void setIgnorePalmApp(boolean ignore) {
        mIgnorePalmApp = ignore;
    }

    public boolean isShowTPLogin() {
        return mShowTPLogin;
    }

    public int getServerMode() {
        return mServerMode;
    }

    public boolean hasTudc() {
        return mTudcSpName != null;
    }

    public String getTudcSpName() {
        return mTudcSpName;
    }

    public boolean isOnlyPhoneNum() {
        return mOnlyPhoneNum;
    }

    public boolean isShowPrevLogin() {
        return mShowPrevLogin;
    }

    public String getIntentActivity() {
        return mIntentActivity;
    }

    public String getAppProcess() {
        return mAppProcess;
    }

    public boolean isBindTransID() {
        return mBindTransID;
    }

    protected void setupApp(Context context) {
        mPkgName = context.getPackageName();
        mAppSign = PalmUtil.getAppSign(context, "MD5");
    }

    public String getPkgName() {
        return mPkgName;
    }

    public String getAppSign() {
        return mAppSign;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mAppid);
        dest.writeString(mScopes);
        dest.writeString(mRedirectUri);
        dest.writeString(mState);
        dest.writeInt(mReqToken ? 1 : 0);
        dest.writeInt(mShowTPLogin ? 1 : 0);
        dest.writeString(mPkgName);
        dest.writeString(mAppSign);
        try {
            dest.writeString(new JSONObject()
                    .put("onlyPhoneNum", mOnlyPhoneNum)
                    .put("showPrevLogin", mShowPrevLogin)
                    .put("intentActivity", mIntentActivity).toString());  // 方便扩展
        } catch (Exception e) {
//            e.printStackTrace();
        }
    }

    public Map<String, Object> toMap() {
        Map<String, Object> map = new HashMap<>();
        map.put("client_id", mAppid);
        map.put("redirect_uri", mRedirectUri);
        map.put("scope", mScopes);
        map.put("state", mState);
        return map;
    }

    @Override
    public String toString() {
        return mAppid + "," + mScopes + "," + mRedirectUri;
    }

    public static class Builder {
        private String appid;
        private String[] scopes;
        private String redirectUri;
        private boolean reqToken = false;
        private boolean ignorePalmApp = false;
        private boolean showTPLogin = true;
        private int serverMode = PalmConstants.SERVER_ONLINE;  // 默认正式
        private String tudcSpName = null;
        private boolean onlyPhoneNum = false;
        private boolean showPrevLogin = true;  // 默认显示
        private String intentActivity = null;
        private String appProcess = null;
        private boolean bindTransID = true;

        public Builder setAppid(String appid) {
            this.appid = appid;
            return this;
        }

        public Builder setScopes(String[] scopes) {
            this.scopes = scopes;
            return this;
        }

        public Builder setRedirectUri(String redirectUri) {
            this.redirectUri = redirectUri;
            return this;
        }

        public Builder ignorePalmApp(boolean ignorePalmApp) {
            this.ignorePalmApp = ignorePalmApp;
            return this;
        }

        public Builder supportTPLogin(boolean showTPLogin) {
            this.showTPLogin = showTPLogin;
            return this;
        }

        public Builder requestToken() {
            this.reqToken = true;
            return this;
        }

        public Builder setServerMode(int serverMode) {
            this.serverMode = serverMode;
            return this;
        }

        public Builder hasTudc(String spName) {
            this.tudcSpName = spName;
            return this;
        }

        public Builder onlyPhoneNum(boolean onlyPhoneNum) {
            this.onlyPhoneNum = onlyPhoneNum;
            return this;
        }

        public Builder showPrevLogin(boolean showPrevLogin) {
            this.showPrevLogin = showPrevLogin;
            return this;
        }

        public Builder intentActivity(Class<?> intentActivity) {
            if (intentActivity != null) {
                this.intentActivity = intentActivity.getName();
            }
            return this;
        }

        public Builder setAppProcess(String appProcess) {
            this.appProcess = appProcess;
            return this;
        }

        public Builder bindTransID(boolean bindTransID) {
            this.bindTransID = bindTransID;
            return this;
        }

        public PalmAuthParam build() {
            return new PalmAuthParam(this);
        }
    }
}
