package com.transsion.palmsdk.util;

import android.annotation.SuppressLint;
import android.app.ActivityManager;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.content.res.Resources;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.provider.Settings;
import android.text.TextUtils;
import android.util.Log;

import com.transsion.core.deviceinfo.DeviceInfo;
import com.transsion.core.log.ObjectLogUtils;
import com.transsion.palmsdk.PalmConstants;

import org.json.JSONObject;

import java.lang.reflect.Method;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Random;

public class PalmUtil {

    /**
     * 日志输出工具
     */
    public static ObjectLogUtils LOG = new ObjectLogUtils.Builder()
            .setGlobalTag("PalmSDK").setLogHeadSwitch(true).setLogSwitch(false).setBorderSwitch(false).create();

    /**
     * map -> url
     */
    public static String urlEncodeUTF8(Map<String, Object> map) {
        StringBuilder builder = new StringBuilder();
        try {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                if (builder.length() > 0) {
                    builder.append("&");
                }
                builder.append(String.format("%s=%s",
                        URLEncoder.encode(entry.getKey(), "UTF-8"),
                        URLEncoder.encode(entry.getValue().toString(), "UTF-8")
                ));
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        return builder.toString();
    }

    /**
     * url -> map
     */
    public static Bundle urlParse(String url) {
        int index = url.indexOf('?');
        if (index > 0) {
            String param = url.substring(index + 1);
            if (param.startsWith("code=") || param.contains("&code=")) {
                return parseParam(param);
            }
            if (param.startsWith("error=") || param.contains("&error=")) {
                return parseParam(param);
            }
        } else {
            int index2 = url.indexOf('#');
            if (index2 > 0) {
                String param = url.substring(index2 + 1);
                if (param.startsWith("access_token=") || param.contains("&access_token=")) {
                    return parseParam(url.replace("#", "?"));
                }
                if (param.startsWith("error=") || param.contains("&error=")) {
                    return parseParam(url.replace("#", "?"));
                }
            }
        }
        return null;
    }

    private static Bundle parseParam(String param) {
        Bundle bundle = new Bundle();
        if (param != null) {
            String[] params = param.split("&");
            for (int i = 0; i < params.length; i++) {
                String[] p = params[i].split("=");
                if (p.length == 0) {
                    continue;
                }
                try {
                    String key = URLDecoder.decode(p[0], "UTF-8");
                    if (TextUtils.isEmpty(key)) {
                        continue;
                    }
                    String value;
                    if (p.length == 2) {
                        value = URLDecoder.decode(p[1], "UTF-8");
                    } else {
                        value = "";
                    }
                    bundle.putString(key, value);
                } catch (Exception e) {
                    PalmUtil.LOG.e(Log.getStackTraceString(e));
                }
            }
        }
        return bundle;
    }

    /**
     * 生成指定长度的随机字串
     */
    public static String randomString(int length) {
        String str = "0123456789abcdefghijklmnopqrstuvwxyz";
        Random random = new Random();
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < length; i++) {
            builder.append(str.charAt(random.nextInt(36)));
        }
        return builder.toString();
    }

    /**
     * 返回协议://域名格式字串
     */
    public static String formatUrlHost(String url) {
        Uri uri = Uri.parse(url);
        return uri.getScheme() + "://" + uri.getHost();
    }

    /**
     * 获取应用签名 SHA1/MD5/SHA256
     */
    public static String getAppSign(Context context, String type) {
        try {
            @SuppressLint("PackageManagerGetSignatures")
            PackageInfo packageInfo = context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), PackageManager.GET_SIGNATURES);
            MessageDigest digest = MessageDigest.getInstance(type);
            Signature[] signatures = packageInfo.signatures;
            if (signatures != null) {
                for (Signature signature : signatures) {
                    byte[] digestBytes = digest.digest(signature.toByteArray());
                    StringBuilder builder = new StringBuilder();
                    for (byte digestByte : digestBytes) {
                        builder.append((Integer.toHexString((digestByte & 0xFF) | 0x100)).substring(1, 3));
                    }
                    return builder.toString().toUpperCase();
                }
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        return null;
    }

    /**
     * 网络连接是否可用
     */
    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager cm =
                (ConnectivityManager) context.getApplicationContext().getSystemService(Context.CONNECTIVITY_SERVICE);
        if (cm == null) {
            return false;
        }
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                Network activeNetwork = cm.getActiveNetwork();
                if (activeNetwork != null) {
                    NetworkCapabilities networkCapabilities = cm.getNetworkCapabilities(activeNetwork);
                    if (networkCapabilities != null) {
                        return networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
                                || networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
                                || networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_VPN)
                                || networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET);
                    }
                }
            } else {
                NetworkInfo networkInfo = cm.getActiveNetworkInfo();
                return networkInfo != null && networkInfo.isConnected();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    public static String getLocalLanguage(Context context) {
        if (Build.VERSION.SDK_INT >= 24) {
            Resources res = context.getResources();
            if (res != null) {
                return res.getConfiguration().getLocales().get(0).getLanguage();
            }
        } else {
            return Locale.getDefault().getLanguage();
        }
        return "";
    }

    public static String getLanguageWithCountry(Context context) {
        Locale locale;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else {
            locale = context.getResources().getConfiguration().locale;
        }
        String language = locale.getLanguage();
        String country = locale.getCountry();
        if (TextUtils.isEmpty(country)) {
            return language;
        }
        return language + "-r" + country;
    }

    public static String getMcc() {
        String deviceOperator = DeviceInfo.getSimOperator();
        if (!TextUtils.isEmpty(deviceOperator) && deviceOperator.length() >= 3) {
            return deviceOperator.substring(0, 3);
        }
        return "";
    }

    /**
     * 返回当前程序版本号
     */
    public static String getAppVersion(Context context) {
        try {
            PackageInfo pi = context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), 0);
            return pi.versionName;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 获取定制化OS版本号
     */
    public static String getOSVersion() {
        try {
            Class<?> cls = Class.forName("android.os.SystemProperties");
            Method method = cls.getMethod("get", String.class, String.class);
            String osVersion = (String) method.invoke(null, "ro.tranos.version", "");
            if (TextUtils.isEmpty(osVersion)) {
                osVersion = (String) method.invoke(null, "ro.os_product.version", "");
            }
            return osVersion;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 获取当前进程名
     */
    public static String getCurProcessName(Context context) {
        String processName = null;
        try {
            if (Build.VERSION.SDK_INT >= 28) {
                processName = Application.getProcessName();
            } else {
                int pid = android.os.Process.myPid();
                ActivityManager activityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
                List<ActivityManager.RunningAppProcessInfo> appProcesses = activityManager.getRunningAppProcesses();
                if (appProcesses != null) {
                    for (ActivityManager.RunningAppProcessInfo appProcess : appProcesses) {
                        if (appProcess.pid == pid) {
                            processName = appProcess.processName;
                        }
                    }
                }
            }
        }catch (Exception e) {
            e.printStackTrace();
        }
        if (TextUtils.isEmpty(processName)) {
            processName = context.getPackageName();
        }
        return processName;
    }

    /**
     * 请求云控的参数
     */
    public static Map<String, Object> genConfigParam(Context context) {
        Map<String, Object> map = new HashMap<>();
        map.put("mcc", PalmUtil.getMcc());
        map.put("osVersion", PalmUtil.getOSVersion());
        map.put("buildNumber", Build.DISPLAY);
        map.put("androidVersion", Build.VERSION.RELEASE);
        map.put("brand", Build.BRAND);
        map.put("language", Locale.getDefault().getLanguage());
        map.put("appVersion", PalmUtil.getAppVersion(context));
        try {
            AdvertisingIdClient.AdInfo adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
            map.put("gaid", adInfo.getId());  // 获取最新的gaid
        } catch (Exception e) {
            // PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        return map;
    }

    /**
     * 手机上是否安装palm id app
     * @param onlyPalmID 只针对云控配置的PalmID应用
     */
    public static Intent queryPalmAppIntent(Context context, boolean onlyPalmID) {
        Intent intent = new Intent(PalmConstants.ACTION_PALM_ACCOUNT_AUTH);
        PackageManager packageManager = context.getPackageManager();
        try {
            ApplicationInfo applicationInfo
                    = packageManager.getApplicationInfo("tech.palm.id", PackageManager.GET_META_DATA);
            Bundle bundle = applicationInfo.metaData;
            if (bundle != null
                    && bundle.getString("palm_sdk_version", "").compareTo("1.0.0.4") >= 0) {
                intent.setPackage("tech.palm.id");
                intent.putExtra(PalmConstants.EXTRA_HOST_MODE, false);
                return intent;
            }
        } catch (Exception e) {
//            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        try {
            SharedPreferences palmPrefs =
                    context.getSharedPreferences(PalmConstants.SP_NAME, Context.MODE_PRIVATE);
            String palmAppConfig = palmPrefs.getString(PalmConstants.KEY_PALM_APP, null);
            if (palmAppConfig != null && palmAppConfig.length() > 0) {  // 云控palm id列表
                JSONObject palmAppJson = new JSONObject(palmAppConfig);
                Iterator<String> keys = palmAppJson.keys();
                while (keys.hasNext()) {
                    String pkgName = keys.next();
//                    String versionName = palmAppJson.optString(pkgName);
                    try {
                        ApplicationInfo applicationInfo
                                = packageManager.getApplicationInfo(pkgName, PackageManager.GET_META_DATA);
                        Bundle bundle = applicationInfo.metaData;
                        if (bundle != null
                                && bundle.getString("palm_sdk_version", "").compareTo("1.0.0.4") >= 0) {
                            intent.setPackage(pkgName);
                            intent.putExtra(PalmConstants.EXTRA_HOST_MODE, false);
                            return intent;
                        }
                    } catch (Exception e) {
//                      PalmUtil.LOG.e(Log.getStackTraceString(e));
                    }
                }
            }
        } catch (Exception e) {
//            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        if (!onlyPalmID) {
            try {
                String hostPalmApp = Settings.Global.getString(context.getContentResolver(),
                        PalmConstants.KEY_HOST_PALM_APP);  // 查询HOST应用
                PalmUtil.LOG.d("hostPalmApp = " + hostPalmApp);
                if (!TextUtils.isEmpty(hostPalmApp)
                        && !TextUtils.equals(hostPalmApp, context.getPackageName())) {
                    try {
                        ApplicationInfo applicationInfo
                                = packageManager.getApplicationInfo(hostPalmApp, PackageManager.GET_META_DATA);
                        Bundle bundle = applicationInfo.metaData;
                        if (bundle != null
                                && bundle.getString("palm_sdk_version", "").compareTo("1.0.0.4") >= 0) {
                            intent.setPackage(hostPalmApp);
                            intent.putExtra(PalmConstants.EXTRA_HOST_MODE, true);
                            return intent;
                        }
                    } catch (Exception e) {
//                       PalmUtil.LOG.e(Log.getStackTraceString(e));
                    }
                }
            } catch (Exception e) {
                PalmUtil.LOG.e(Log.getStackTraceString(e));
            }
        }
        return null;
    }
}
