package com.transsion.palmsdk;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.RemoteException;
import android.os.UserManager;
import android.text.TextUtils;
import android.util.Log;

import com.transsion.athenacust.AthenaCust;
import com.transsion.ga.AthenaAnalytics;
import com.transsion.gslb.GslbSdk;
import com.transsion.palmsdk.account.XNManager;
import com.transsion.palmsdk.auth.IPalmAuthResponse;
import com.transsion.palmsdk.auth.PalmAuthManager;
import com.transsion.palmsdk.auth.PalmAuthSession;
import com.transsion.palmsdk.data.PalmAuthRequest;
import com.transsion.palmsdk.data.PalmAuthResult;
import com.transsion.palmsdk.data.PalmTokenInfo;
import com.transsion.palmsdk.http.PalmHttpCallback;
import com.transsion.palmsdk.http.PalmHttpClient;
import com.transsion.palmsdk.tudc.TudcAccount;
import com.transsion.palmsdk.util.PalmUtil;
import com.transsion.palmsdk.util.ThreadExecutors;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

/**
 * PalmID接口实现
 */
public class PalmDefaultImpl extends PalmID {

    private final Context mContext;
    private final PalmAuthParam mPalmAuthParam;
    private boolean isMainProcess = true;
    private boolean isFullSDK = false;  // 是否带注册/登录页面

    PalmDefaultImpl(Context context, PalmAuthParam palmAuthParam) {
        mContext = context.getApplicationContext();
        mPalmAuthParam = palmAuthParam;
        mPalmAuthParam.setupApp(context);
        initSDK(mContext);
    }

    /**
     * SDK初始化
     */
    private void initSDK(Context context) {
        AthenaAnalytics.init(context, "", 7710, false);  // 初始化Athena
//        AthenaAnalytics.setTest(true);
        Bundle eparam = new Bundle();
        eparam.putString("appid", mPalmAuthParam.getAppid());
        eparam.putString("version", BuildConfig.SDK_VERSION);
        new AthenaCust("sdk_init", 7710).trackCommon(eparam, null).submit();

        GslbSdk.init(context, new String[]{"account.palm.tech",
                "auth-account.palm.tech", "api-cloud.palm.tech"}, new GslbSdk.InitListener() {
            @Override
            public void onInitSuccess(Map<String, String> map) {
                PalmUtil.LOG.d("GslbSdk sdk_init success");
            }

            @Override
            public void onInitFail() {
                PalmUtil.LOG.d("GslbSdk sdk_init fail");
            }
        });

        String appProcess = mPalmAuthParam.getAppProcess();
        if (TextUtils.isEmpty(appProcess)) {
            isMainProcess = true;
        } else {
            isMainProcess = TextUtils.equals(appProcess, PalmUtil.getCurProcessName(mContext));
        }

        try {
            if (isMainProcess) {
                Class.forName("com.transsion.xuanniao.account.api.PalmInitialize");  // 触发xn_core初始化
            }
            Class.forName("com.transsion.xuanniao.account.login.view.LoginActivity");
            isFullSDK = true;
        } catch (Exception e) {
//            e.printStackTrace();
        }

        if (isMainProcess) {
            // 网络请求和TEE初始化置于工作线程
            ThreadExecutors.getInstance().execute(new Runnable() {
                @Override
                public void run() {
                    PalmConstants.SERVER_MODE = mPalmAuthParam.getServerMode();
                    initCloudConfig();

                    if (isFullSDK) {
                        XNManager.getInstance()
                                .applyKey(mContext, mPalmAuthParam.getAppid(), mPalmAuthParam.getServerMode());
                    }
                }
            });
        } else {
            PalmConstants.SERVER_MODE = mPalmAuthParam.getServerMode();  // 非主进程
        }
        PalmUtil.LOG.d("initSDK isFullSDK = " + isFullSDK + ", version = " + BuildConfig.SDK_VERSION);
    }

    /**
     * 请求云控配置
     */
    private void initCloudConfig() {
        try {
            Context safeContext = mContext;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                UserManager um = (UserManager) mContext.getSystemService(Context.USER_SERVICE);
                if (!um.isUserUnlocked()) {
                    safeContext = mContext.createDeviceProtectedStorageContext();
                }
            }
            SharedPreferences palmPrefs =
                    safeContext.getSharedPreferences(PalmConstants.SP_NAME, Context.MODE_PRIVATE);
            String palmAppConfig = palmPrefs.getString(PalmConstants.KEY_PALM_APP, null);
            long palmUpdateTime = palmPrefs.getLong(PalmConstants.KEY_PALM_UPDATE_TIME, 0);
            if ((palmAppConfig == null
                    || Math.abs(System.currentTimeMillis() - palmUpdateTime) > PalmConstants.ONE_DAY)
                    && PalmUtil.isNetworkAvailable(mContext)) {
                Map<String, Object> formData = PalmUtil.genConfigParam(mContext);
                Map<String, String> header = new HashMap<>();
                header.put("Client-ID", "palmid");
                String response = PalmHttpClient.getInstance().getPalmAppConfig(header, formData);
                if (response != null) {
                    try {
                        JSONObject jsonObject = new JSONObject(response);
                        JSONArray jsonArray = jsonObject.getJSONArray("cloudClientConfigs");
                        SharedPreferences.Editor editor = palmPrefs.edit()
                                .putString(PalmConstants.KEY_PALM_CONFIG, response);  // for h5
                        for (int i = 0; i < jsonArray.length(); i++) {
                            JSONObject configJson = jsonArray.getJSONObject(i);
                            if (configJson.getInt("type") == PalmConstants.TYPE_PALM_ID) {
                                editor.putString(PalmConstants.KEY_PALM_APP, configJson.getJSONObject("configMap").toString())
                                        .putLong(PalmConstants.KEY_PALM_UPDATE_TIME, System.currentTimeMillis());
                                break;
                            }
                        }
                        editor.apply();
                    } catch (Exception e) {
                        PalmUtil.LOG.e(Log.getStackTraceString(e));
                    }
                }
            }
        } catch (Exception e) {
//            e.printStackTrace();
        }
    }

    /**
     * 授权接口
     */
    @Override
    public void authorize(Context context, IPalmAuthListener authListener) {
        Bundle eparam = new Bundle();
        eparam.putString("appid", mPalmAuthParam.getAppid());
        eparam.putString("version", BuildConfig.SDK_VERSION);
        new AthenaCust("sdk_start_auth", 7710).trackCommon(eparam, null).submit();

        final IPalmAuthListener wrapAuthListener = new IPalmAuthListener() {
            /**
             * 页面回调
             */
            private void startIntentActivity(String intentActivity, Bundle extras) {
                try {
                    Intent intent = new Intent(Intent.ACTION_VIEW)
                            .setClassName(mContext, intentActivity)
                            .addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                            .putExtras(extras);
                    mContext.startActivity(intent);
                } catch (Exception e) {
                    PalmUtil.LOG.e(Log.getStackTraceString(e));
                }
            }

            @Override
            public void onSuccess(PalmAuthResult result) {
                PalmUtil.LOG.d("onSuccess");
                String intentActivity = mPalmAuthParam.getIntentActivity();
                if (!TextUtils.isEmpty(intentActivity)) {
                    String tokenInfo = "";
                    String userInfo = "";
                    try {
                        tokenInfo = new JSONObject().put("access_token", result.getAccessToken())
                                .put("refresh_token", result.getRefreshToken())
                                .put("open_id", result.getOpenid())
                                .toString();
                        if (result.getNickName() != null || result.getUserName() != null) {
                            userInfo = new JSONObject().put("nickName", result.getNickName())
                                    .put("userName", result.getUserName())
                                    .put("avatarUrl", result.getAvatarUrl()).toString();
                        }
                    } catch (Exception e) {
                        PalmUtil.LOG.e(Log.getStackTraceString(e));
                    }
                    Bundle bundle = new Bundle();
                    bundle.putString(PalmConstants.EXTRA_TOKEN_INFO, tokenInfo);
                    bundle.putString(PalmConstants.EXTRA_USER_INFO, userInfo);
                    startIntentActivity(intentActivity, bundle);
                } else {
                    if (authListener != null) {
                        authListener.onSuccess(result);
                    }
                }
            }

            @Override
            public void onCancel() {
                PalmUtil.LOG.d("onCancel");
                if (authListener != null) {
                    authListener.onCancel();
                }
            }

            @Override
            public void onFailure(int errorCode, String errorMessage) {
                PalmUtil.LOG.d("onFailure errorCode = " + errorCode
                        + ", errorMessage = " + errorMessage);
                if (authListener != null) {
                    authListener.onFailure(errorCode, errorMessage);
                }
            }
        };

        if (!PalmUtil.isNetworkAvailable(mContext)) {
            wrapAuthListener.onFailure(PalmConstants.ERROR_NETWORK, "network unavailable");

            eparam.putString("result", "failure");
            eparam.putInt("errcode", PalmConstants.ERROR_NETWORK);
            new AthenaCust("sdk_start_auth_callback", 7710).trackCommon(eparam, null).submit();
            return;
        } else {
            if (isMainProcess) {
                XNManager.getInstance().applyKey(mContext,
                        mPalmAuthParam.getAppid(), mPalmAuthParam.getServerMode());  // 提前完成applyKey接口
            }
        }
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                Bundle aTempResult = new Bundle();  // 保存PalmID返回用户基本信息->复用
                PalmAuthRequest palmAuthRequest =
                        new PalmAuthRequest(mPalmAuthParam, isFullSDK, new IPalmAuthResponse.Stub() {
                            @Override
                            public void onCancel() throws RemoteException {
                                wrapAuthListener.onCancel();

                                eparam.putString("result", "cancel");
                                new AthenaCust("sdk_start_auth_callback", 7710).trackCommon(eparam, null).submit();
                            }

                            @Override
                            public void setResult(Bundle result) throws RemoteException {
                                if (aTempResult.size() > 0) {
                                    result.putAll(aTempResult);  // 用户基本信息
                                    aTempResult.clear();
                                }
                                if (result.containsKey(PalmConstants.EXTRA_TOKEN_INFO)) {
                                    wrapAuthListener.onSuccess(new PalmAuthResult(result));

                                    eparam.putString("result", "success");
                                    new AthenaCust("sdk_start_auth_callback", 7710).trackCommon(eparam, null).submit();
                                } else {
                                    int errorCode = result.getInt(PalmConstants.EXTRA_ERROR_CODE, 0);
                                    String errorMessage = result.getString(PalmConstants.EXTRA_ERROR_MESSAGE);
                                    wrapAuthListener.onFailure(errorCode, errorMessage);

                                    eparam.putString("result", "failure");
                                    eparam.putInt("errcode", errorCode);
                                    new AthenaCust("sdk_start_auth_callback", 7710).trackCommon(eparam, null).submit();
                                }
                            }
                        });
                if (mPalmAuthParam.hasTudc()) {  // 调用登录接口清除TUDC的token
                    TudcAccount.getInstance(mPalmAuthParam.getTudcSpName()).clear(mContext);
                }
                Intent palmIntent =
                        mPalmAuthParam.ignorePalmApp() ? null : PalmUtil.queryPalmAppIntent(mContext, false);
                PalmUtil.LOG.d("palm id intent = " + palmIntent);
                if (palmIntent != null) {  // 支持PalmID和HOST登录
                    Cursor cursor = null;
                    if ("tech.palm.id".equals(palmIntent.getPackage())) {
                        cursor = mContext.getContentResolver()
                                .query(Uri.parse("content://com.palm.id.account.AccountProvider/logged"), null, null, null, null);
                    } else if ("com.rlk.mi".equals(palmIntent.getPackage())) {
                        cursor = mContext.getContentResolver()
                                .query(Uri.parse("content://com.palm.id.old.account.AccountProvider/logged"), null, null, null, null);
                    }
                    boolean isPalmLogged = false;
                    if (cursor != null) {
                        isPalmLogged = cursor.getCount() > 0;
                        cursor.close();
                    }
                    if (isPalmLogged) {
                        Bundle param = new Bundle();
                        if (isFullSDK) {
                            param.putParcelable(PalmConstants.EXTRA_AUTH_PARAM, mPalmAuthParam.copy());  // 调包请求token
                        } else {
                            param.putParcelable(PalmConstants.EXTRA_AUTH_PARAM, mPalmAuthParam);
                        }
                        param.putBoolean(PalmConstants.EXTRA_HOST_MODE,
                                palmIntent.getBooleanExtra(PalmConstants.EXTRA_HOST_MODE, false));
                        IPalmAuthListener sdkAuthListener = new IPalmAuthListener() {
                            @Override
                            public void onSuccess(PalmAuthResult result) {
                                if (isFullSDK) {
                                    if (result.getNickName() != null || result.getUserName() != null) {
                                        try {
                                            aTempResult.putString(PalmConstants.EXTRA_USER_INFO,
                                                    new JSONObject().put("nickName", result.getNickName())
                                                            .put("userName", result.getUserName())
                                                            .put("avatarUrl", result.getAvatarUrl()).toString());
                                        } catch (Exception e) {
                                            PalmUtil.LOG.e(Log.getStackTraceString(e));
                                        }
                                    }
                                }
                                wrapAuthListener.onSuccess(result);
                            }

                            @Override
                            public void onCancel() {
                                wrapAuthListener.onCancel();
                            }

                            @Override
                            public void onFailure(int errorCode, String errorMessage) {
                                if (errorCode == PalmConstants.ERROR_HOST_REQ) {  // HOST登录失败，执行本地登录
                                    if (isFullSDK) {
                                        if (isMainProcess) {
                                            PalmAuthManager.getInstance(mContext).handleAuthRequest(palmAuthRequest);
                                        }
                                    }
                                } else {
                                    wrapAuthListener.onFailure(errorCode, errorMessage);
                                }
                            }
                        };
                        int retryCount = 0;  // 2次失败后使用web方式授权
                        while (retryCount < 2) {
                            if (new PalmAuthSession(mContext,
                                    palmIntent, param, sdkAuthListener, isMainProcess).create()) {
                                return;
                            }
                            retryCount++;
                        }
                    }
                }

                if (isFullSDK) {
                    if (isMainProcess) {
                        PalmAuthManager.getInstance(mContext).handleAuthRequest(palmAuthRequest);  // 本地FullSDK登录+授权
                    }
                }
            }
        });
    }

    /**
     * 退出登录状态(Full PalmSDK)
     */
    @Override
    public void logout(String accessToken, String refreshToken, IPalmApiListener<Boolean> apiListener) {
        Bundle eparam = new Bundle();
        eparam.putString("appid", mPalmAuthParam.getAppid());
        eparam.putString("version", BuildConfig.SDK_VERSION);
        new AthenaCust("sdk_logout", 7710).trackCommon(eparam, null).submit();

        if (!isMainProcess) {
            if (apiListener != null) {
                apiListener.onFailure(PalmConstants.ERROR_PROCESS, "not main process");
            }
            return;
        }
        if (!PalmUtil.isNetworkAvailable(mContext)) {
            if (apiListener != null) {
                apiListener.onFailure(PalmConstants.ERROR_NETWORK, "network unavailable");
            }
            return;
        }
        Map<String, String> header = new HashMap<>();
        header.put("Authorization", "Bearer " + accessToken);
        header.put("Client-ID", mPalmAuthParam.getAppid());

        Map<String, Object> params = new HashMap<>();
        params.put("refresh-token", refreshToken);
        PalmHttpClient.getInstance().expireToken(header, params, new PalmHttpCallback<String>() {
            @Override
            public void onSuccess(String response) {
                if (apiListener != null) {
                    apiListener.onSuccess(true);
                }
            }

            @Override
            public void onFailure(int errorCode, String errorMessage) {
                if (apiListener != null) {
                    apiListener.onFailure(errorCode, errorMessage);
                }
            }
        });
    }

    /**
     * 调用开放接口
     * 若权限不足，会唤起动态授权界面
     */
    @Override
    public void callOpenApi(String path, String accessToken, Map<String, Object> formData,
                            IPalmApiListener<String> apiListener) {
        Bundle eparam = new Bundle();
        eparam.putString("appid", mPalmAuthParam.getAppid());
        eparam.putString("version", BuildConfig.SDK_VERSION);
        eparam.putString("param", path);
        new AthenaCust("sdk_openapi", 7710).trackCommon(eparam, null).submit();

        if (TextUtils.isEmpty(path) || TextUtils.isEmpty(accessToken)) {
            if (apiListener != null) {
                apiListener.onFailure(PalmConstants.ERROR_ARGUMENT, "invalid arguments");
            }
            eparam.putString("result", "failure");
            eparam.putInt("errcode", PalmConstants.ERROR_ARGUMENT);
            new AthenaCust("sdk_openapi_callback", 7710).trackCommon(eparam, null).submit();
            return;
        }
        if (!PalmUtil.isNetworkAvailable(mContext)) {
            if (apiListener != null) {
                apiListener.onFailure(PalmConstants.ERROR_NETWORK, "network unavailable");
            }
            eparam.putString("result", "failure");
            eparam.putInt("errcode", PalmConstants.ERROR_NETWORK);
            new AthenaCust("sdk_openapi_callback", 7710).trackCommon(eparam, null).submit();
            return;
        }
        Map<String, String> header = new HashMap<>();
        header.put("Authorization", "Bearer " + accessToken);
        header.put("Client-ID", mPalmAuthParam.getAppid());
        PalmHttpClient.getInstance().callOpenApi(path, header, formData, new PalmHttpCallback<String>() {
            @Override
            public void onSuccess(String response) {
                if (apiListener != null) {
                    apiListener.onSuccess(response);
                }
                eparam.putString("result", "success");
                new AthenaCust("sdk_openapi_callback", 7710).trackCommon(eparam, null).submit();
            }

            @Override
            public void onFailure(int errorCode, String errorMessage) {
                if (apiListener != null) {
                    apiListener.onFailure(errorCode, errorMessage);
                }
                eparam.putString("result", "failure");
                eparam.putInt("errcode", errorCode);
                new AthenaCust("sdk_openapi_callback", 7710).trackCommon(eparam, null).submit();
            }
        });
    }

    /**
     * 强制触发一次token刷新操作
     */
    @Override
    public void refreshToken(String refreshToken, IPalmApiListener<PalmTokenInfo> apiListener) {
        final String appid = mPalmAuthParam.getAppid();
        Bundle eparam = new Bundle();
        eparam.putString("appid", appid);
        eparam.putString("version", BuildConfig.SDK_VERSION);
        new AthenaCust("sdk_refresh_token", 7710).trackCommon(eparam, null).submit();

        if (TextUtils.isEmpty(refreshToken)) {
            if (apiListener != null) {
                apiListener.onFailure(PalmConstants.ERROR_ARGUMENT, "invalid arguments");
            }
            eparam.putString("result", "failure");
            eparam.putInt("errcode", PalmConstants.ERROR_ARGUMENT);
            new AthenaCust("sdk_refresh_token_callback", 7710).trackCommon(eparam, null).submit();
            return;
        }
        if (!PalmUtil.isNetworkAvailable(mContext)) {
            if (apiListener != null) {
                apiListener.onFailure(PalmConstants.ERROR_NETWORK, "network unavailable");
            }
            eparam.putString("result", "failure");
            eparam.putInt("errcode", PalmConstants.ERROR_NETWORK);
            new AthenaCust("sdk_refresh_token_callback", 7710).trackCommon(eparam, null).submit();
            return;
        }
        Map<String, Object> params = new HashMap<>();
        params.put("refresh_token", refreshToken);
        params.put("grant_type", PalmConstants.GRANT_TYPE_REFRESH);
        params.put("client_id", appid);
        params.put("scope", mPalmAuthParam.getScopes());
        PalmHttpClient.getInstance().refreshToken(params, new PalmHttpCallback<PalmTokenInfo>() {
                @Override
                public void onSuccess(PalmTokenInfo palmTokenInfo) {
                    PalmUtil.LOG.e("refreshToken success");
                    if (apiListener != null) {
                        apiListener.onSuccess(palmTokenInfo);

                        Bundle eparam = new Bundle();
                        eparam.putString("appid", appid);
                        eparam.putString("version", BuildConfig.SDK_VERSION);
                        eparam.putString("result", "success");
                        new AthenaCust("sdk_refresh_token_callback", 7710).trackCommon(eparam, null).submit();
                    }
                }

                @Override
                public void onFailure(int errorCode, String errorMessage) {
                    PalmUtil.LOG.e("refreshToken errorCode = " + errorCode
                            + " errorMessage = " + errorMessage);
                    if (apiListener != null) {
                        apiListener.onFailure(errorCode, errorMessage);

                        Bundle eparam = new Bundle();
                        eparam.putString("appid", appid);
                        eparam.putString("version", BuildConfig.SDK_VERSION);
                        eparam.putString("result", "failure");
                        eparam.putInt("errcode", errorCode);
                        new AthenaCust("sdk_refresh_token_callback", 7710).trackCommon(eparam, null).submit();
                    }
                }
            });
    }

    /**
     * 返回当前登录配置参数
     */
    public PalmAuthParam getPalmAuthParam() {
        return mPalmAuthParam;
    }
}
