package com.transsion.palmsdk.auth;

import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import com.transsion.palmsdk.PalmConstants;
import com.transsion.palmsdk.account.XNManager;
import com.transsion.palmsdk.data.PalmAuthRequest;
import com.transsion.palmsdk.http.PalmHttpClient;
import com.transsion.palmsdk.util.PalmUtil;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Native方式处理授权
 */
public class PalmAuthManager {

    private static volatile PalmAuthManager sPalmAuthManager;
    private final Context mContext;
    private final Map<String, PalmAuthRequest> mAuthRequestMap;

    private PalmAuthManager(Context context) {
        mContext = context.getApplicationContext(); // 有泄露风险
        mAuthRequestMap = new ConcurrentHashMap<>();
    }

    public static PalmAuthManager getInstance(Context context) {
        if (sPalmAuthManager == null) {
            synchronized (PalmAuthManager.class) {
                if (sPalmAuthManager == null) {
                    sPalmAuthManager = new PalmAuthManager(context);
                }
            }
        }
        return sPalmAuthManager;
    }

    /**
     * 跳转到palm id授权界面
     */
    public void handleAuthRequest(PalmAuthRequest authRequest) {
        addAuthRequest(authRequest);
        try {
            boolean isLogged = XNManager.getInstance().isLogged(mContext);
            PalmUtil.LOG.d("isLogged = " + isLogged + " isHostMode = " + authRequest.isHostMode());
            if (isLogged) {
                if (authRequest.isSilentMode()) {  // 启动异步service完成授权请求
                    XNManager.getInstance().silentLogin(mContext, authRequest.getAuthRequestId());
                } else {
                    Intent intent;
                    if (authRequest.showPrevLogin()) {
                        intent = new Intent(PalmConstants.ACTION_PALM_ID_PREV);  // 启动预登录界面
                    } else {
                        intent = new Intent(PalmConstants.ACTION_PALM_ID_AUTH);  // 启动授权界面
                    }
                    intent.setPackage(mContext.getPackageName());
                    intent.putExtra(PalmConstants.EXTRA_AUTH_REQUEST, authRequest.getAuthRequestId());
                    if ("tech.palm.id".equals(mContext.getPackageName())) {
                        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK
                                | Intent.FLAG_ACTIVITY_CLEAR_TASK);
                    } else {
                        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    }
                    mContext.startActivity(intent);
                }
            } else {
                if (authRequest.isSilentMode() || authRequest.isHostMode()) {  // 静默授权/HOST应用未登录
                    authRequest.setException(PalmConstants.ERROR_NOT_LOGIN, "palm id not login");
                } else {
//                    Intent intent = new Intent(PalmConstants.ACTION_PALM_ID_LOGIN);  // 传递request
                    Intent intent = new Intent();
                    intent.setClassName(mContext.getPackageName(),
                            "com.transsion.xuanniao.account.login.view.LoginActivity");
                    intent.putExtra("isFinish", true);  // 登录成功后关闭并重新回到当前页面
                    intent.putExtra("source", authRequest.getAuthParam().getPkgName());
                    intent.putExtra(PalmConstants.EXTRA_AUTH_REQUEST, authRequest.getAuthRequestId());
                    if ("tech.palm.id".equals(mContext.getPackageName())) {
                        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK
                                | Intent.FLAG_ACTIVITY_CLEAR_TASK);
                    } else {
                        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    }
                    mContext.startActivity(intent);
                }
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
            authRequest.setException(PalmConstants.ERROR_REMOTE, "palm id app error");
        }
    }

    /**
     * 获取OS PalmID登录状态
     */
    public void handleStatusRequest(PalmAuthRequest authRequest) {
        addAuthRequest(authRequest);
        try {
            boolean isLogged = XNManager.getInstance().isLogged(mContext);
            PalmUtil.LOG.d("isLogged = " + isLogged);
            if (isLogged) {
                Bundle bundle = new Bundle();
                try {
                    String accountInfo = XNManager.getInstance().getAccountInfo(mContext);
                    if (!TextUtils.isEmpty(accountInfo)) {
                        JSONObject jsonObject = new JSONObject(accountInfo);
                        // 返回基本信息for允许切换账号登录
                        bundle.putString(PalmConstants.EXTRA_USER_INFO,
                                new JSONObject().put("nickName", jsonObject.optString("nickname"))
                                        .put("userName", jsonObject.optString("username"))
                                        .put("avatarUrl", jsonObject.optString("avatarUrl")).toString());
                        bundle.putString(PalmConstants.EXTRA_LOGGED_ID,
                                String.valueOf(jsonObject.getString("xuanniaoId").hashCode()));
                    }
                } catch (Exception e) {
                    PalmUtil.LOG.e(Log.getStackTraceString(e));
                }
                authRequest.setResult(bundle);
            } else {
                Bundle bundle = new Bundle();
                bundle.putInt(PalmConstants.EXTRA_STATUS, PalmConstants.STATUS_NOT_LOGIN);
                authRequest.setResult(bundle);
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
            authRequest.setException(PalmConstants.ERROR_REMOTE, "palm id app error");
        }
    }

    /**
     * 注册请求
     */
    public void addAuthRequest(PalmAuthRequest authRequest) {
        authRequest.setAuthManager(this);
        mAuthRequestMap.put(authRequest.getAuthRequestId(), authRequest);
        PalmUtil.LOG.d("begin request id = " + authRequest.getAuthRequestId());
    }

    /**
     * 取消注册
     */
    public void removeAuthRequest(String requestId) {
        mAuthRequestMap.remove(requestId);
        PalmUtil.LOG.d("end request id = " + requestId);
    }

    /**
     * 返回登录回调接口
     */
    public PalmAuthRequest getAuthRequest(String requestId) {
        if (TextUtils.isEmpty(requestId)) {
            return null;
        }
        return mAuthRequestMap.get(requestId);
    }
}
