package com.transsion.palmsdk;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;

import com.google.gson.annotations.SerializedName;
import com.transsion.palmsdk.data.PalmAuthResult;
import com.transsion.palmsdk.data.PalmTokenInfo;
import com.transsion.palmsdk.util.PalmUtil;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

public abstract class PalmID {

    private static final HashMap<Context, PalmID> sInstanceMap = new HashMap<>(4);
    private static Context sAppContext;
    public static final int SERVER_DEV = 1;
    public static final int SERVER_TEST = 2;
    public static final int SERVER_PRE = 3;
    public static final int SERVER_ONLINE = 4;

    /**
     * 登录/授权结果回调
     */
    public interface IPalmAuthListener {
        void onSuccess(PalmAuthResult result);
        void onCancel();
        void onFailure(int errorCode, String errorMessage);
    }

    /**
     * 监听palm id登录状态
     */
    public interface IPalmStatusListener {
        void onSignIn(int status, String loggedUser);
        void onSignOut(int status);
    }

    /**
     * 访问open api结果回调
     */
    public interface IPalmApiListener<T> {
        void onSuccess(T response);
        void onFailure(int errorCode, String errorMessage);
    }

    /**
     * 返回上下文
     */
    public static Context getContext() {
        return sAppContext;
    }

    public static PalmID createInstance(Context context, PalmAuthParam palmAuthParam) {
        if (context == null || palmAuthParam == null) {
            throw new IllegalArgumentException("invalid arguments");
        }
        if (sAppContext == null) {
            sAppContext = context.getApplicationContext();
        }
        PalmID palmID;
        synchronized (PalmID.class) {
            Context appContext = context.getApplicationContext();
            if (!sInstanceMap.containsKey(appContext)) {
                palmID = new PalmDefaultImpl(context, palmAuthParam);
                sInstanceMap.put(appContext, palmID);
            } else {
                palmID = sInstanceMap.get(appContext);
            }
        }
        return palmID;
    }

    /**
     * 返回当前进程中PalmID实例
     */
    public static PalmID instance(Context context) {
        if (context == null) {
            return new PalmEmptyImpl();
        }
        synchronized (PalmID.class) {
            Context appContext = context.getApplicationContext();
            PalmID palmID = sInstanceMap.get(appContext);
            return palmID != null ? palmID : new PalmEmptyImpl();
        }
    }

    /**
     * 返回当前登录配置参数
     */
    public PalmAuthParam getAuthParam() {
        if (this instanceof PalmDefaultImpl) {
            return ((PalmDefaultImpl) this).getPalmAuthParam();
        }
        return null;
    }

    /**
     * 解析授权结果
     * @param intent 回调结果
     * @param authListener 业务方
     */
    public static void parseIntent(Intent intent, IPalmAuthListener authListener) {
        if (intent == null || authListener == null) {
            PalmUtil.LOG.e("invalid arguments");
            return;
        }
        Bundle result = intent.getExtras();
        if (result.containsKey(PalmConstants.EXTRA_TOKEN_INFO)) {
            authListener.onSuccess(new PalmAuthResult(result));
        } else {
            int errorCode = result.getInt(PalmConstants.EXTRA_ERROR_CODE, 0);
            String errorMessage = result.getString(PalmConstants.EXTRA_ERROR_MESSAGE);
            if (errorCode == PalmConstants.ERROR_CANCEL) {
                authListener.onCancel();
            } else {
                authListener.onFailure(errorCode, errorMessage);
            }
        }
    }

    /**
     * 授权接口
     */
    public abstract void authorize(Context context, IPalmAuthListener authListener);

    /**
     * 调用开放接口
     * 若权限不足，会唤起动态授权界面
     */
    public abstract void callOpenApi(String path, String accessToken, Map<String, Object> formData,
                                     IPalmApiListener<String> apiListener);

    /**
     * 强制触发一次token刷新操作
     */
    public abstract void refreshToken(String refreshToken, IPalmApiListener<PalmTokenInfo> apiListener);

    /**
     * 退出登录状态(Full PalmSDK)
     */
    public abstract void logout(String accessToken, String refreshToken, IPalmApiListener<Boolean> apiListener);
}
