package com.transsion.palmsdk.auth;

import android.content.ComponentName;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.text.TextUtils;
import android.util.Log;

import com.transsion.palmsdk.PalmConstants;
import com.transsion.palmsdk.PalmID;
import com.transsion.palmsdk.account.LiteTeeManager;
import com.transsion.palmsdk.account.PalmAccProvider;
import com.transsion.palmsdk.data.PalmAuthResult;
import com.transsion.palmsdk.util.PalmUtil;

/**
 * 访问palm id app获取授权
 */
public class PalmAuthSession implements ServiceConnection {

    private IPalmAuthService mAuthService;
    private final Context mContext;
    private final Intent mPalmIntent;
    private final Bundle mParam;
    private final PalmID.IPalmAuthListener mPalmAuthListener;

    private final IPalmAuthResponse.Stub mPalmAuthResponse = new IPalmAuthResponse.Stub() {
        @Override
        public void onCancel() throws RemoteException {
//            PalmUtil.LOG.d("onCancel");
            unbind();
            if (mPalmAuthListener != null) {
                mPalmAuthListener.onCancel();
            }
        }

        @Override
        public void setResult(Bundle result) throws RemoteException {
//            PalmUtil.LOG.d("setResult result = " + result);
            unbind();
            result.setClassLoader(getClass().getClassLoader());
            if (result.containsKey(PalmConstants.EXTRA_TOKEN_INFO)) {
                if (mPalmAuthListener != null) {
                    mPalmAuthListener.onSuccess(new PalmAuthResult(result));
                }
            } else {
                int errorCode = result.getInt(PalmConstants.EXTRA_ERROR_CODE, 0);
                String errorMessage = result.getString(PalmConstants.EXTRA_ERROR_MESSAGE);
                if (mPalmAuthListener != null) {
                    mPalmAuthListener.onFailure(errorCode, errorMessage);
                }
            }
        }
    };

    public PalmAuthSession(Context context, Intent palmIntent, Bundle param,
                           PalmID.IPalmAuthListener palmAuthListener, boolean isMProcess) {
        mContext = context;
        mPalmIntent = palmIntent;
        mParam = param;
        mPalmAuthListener = palmAuthListener;
    }

    public boolean create() {
        try {
            if (Build.VERSION.SDK_INT >= 34) {
                return mContext.bindService(mPalmIntent, this,
                        Context.BIND_AUTO_CREATE | 512);  // 修复Android U后台打开界面限制 Context.BIND_ALLOW_ACTIVITY_STARTS = 512
            } else {
                return mContext.bindService(mPalmIntent, this, Context.BIND_AUTO_CREATE);
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        return false;
    }

    private void unbind() {
        if (mAuthService != null) {
            mAuthService = null;
            mContext.unbindService(this);
        }
    }

    @Override
    public void onServiceConnected(ComponentName name, IBinder service) {
        PalmUtil.LOG.d("onServiceConnected");
        try {
            mAuthService = IPalmAuthService.Stub.asInterface(service);
            mAuthService.authorize(mParam, mPalmAuthResponse);
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
            if (mPalmAuthListener != null) {
                mPalmAuthListener.onFailure(PalmConstants.ERROR_REMOTE, "remote exception");
            }
        }
    }

    @Override
    public void onServiceDisconnected(ComponentName name) {
        mAuthService = null;
        if (mPalmAuthListener != null) {
            mPalmAuthListener.onFailure(PalmConstants.ERROR_REMOTE, "service disconnected");
        }
    }
}