package com.transsion.palmsdk.data;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import android.os.RemoteException;
import android.util.Log;

import com.transsion.palmsdk.PalmAuthParam;
import com.transsion.palmsdk.PalmConstants;
import com.transsion.palmsdk.auth.IPalmAuthResponse;
import com.transsion.palmsdk.auth.PalmAuthManager;
import com.transsion.palmsdk.util.PalmUtil;

/**
 * 一次授权请求
 */
public class PalmAuthRequest implements Parcelable {

    private final String authRequestId;
    private final boolean isFullSDK;
    private final PalmAuthParam palmAuthParam;
    private final IPalmAuthResponse response;
    private PalmAuthManager authManager;
    private boolean isSilentMode = false;  // 是否静默
    private boolean isHostMode = false;  // 是否HOST角色

    public PalmAuthRequest(PalmAuthParam palmAuthParam, boolean isFullSDK, IPalmAuthResponse response) {
        this.palmAuthParam = palmAuthParam;
        this.isFullSDK = isFullSDK;
        this.response = response;
        this.authRequestId = PalmUtil.randomString(8);
    }

    protected PalmAuthRequest(Parcel in) {
        authRequestId = in.readString();
        isFullSDK = in.readInt() == 1;
        isSilentMode = in.readInt() == 1;
        palmAuthParam = in.readParcelable(PalmAuthParam.class.getClassLoader());
        response = IPalmAuthResponse.Stub.asInterface(in.readStrongBinder());
        isHostMode = in.readInt() == 1;
    }

    public static final Creator<PalmAuthRequest> CREATOR = new Creator<PalmAuthRequest>() {
        @Override
        public PalmAuthRequest createFromParcel(Parcel in) {
            return new PalmAuthRequest(in);
        }

        @Override
        public PalmAuthRequest[] newArray(int size) {
            return new PalmAuthRequest[size];
        }
    };

    public PalmAuthParam getAuthParam() {
        return palmAuthParam;
    }

    public String getAuthRequestId() {
        return authRequestId;
    }

    public boolean isFullSDK() {
        return isFullSDK;
    }

    public void setSilentMode(boolean silentMode) {
        isSilentMode = silentMode;
    }

    public boolean isSilentMode() {
        return isSilentMode;
    }

    public void setHostMode(boolean hostMode) {
        isHostMode = hostMode;
    }

    public boolean isHostMode() {
        return isHostMode;
    }

    /**
     * 显示预登录弹框
     */
    public boolean showPrevLogin() {
        return !isHostMode && palmAuthParam.isShowPrevLogin();
    }

    public void onCancel() {
        try {
            if (response != null) {
                response.onCancel();
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        finishRequest();
    }

    public void setResult(Bundle bundle) {
        try {
            if (response != null) {
                response.setResult(bundle);
            }
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        finishRequest();
    }

    public void setException(int errorCode, String errorMessage) {
        try {
            if (isHostMode) errorCode = PalmConstants.ERROR_HOST_REQ;  // 辅助登录异常码统一
            if (response != null) {
                Bundle bundle = new Bundle();
                bundle.putInt(PalmConstants.EXTRA_ERROR_CODE, errorCode);
                bundle.putString(PalmConstants.EXTRA_ERROR_MESSAGE, errorMessage);
                response.setResult(bundle);
            }
        } catch (RemoteException e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        finishRequest();
    }

    public void setAuthManager(PalmAuthManager authManager) {
        this.authManager = authManager;
    }

    /**
     * 结束此次请求
     */
    private void finishRequest() {
        if (authManager != null) {
            authManager.removeAuthRequest(authRequestId);
        }
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(authRequestId);
        dest.writeInt(isFullSDK ? 1: 0);
        dest.writeInt(isSilentMode ? 1: 0);
        dest.writeParcelable(palmAuthParam, flags);
        dest.writeStrongBinder(response.asBinder());
        dest.writeInt(isHostMode ? 1: 0);
    }
}
